#include "exerciserapid.h"

#include <QVBoxLayout>
#include <QFont>
#include <QPushButton>
#include <QSpinBox>
#include <QCheckBox>
#include <QFormLayout>
#include <QResizeEvent>

ExerciseRapid::ExerciseRapid(QWidget *parent) :
        Exercise(parent)
{
    m_currentPos=0;
    m_interval = 900;

    QVBoxLayout *vbLayout = new QVBoxLayout;

    m_lblText.setSizePolicy(QSizePolicy::Preferred,QSizePolicy::MinimumExpanding);
    m_lblText.setAlignment(Qt::AlignCenter);
    m_lblText.setWordWrap(true);

    m_lblExerciseInfo.setSizePolicy(QSizePolicy::Preferred,QSizePolicy::MinimumExpanding);
    m_lblExerciseInfo.setGeometry(geometry());
    m_lblExerciseInfo.setAlignment(Qt::AlignJustify|Qt::AlignTop);
    m_lblExerciseInfo.setWordWrap(true);

    vbLayout->addWidget(&m_lblText);
    vbLayout->addWidget(&m_lblExerciseInfo);

    setLayout(vbLayout);

    showExerciseInfo();

    generateSettings();

    m_lblPause.setParent(this);
    m_lblPause.hide();
    m_lblPause.setStyleSheet("background: rgba(133,133,135,50%); border: 0px;border-radius: 5px;");
    m_lblPause.setPixmap(QIcon::fromTheme("media-playback-pause", QIcon(":/img/paused.png")).pixmap(100));
    m_lblPause.setAlignment(Qt::AlignCenter);
    m_lblPause.raise();
    m_lblPause.resize(100,100);

    connect(&m_timer,SIGNAL(timeout()),this,SLOT(tick()));
}

void ExerciseRapid::showExerciseInfo(){
    m_lblText.hide();

    QString info = "<b>"+tr("Rapid mode will help you significantly increase your reading speed.")+"</b><br><br>"+
                   tr("You will see words on the screen displaying one by one. Fast speed will help you to understand the words without internal pronunciation.")+"<br><br>"+
                   tr("Tap to screen or press \"Start\" button to start (if text is loaded).");

    m_lblExerciseInfo.setText(info);

    m_lblExerciseInfo.show();
}

QWidget* ExerciseRapid::settingsWidget(){
    return &m_settingsWidget;
}

bool ExerciseRapid::advancedMode(){
    return true;
}

void ExerciseRapid::setText(QString& str){
    m_strList = str.split(QRegExp("[ \n\r]"),QString::SkipEmptyParts);
}

void ExerciseRapid::start(){
    if (m_currentPos>=m_strList.count())
        m_currentPos = 0;
    m_lblPause.hide();
    m_lblExerciseInfo.hide();
    m_lblText.show();
    tick();
    m_timer.start(m_interval);
}

void ExerciseRapid::stop(){
    m_currentPos=0;
    m_lblPause.hide();
    showExerciseInfo();
    m_timer.stop();
}

void ExerciseRapid::pause(){
    m_timer.stop();
    m_lblPause.show();
}

void ExerciseRapid::tick(){
    static int ticksDone=0;
    if (m_currentPos<m_strList.count()){
        int p = qRound((double)m_currentPos/(m_settings.wordsOnScreen*m_settings.lineNumbers));
        emit progressChanged(p);
        QString str="";
        for (int i=0;i<m_settings.lineNumbers;i++){
            for (int j=0;j<m_settings.wordsOnScreen;j++){
                if (m_currentPos<m_strList.count())
                    str += " "+m_strList[m_currentPos++];
                else
                    m_currentPos=m_strList.count()-1;
            }
            if (m_currentPos<m_strList.count())
                str+="\n";
        }
        m_lblText.setText(str);
        ticksDone++;
        if (m_settings.accelerate&&(m_timer.interval()!=0)&&(ticksDone>=30000/m_timer.interval())) {
            ticksDone = 0;
            faster();
        }
    } else {
        m_currentPos=m_strList.count()-1;
        m_timer.stop();
    }
}

void ExerciseRapid::zoomIn(){
    QFont font(m_lblText.font());
    int i=font.pointSize();
    if (i<24) {
        font.setPointSize(++i);

        m_lblText.setFont(font);
        emit minZoomExceed(false);
        if (i>=24)
            emit maxZoomExceed(true);
    } else
        emit maxZoomExceed(true);
}

void ExerciseRapid::zoomOut(){
    QFont font(m_lblText.font());
    int i=font.pointSize();
    if (i>8){
        font.setPointSize(--i);

        m_lblText.setFont(font);
        emit maxZoomExceed(false);
        if (i<=8)
            emit minZoomExceed(true);
    } else
        emit minZoomExceed(true);
}

bool ExerciseRapid::isEmptyText(){
    return m_strList.isEmpty();
}

int ExerciseRapid::maxProgress(){
    if (m_strList.isEmpty())
        return 0;
    else
        return (m_strList.count()/(m_settings.wordsOnScreen*m_settings.lineNumbers))-1;
}

void ExerciseRapid::mousePressEvent(QMouseEvent *){
    emit clicked();
}

void ExerciseRapid::faster(){
    int n=60000/m_settings.maxSpeed;
    if (m_interval>(n+40)){
        m_interval-=40;
        emit speedChanged(60000/m_interval);
        emit minSpeedExceed(false);
        if (m_interval<=n)
            emit maxSpeedExceed(true);
    } else {
        m_interval=n;
        emit speedChanged(60000/m_interval);
        emit maxSpeedExceed(true);
    }
    m_timer.setInterval(m_interval);
}

void ExerciseRapid::slower(){
    int n=60000/m_settings.minSpeed;
    if (m_interval<(n-40)){
        m_interval+=40;
        emit speedChanged(60000/m_interval);
        emit maxSpeedExceed(false);
        if (m_interval>=n)
            emit minSpeedExceed(true);
    } else {
        m_interval=n;
        emit speedChanged(60000/m_interval);
        emit minSpeedExceed(true);
    }
    m_timer.setInterval(m_interval);
}

void ExerciseRapid::generateSettings(){
    m_settingsWidget.setAutoFillBackground(true);

    QFormLayout *formLayout = new QFormLayout(&m_settingsWidget);
    formLayout->setRowWrapPolicy(QFormLayout::WrapLongRows);

    QLabel *lbl = new QLabel("<b>"+tr("Settings of Rapid exercise")+"</b>");

    formLayout->addRow(lbl);
    QSpinBox *sb = new QSpinBox;
    sb->setMinimum(1);
    sb->setMaximum(4);
    sb->setValue(m_settings.lineNumbers);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setLinesNumber(int)));

    formLayout->addRow(tr("Lines number:"),sb);

    sb = new QSpinBox;
    sb->setMinimum(1);
    sb->setMaximum(4);
    sb->setValue(m_settings.wordsOnScreen);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setWordsOnScreen(int)));

    formLayout->addRow(tr("Words at line count:"),sb);

    sb = new QSpinBox;
    sb->setMinimum(20);
    sb->setMaximum(300);
    sb->setValue(m_settings.minSpeed);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setMinSpeed(int)));

    formLayout->addRow(tr("Minimum speed (updates/minute):"),sb);

    sb = new QSpinBox;
    sb->setMinimum(20);
    sb->setMaximum(300);
    sb->setValue(m_settings.maxSpeed);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setMaxSpeed(int)));

    formLayout->addRow(tr("Maximum speed (updates/minute):"),sb);

    QCheckBox *cb = new QCheckBox;
    cb->setChecked(m_settings.accelerate);
    connect(cb,SIGNAL(toggled(bool)),this,SLOT(setAccelerate(bool)));

    formLayout->addRow(tr("Accelerate"),cb);

    QPushButton *btn = new QPushButton(tr("OK"));
    btn->setIcon(QIcon::fromTheme("window-close", QIcon(":/img/ok.png")));
    connect(btn,SIGNAL(clicked()),&m_settingsWidget,SLOT(hide()));

    formLayout->addRow(btn);
}

void ExerciseRapid::resizeEvent(QResizeEvent *event){
    QWidget::resizeEvent(event);
    m_lblPause.move((width()-m_lblPause.width())/2,(height()-m_lblPause.height())/2);
    if (parent()!=NULL){
        QSize s=parentWidget()->size();
        m_settingsWidget.setGeometry(0,0,s.width(),s.height());
    }
}

bool ExerciseRapid::isActive(){
    return m_timer.isActive();
}

void ExerciseRapid::setWordsOnScreen(int n){
    m_settings.wordsOnScreen = n;
    int k = (m_strList.count()/(m_settings.wordsOnScreen*m_settings.lineNumbers));
    if (k>0)
        emit maxProgressChanged(k-1);
    else
        emit maxProgressChanged(0);
}

void ExerciseRapid::setLinesNumber(int n){
    m_settings.lineNumbers = n;
    int k = (m_strList.count()/(m_settings.wordsOnScreen*m_settings.lineNumbers));
    if (k>0)
        emit maxProgressChanged(k-1);
    else
        emit maxProgressChanged(0);
}

void ExerciseRapid::setMinSpeed(int n){
    m_settings.minSpeed = n;
    if (m_interval<60000/n) {
        m_interval=60000/n;
        m_timer.setInterval(m_interval);
    }
}

void ExerciseRapid::setMaxSpeed(int n){
    m_settings.maxSpeed = n;
    if (m_interval>60000/n) {
        m_interval=60000/n;
        m_timer.setInterval(m_interval);
    }
}

void ExerciseRapid::setAccelerate(bool b){
    m_settings.accelerate = b;
}

int ExerciseRapid::speed(){
    return 60000/m_interval;
}

void ExerciseRapid::setOffset(int n){
    m_currentPos = n*m_settings.wordsOnScreen*m_settings.lineNumbers;
    if (m_currentPos>=m_strList.count())
        m_currentPos = m_strList.count()-1;
    tick();
}

int ExerciseRapid::mode(){
    return Exercise::Rapid;
}
