/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  Utility functions to fuelpad
 * \author Julius Luukko
 *
 *
 */

#define _XOPEN_SOURCE
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <locale.h>
#include <stdio.h>

#include "util.h"

/*******************************************************************
 *
 * Private defines
 *
 *******************************************************************/

#define TIMELENMAX 12
#define DATELENMAX 20

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

/* lcf = length conversion factor */
/* vcf = volume conversion factor */
/* mcf = mass conversion factor */
double lcf[]={1.0, 1.609344, 1.609344};
double vcf[]={1.0, 3.785411784, 4.54609};
double mcf[]={1.0, 453.59237, 453.59237};

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

unit_t curunit = {
  .mainunit = SI,
  .lengthunit=SI,
  .volumeunit=SI,
  .consumeunit=SI,
  .massunit=SI,
  .individualunit = FALSE
};

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

/*************************************************************
 * Unit conversion functions 
 ************************************************************/


/**
 * \fn double SIlength2user(double length)
 * \brief Converts length from SI units to user units
 * \param length Length in SI units (meters)
 * \return Length in user units
 */
double SIlength2user(double length)
{
  return length/lcf[curunit.lengthunit];
}

/**
 * \fn double user2SIlength(double length)
 * \brief Converts length from user units to SI units
 * \param length Length in user units
 * \return Length in SI units
 */
double user2SIlength(double length)
{
  return length*lcf[curunit.lengthunit];
}

/**
 * \fn double SIvolume2user(double length)
 * \brief Converts volume from SI units to user units
 * \param length Volume in "SI" units (litres)
 * \return Volume in user units
 */
double SIvolume2user(double length)
{
  return length/vcf[curunit.volumeunit];
}

/**
 * \fn double user2SIvolume(double length)
 * \brief Converts volume from user units to SI units
 * \param length Volume in user units
 * \return Volume in SI units
 */
double user2SIvolume(double length)
{
  return length*vcf[curunit.volumeunit];
}

/**
 * \fn double SIconsumption2user(double consum)
 * \brief Converts consumption from SI units to user units
 * \param consum Consumption in litres/100 km
 * \return Consumption in user units
 *
 * If the global consumeunit is other than SI, the consumption
 * is converted from litres/100 km to miles per gallon (MPG).
 *
 */
double SIconsumption2user(double consum)
{
  if (curunit.consumeunit==SI)
    return consum;
  else
    return vcf[curunit.consumeunit]/lcf[curunit.consumeunit]*100.0/consum;
}

/**
 * \fn double user2SIconsumption(double consum)
 * \brief Converts consumption from  user units to SI units
 * \param consum Consumption in user units
 * \return Consumption in litres/100 km
 */
double user2SIconsumption(double consum)
{
  return SIconsumption2user(consum);
}

/**
 * \fn double SIppl2user(double ppl)
 * \brief Converts price per volume from SI units to user units
 * \param ppl Price/litre
 * \return Price/user unit
 */
double SIppl2user(double ppl)
{
  return user2SIvolume(ppl);
}

/**
 * \fn double SImass2user(double mass)
 * \brief Converts mass from SI units to user units
 * \param mass Mass in SI units (g)
 * \return Mass in user units
 */
double SImass2user(double mass)
{
  return mass/mcf[curunit.massunit];
}

/**
 * \fn double SIemission2user(double emission)
 * \brief Converts emission from SI units to user units
 * \param emission Emission in SI units (g/km)
 * \return Mass in user units
 */
double SIemission2user(double emission)
{
  double result;

  if (curunit.massunit==SI)
    result=SImass2user(emission)*lcf[curunit.lengthunit];
  else
    result=SImass2user(emission)*lcf[curunit.lengthunit]*100;
  return result;
}

/**
 * \fn double SIpricepertrip2user(double ppt)
 * \brief Converts price per trup from SI units to user units
 * \param ppt Price per trip in SI units (currency/km)
 * \return Price per trip in user length units
 */
double SIpricepertrip2user(double ppt)
{
  return ppt*lcf[curunit.lengthunit];
}

/**
 * \fn double user2SIpricepertrip(double ppt)
 * \brief Converts price per trip from user units to SI units
 * \param ppt Price per trip in SI units
 * \return Price per trip SI units
 */
double user2SIpricepertrip(double ppt)
{
  return ppt/lcf[curunit.lengthunit];
}


/**
 * \fn double grams2kg(double grams)
 * \brief Converts grams to kilograms if massunit is SI, otherwise do nothing
 * \param grams Mass in grams
 * \return Kilograms if massunit is SI
 */
double grams2kg(double grams)
{
  double result;

  if (curunit.massunit==SI)
    result=grams/1000;
  else
    result=grams;
  return result;
}

/**
 * \fn void convdate(char **s, const char *format, const char *date)
 * \brief Converts date with given format to %Y-%m-%d format
 * \param **s pointer to buffer where converted date will be stored
 * \param *format input date format in the format undestood by strptime
 * \param *date input date
 *
 * This function is used to convert a date in given format to a format
 * that is understood as a date by sqlite3.
 *
 */
void convdate(char **s, const char *outformat, 
	      const char *informat, const char *date)
{
  time_t t1;
  struct tm *tm;

  *s=(char *)malloc((DATELENMAX+1)*sizeof(char));

  time(&t1);
  tm=localtime(&t1);
  strptime(date, informat, tm);
  strftime(*s, DATELENMAX, outformat, tm);
}

/**
 * \fn void void convunixtime(char **s, const char *outformat, time_t time)
 * \brief Converts unix time to given format
 * \param **s pointer to buffer where converted date will be stored
 * \param *outformat output time format, see man strftime
 * \param time input unix time
 *
 * This function is used to convert a time in unix format to a human readable
 * time format.
 *
 */
void convunixtime(char **s, const char *outformat, time_t time)
{
  struct tm *tm;

  *s=(char *)malloc((DATELENMAX+1)*sizeof(char));

  tm=gmtime(&time);
  strftime(*s, DATELENMAX, outformat, tm);
}

void date2sqlite(char **s, const char *format, const char *date)
{
  convdate(s, "%Y-%m-%d", format, date);
}

void gettimefmt(char **aika, const char *format)
{
  time_t t1;
  struct tm *lctime;

  *aika=(char *)malloc((TIMELENMAX+1)*sizeof(char));

  time(&t1);
  lctime=localtime(&t1);
  strftime(*aika, TIMELENMAX, format, lctime);
}

void unixtimefmt(const time_t unixtime, char **aika, const char *format)
{
  time_t t1;
  struct tm *lctime;

  *aika=(char *)malloc((DATELENMAX+1)*sizeof(char));

  t1=unixtime;
  lctime=localtime(&t1);
  strftime(*aika, DATELENMAX, format, lctime);
}

void gettime(char **aika)
{
  gettimefmt(aika, "%F");
}

int getyear(void)
{
  char *aika;
  int ret;

  gettimefmt(&aika, "%Y");
  if (strlen(aika)>0)
    ret=atof(aika);
  else
    ret=2009; /* fallback to something reasonable */
  free(aika);
  return ret;
}

void getyearmonthday(const char *datestr, int *year, int *month, int *day)
{
  char *date;

  convdate(&date, "%Y", "%Y-%m-%d", datestr);
  *year=atoi(date);
  if (date != NULL) free(date);

  convdate(&date, "%m", "%Y-%m-%d", datestr);
  *month=atoi(date);
  if (date != NULL) free(date);

  convdate(&date, "%d", "%Y-%m-%d", datestr);
  *day=atoi(date);
  if (date != NULL) free(date);
}

time_t getmktime(void)
{
  time_t t1;
  struct tm *lctime;
  time_t output;

  time(&t1);
  lctime=localtime(&t1);
  output=mktime(lctime);

  return output;
}

/**
 * \fn double doubleornothing(double input)
 * \brief Returns zero if input is nan or inf, otherwise returns input
 * \param input input value
 *
 * This function returns zero (0.0) if the input is nan or inf, otherwise
 * the input is returned as is.
 *
 */
double doubleornothing(double input)
{
  double ret;

  if (isnan(input) || isinf(input))
    ret=0.0;
  else
    ret=input;

  return ret;
}

char *store_locale(int category)
{
  char *currentlocale;
  char *psave;
  size_t len;

  currentlocale=setlocale(category, NULL);
  len=strlen(currentlocale);
  psave=(char *)malloc((len+1)*sizeof(char));
  strncpy(psave, currentlocale, len+1);

  return psave;
}

void restore_locale(int category, char *psave)
{
  setlocale(category, psave);
  free(psave);
}

/**
 * \fn double atof_l(const char *nptr, const char *locale)
 * \brief Locale independent atof
 * \param *nptr string to be converted
 * \param *locale locale used in conversion
 * \return String as numeric value
 *
 * Locale independent atof. Changes the locale to *locale, makes the
 * conversion and changes the locale back to original.
 *
 */
double atof_l(const char *nptr, const char *locale)
{
  char *currentlocale;
  char *psave;
  size_t len;
  double ret;

  if (nptr==NULL)
    ret=0.0;
  else {
    currentlocale=setlocale(LC_NUMERIC, NULL);
    len=strlen(currentlocale);
    psave=(char *)malloc((len+1)*sizeof(char));
    strncpy(psave, currentlocale, len+1);
    
    setlocale(LC_NUMERIC, locale);
    ret=atof(nptr);

    setlocale(LC_NUMERIC, psave);
    free(psave);
  }

  return ret;
}

/**
 * \fn float get_emission_per_litre(int fueltype)
 * \brief Returns the CO2 emission in g/l
 * \param fueltype 0 for petrol, 1 for diesel
 * \return CO2 emission in g/l
 *
 */
float get_emission_per_litre(int fueltype)
{
  float emissionperlitre[]={2350.0, 2660.0};

  return emissionperlitre[fueltype];

}

/**
 * \fn gboolean is_sqlite3_file(const char *filename)
 * \brief Checks if a file is an SQLite 3 file
 * \param filename full path to the file
 * \return true if the file is an SQLite 3 file
 *
 */
gboolean is_sqlite3_file(const char *filename)
{
 FILE *f;
 const char sqlite3fingerprint[]="SQLite format 3";
 char readfingerprint[sizeof(sqlite3fingerprint)/sizeof(char)];
 size_t numread;
 gboolean ret=FALSE;

 f=fopen (filename, "rb");
 numread=fread(readfingerprint, sizeof(char), sizeof(sqlite3fingerprint)/sizeof(char), f);

 if (numread == sizeof(sqlite3fingerprint)/sizeof(char)) {
   if (strncmp(sqlite3fingerprint, readfingerprint, sizeof(sqlite3fingerprint)/sizeof(char)) == 0) {
       ret=TRUE;
   }
 }

 fclose(f);
 return ret;
}
