/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <math.h>

#include "uisettings.h"
#include "uiexpimp.h"
#include "uiview.h"

/**
 *
 * \file
 * \brief  Settings part of GTK+/Hildon based user interface to fuelpad
 * \author Julius Luukko
 *
 *
 */

/*******************************************************************
 *
 * Private defines
 *
 *******************************************************************/

#define NICKMAXLEN 30

#define RESPONSE_SELECT_COLUMNS 1
#define MAXWARNLEN 10

/*******************************************************************
 *
 * Public variables stored in gconf
 *
 *******************************************************************/

/* uialarm.c */
double warnkm=1000.0;
double warndays=7.0;

/* uireport.c */
/* Paper used in report and statistics */
enum papertype reportpaper=A4PAPER;

/* Classic column view or the new fancier one */
enum userinterfacestyle uistyle=UIFANCY;

/* Which columns are shown */
gint dispcol=DISPCOLDEFAULT;

/* uidrivinglog.c */
gboolean autostartGPS=FALSE;
gboolean autostartnetwork=FALSE;
double locationaliasradius=50.0;

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

/* Possible date format strings (see strptime(3) for format descri ption) */
char *datefmtstr[]={"%Y-%m-%d", "%d.%m.%Y", "%d/%m/%Y", "%d/%m/%y", 
		    "%d-%m-%Y", "%m/%d/%Y", "%m/%d/%y"};


/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

/* Warn times */
gchar *warntexts[] = {gettext_noop("1 day"),
		      gettext_noop("2 days"),
		      gettext_noop("1 week"),
		      gettext_noop("2 weeks"),
		      gettext_noop("1 month")};

/* Possible values of warning times in days */
float wtimes[]={1.0, 2.0, 7.0, 14.0, 30.0};


/*******************************************************************
 *
 * Private function prototypes
 *
 *******************************************************************/

static
double get_entrywarnkm(AppUIData *pui);
static
void update_warn_label(GtkWidget *label);
static
void update_warn_entry(GtkWidget *entry, double value);
static
void settings_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static
void uistyle_changed_cb (GtkWidget *widget, AppUIData *pui );
static
void paper_changed_cb (GtkWidget *widget, gpointer   callback_data );
static
void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data );
static
enum unit get_unit(GtkWidget *widget);
static
void unit_changed_cb (GtkWidget *widget, AppUIData *pui );
static
void l_unit_changed_cb(GtkWidget *widget, AppUIData *pui);
static
void v_unit_changed_cb(GtkWidget *widget, AppUIData *pui);
static
void c_unit_changed_cb(GtkWidget *widget, AppUIData *pui);
static
void m_unit_changed_cb(GtkWidget *widget, AppUIData *pui);
static
void dateformat_changed_cb (GtkWidget *widget, gpointer   callback_data );
static
GtkWidget *create_warntime_combo(AppUIData *pui);
static
GtkWidget *create_uistyle_combo(AppUIData *pui);
static
GtkWidget *create_paper_combo(AppUIData *pui);
static
GtkWidget *create_dateformat_combo(AppUIData *pui);
static
GtkWidget *create_fueltype_combo(AppUIData *pui);
static
void set_columns_visible(AppUIData *pui);
static
void select_columns_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui, GtkWidget *button);
#else
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui);
#endif
static
void create_car_editwin(AppUIData *pui, GtkWidget *dialog);
static
void create_driver_editwin(AppUIData *pui, GtkWidget *dialog);


/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

static
double get_entrywarnkm(AppUIData *pui)
{
  double warn;

  warn = atof(gtk_entry_get_text(GTK_ENTRY(pui->entrywarnkm)));
  return user2SIlength(warn);
}

static
void update_warn_label(GtkWidget *label)
{
  if (curunit.lengthunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("km before"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("miles before"));
}

static
void update_warn_entry(GtkWidget *entry, double value)
{
  GString *s;

  s=g_string_new(NULL);
  g_string_printf(s, "%.0f", SIlength2user(value));
  gtk_entry_set_text(GTK_ENTRY(entry),s->str);
  g_string_free(s, TRUE);
}

static
void settings_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gboolean warn;
  GString *str;

  switch (arg1) {
  case GTK_RESPONSE_ACCEPT:
    g_string_assign(curunit.currency, gtk_entry_get_text(GTK_ENTRY(pui->entrycurrency)));
    warnkm=get_entrywarnkm(pui);
    update_car_changed(pui);

    /* GPS and location */
    autostartnetwork = get_check_button(pui->autonetcheck);
    autostartGPS = get_check_button(pui->autogpscheck);
    str = get_entry(pui->entrylocationsradius);
    locationaliasradius = atof(str->str);
    g_string_free(str, TRUE);

    /* Done reading the dialog widgets */
    gtk_widget_destroy(GTK_WIDGET(dialog));

    /* Update the next event estimates */
    pui->alarmview = create_alarmview_and_model (pui);
    pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
    update_reminder_toolbutton (pui, pui->warn);

    /* Show or hide toolbars */
    main_toolbar_show_hide(pui);
    secondary_toolbar_show_hide(pui);

#if MAEMO_VERSION_MAJOR == 5
    update_car_combo(pui->stb_carbutton);
#else
    update_car_combo(pui->stb_carcombo);
#endif

    set_columns_visible(pui);

    break;
  case GTK_RESPONSE_HELP:
    callback_about(NULL, NULL);
    break;
  case RESPONSE_SELECT_COLUMNS:
    callback_select_columns(NULL, pui);
    break;
  }
}

/**
 * \fn void uistyle_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief UI style change callback
 *
 */
static
void uistyle_changed_cb (GtkWidget *widget, AppUIData *pui )
{
  unsigned int index;

#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelector *selector;
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  uistyle=(enum userinterfacestyle)index;

  if (uistyle == UICLASSIC) {
    gtk_widget_set_sensitive(pui->selectcolumnsbutton, TRUE);
#if MAEMO_VERSION_MAJOR == 5
    gtk_widget_show(pui->selectcolumnsbutton);
#endif
  }
  else {
    gtk_widget_set_sensitive(pui->selectcolumnsbutton, FALSE);
#if MAEMO_VERSION_MAJOR == 5
    gtk_widget_hide(pui->selectcolumnsbutton);
#endif
  }

}

/**
 * \fn void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for report paper combo changed
 *
 */
static
void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  reportpaper=(enum papertype)index;
}

/**
 * \fn void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for alarm warning time combo changed
 *
 */
static
void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  warndays=wtimes[index];
}

static
enum unit get_unit(GtkWidget *widget)
{
  enum unit tmpunit;

#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelector *selector;
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));
  tmpunit=(enum unit)hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0);
#else
  tmpunit = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  if (tmpunit>IMPERIAL || tmpunit < SI) tmpunit=SI;

  return tmpunit;
}


/**
 * \fn void unit_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for unit selection combo boxes
 * \param callback_data selects between different unit combo boxes
 *
 * This callback sets the corresponding unit variable (mainunit,
 * lengthunit, volumeunit, consumeunit) to the value
 * read from the combo box.
 *
 */
static
void unit_changed_cb (GtkWidget *widget, AppUIData *pui )
{
  enum unit tmpunit;
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  else
    oldwarnkm=0.0;
  tmpunit=get_unit(widget);
  curunit.mainunit = tmpunit;
  if (!curunit.individualunit) {
    curunit.lengthunit = tmpunit;
    curunit.volumeunit = tmpunit;
    curunit.consumeunit = tmpunit;
    if (pui->lengthcombo != NULL) {
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->lengthcombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->volumecombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->consumecombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->masscombo), 0, (gint)tmpunit);
#else
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->lengthcombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->volumecombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->consumecombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->masscombo),tmpunit);
#endif
    }
  }
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

static
void l_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  else
    oldwarnkm=0.0;
  curunit.lengthunit = get_unit(widget);
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

static
void v_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.volumeunit = get_unit(widget);
}

static
void c_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.consumeunit = get_unit(widget);
}

static
void m_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.massunit = get_unit(widget);
}

/**
 * \fn void dateformat_changed_cb (GtkWidget *widget, gpointer
 *                                 callback_data ) 
 * \brief Callback for date format selection combo box
 *
 * This callback changes the dateformat in the curunit structure
 * according to the selection made in the date format combo box.
 *
 */
static
void dateformat_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
#if MAEMO_VERSION_MAJOR == 5
  curunit.dateformat=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  curunit.dateformat = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
}

static
GtkWidget *create_warntime_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint i;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif

  for (i=0;i<sizeof(warntexts)/sizeof(char *);i++) {
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), gettext(warntexts[i]));
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo),gettext(warntexts[i]));
#endif
  }
  i=0;
  while(i<sizeof(wtimes)/sizeof(float) && warndays!=wtimes[i])
    i++;
#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, i);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),i);
#endif
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(warntime_changed_cb), NULL);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->remindbutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  return combo;
}

static
GtkWidget *create_uistyle_combo(AppUIData *pui)
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),_("Modern"));
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),_("Classic"));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->uibutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Fancy");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Classic");
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(uistyle_changed_cb), pui);
#endif


  switch (uistyle) {
  case UIFANCY:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
#endif
    break;
  case UICLASSIC:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,1);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),1);
#endif
  }

  return combo;
}

static
GtkWidget *create_paper_combo(AppUIData *pui)
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),"A4");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),"Letter");
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->paperbutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"A4");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Letter");
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(paper_changed_cb), NULL);

  switch (reportpaper) {
  case A4PAPER:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
#endif
    break;
  case LETTERPAPER:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,1);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),1);
#endif
  }

  return combo;
}

static
GtkWidget *create_dateformat_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint       i;
  char *timestr;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(dateformat_changed_cb), NULL);
  for (i=0;i<sizeofdatefmtstr();i++) {
    gettimefmt(&timestr, datefmtstr[i]);
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), timestr);
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo),timestr);
#endif
    free(timestr);
  }

#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, curunit.dateformat);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),curunit.dateformat);
#endif

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->datebutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  return combo;
}

static
GtkWidget *create_fueltype_combo(AppUIData *pui)
{
  enum
  {
    FUELTYPE_PETROL=0,
    FUELTYPE_DIESEL
  };

  GtkWidget *combo;
  gchar *labels[] = {gettext_noop("Petrol"),
		   gettext_noop("Diesel")
  };

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[FUELTYPE_PETROL]));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[FUELTYPE_DIESEL]));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 0);
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->fueltypebutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[FUELTYPE_PETROL]));
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[FUELTYPE_DIESEL]));
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 0);
#endif
  return combo;
}

/* \todo: This duplicates a lot of what is in select_columns_response() */
static
void set_columns_visible(AppUIData *pui)
{
  gint i;

  if (uistyle==UIFANCY) {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(pui->view), FALSE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ICON), TRUE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_COMBINED), TRUE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ID), FALSE);
    for (i=COL_DAY;i<=COL_NOTES;i++) {
      gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),i), FALSE);
    }
  } else {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(pui->view), TRUE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ICON), FALSE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_COMBINED), FALSE);
    for (i=COL_DAY;i<=COL_NOTES;i++) {
      gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),i), dispcol & (1<<i));
    }
  }
}

static
void select_columns_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gint i;
  gboolean checked;

  if (uistyle==UIFANCY) {
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ICON), TRUE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_COMBINED), TRUE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ID), FALSE);
    for (i=COL_DAY;i<=COL_NOTES;i++) {
      gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),i), FALSE);
    }
  } else {
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_ICON), FALSE);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),COL_COMBINED), FALSE);
    for (i=COL_DAY;i<=COL_NOTES;i++) {
#if MAEMO_VERSION_MAJOR == 5
      checked = hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->columncheck[i]));
#else
      checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->columncheck[i]));
#endif
      if (checked)
	dispcol |= (1<<i);
      else
	dispcol &= ~(1<<i);
      gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),i), dispcol & (1<<i));
    }
  }

  /* Note! Was this really needed in Diablo? */
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_widget_destroy(GTK_WIDGET(pui->mainvbox));
  create_mainwin_widgets(pui);
#endif

  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param whichunit UNITCOMBO, LENGTHCOMBO, VOLUMECOMBO, CONSUMECOMBO or MASSCOMBO
 * \param pui pointer to user interface data
 * \return Pointer to the created combo box
 *
 * This function is used to create a unit selection combo box. The
 * unit selection combo box containts selections SI, US and
 * Imperial. The default value is read from mainunit,
 * lengthunit, volumeunit or consumeunit
 * depending on the value of the parameter whichunit.
 *
 */
#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
#endif
{
  GtkWidget *combo;
  enum unit tmpunit=curunit.mainunit;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  
#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "SI");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "US");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "Imperial");
#else
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"SI");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"US");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Imperial");
#endif

  switch (whichunit) {
  case UNITCOMBO: 
    tmpunit = curunit.mainunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(unit_changed_cb), pui);
#endif
    break;
  case LENGTHCOMBO:
    tmpunit = curunit.lengthunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(l_unit_changed_cb), pui);
#endif
    break;
  case VOLUMECOMBO:
    tmpunit = curunit.volumeunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(v_unit_changed_cb), pui);
#endif
    break;
  case CONSUMECOMBO:
    tmpunit = curunit.consumeunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(c_unit_changed_cb), pui);
#endif
    break;
  case MASSCOMBO:
    tmpunit = curunit.massunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(m_unit_changed_cb), pui);
#endif
    break;
  }
#if MAEMO_VERSION_MAJOR == 5
/*     hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit); */
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),tmpunit);
#endif


  return combo;
}

/**
 * \fn void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a car editing window
 *
 * This function is used for creating an edit window both for adding a
 * new car and editing an existing one.
 *
 */
static
void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
{
  enum
  {
    CAR_EDIT_MARK=0,
    CAR_EDIT_MODEL,
    CAR_EDIT_YEAR,
    CAR_EDIT_REGISTER,
    CAR_EDIT_FUELTYPE,
    CAR_EDIT_NOTES
  };

  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *scrollwin;
  gchar *labels[] = {gettext_noop("Mark"),
		   gettext_noop("Model"),
		   gettext_noop("Year"),
		   gettext_noop("Register"),
		   gettext_noop("Fuel type"),
		   gettext_noop("Notes")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
#else
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
#endif
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);

  /* Mark */
  table = gtk_table_new(6, 2, FALSE);
/*   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5); */
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);
#else
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);
#endif

  /* Mark */
  label=gtk_label_new(gettext(labels[CAR_EDIT_MARK]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrymark=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrymark = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymark), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymark, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymark);

  /* Model */
  label=gtk_label_new(gettext(labels[CAR_EDIT_MODEL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrymodel=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrymodel = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymodel), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymodel, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymodel);

  /* Year */
  label=gtk_label_new(gettext(labels[CAR_EDIT_YEAR]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryyear=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryyear = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryyear), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryyear, 1, 2, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryyear);

  /* Register */
  label=gtk_label_new(gettext(labels[CAR_EDIT_REGISTER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryregister=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryregister = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryregister), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryregister, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryregister);

  /* Fueltype */
#if MAEMO_VERSION_MAJOR == 5
  pui->fueltypebutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->fueltypebutton), gettext(labels[CAR_EDIT_FUELTYPE]));
  gtk_table_attach(GTK_TABLE(table), pui->fueltypebutton, 0, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->fueltypebutton);
#else
  label=gtk_label_new(gettext(labels[CAR_EDIT_FUELTYPE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);
#endif

  pui->combofueltype = create_fueltype_combo(pui);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_table_attach(GTK_TABLE(table), pui->combofueltype, 1, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->combofueltype);
#endif

  /* Notes */
  label=gtk_label_new(gettext(labels[CAR_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrycarnotes=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrycarnotes = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycarnotes), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrycarnotes, 1, 2, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycarnotes);

  gtk_widget_show(table);
  gtk_widget_show(scrollwin);
}

/**
 * \fn void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a driver editing window
 *
 * This function is used for creating an edit window both for adding a
 * new driver and editing an existing one.
 *
 */
static
void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
{
  GtkWidget *table;
  GtkWidget *label;

  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5);

  label=gtk_label_new(_("Full name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryname=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryname = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryname), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryname, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryname);

  label=gtk_label_new(_("Short name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrynick=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrynick = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynick), NICKMAXLEN);
  gtk_table_attach(GTK_TABLE(table), pui->entrynick, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynick);

  gtk_widget_show(table);

}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

gint sizeofdatefmtstr(void)
{
  return (sizeof(datefmtstr)/sizeof(char *));
}

#if MAEMO_VERSION_MAJOR == 5
#define SELCOLLINES (float)(COL_NOTES-COL_DAY+1)
#else
#define SELCOLLINES 4.0
#endif
void callback_select_columns( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *scrollwin;
#endif
  GString *header;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  gint i;
  gint j;
  gint col;
  gfloat numcols;

  dialog = gtk_dialog_new_with_buttons(_("Select columns"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

  numcols=ceilf(COL_ID/SELCOLLINES);

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);
#endif
  table = gtk_table_new(numcols, COL_ID/numcols, FALSE);
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);
#else
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5);
#endif

  for (col=0,j=0,i=COL_DAY;i<=COL_NOTES;i++,j++) {
    if (j==(int)SELCOLLINES) {
      j=0;
      col++;
    }

    header = get_column_header_one_line(i);
#if MAEMO_VERSION_MAJOR == 5
    pui->columncheck[i]=hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label (GTK_BUTTON(pui->columncheck[i]), header->str);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->columncheck[i]), dispcol & (1<<i));
#else
    pui->columncheck[i]=gtk_check_button_new_with_label (header->str);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->columncheck[i]), dispcol & (1<<i));
#endif
    g_string_free(header, TRUE);
    gtk_table_attach(GTK_TABLE(table), pui->columncheck[i], col, col+1, j, j+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (pui->columncheck[i]);

#if MAEMO_VERSION_MAJOR == 5
/*     g_free(newstr); */
#endif

  }

  gtk_widget_show(table);
#if MAEMO_VERSION_MAJOR == 5
  gtk_widget_show(scrollwin);
#endif

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_MAINWIN,
				   pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (select_columns_response),
		    pui);
  gtk_widget_show(dialog);

}

/**
 * \fn void add_car_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new car data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function add_car which 
 * is called with the data (i.e. sqlite would not be used here)
 *
 */
static
void add_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *listitemtext;
  GString *mark;
  GString *model;
  GString *year;
  GString *regist;
  GString *notes;
  gint fueltype;

  mark=g_string_new(NULL);
  model=g_string_new(NULL);
  year=g_string_new(NULL);
  regist=g_string_new(NULL);
  notes=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(mark,gtk_entry_get_text(GTK_ENTRY(pui->entrymark)));
      g_string_assign(model,gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)));
      g_string_assign(year,gtk_entry_get_text(GTK_ENTRY(pui->entryyear)));
      g_string_assign(regist,gtk_entry_get_text(GTK_ENTRY(pui->entryregister)));
      g_string_assign(notes,gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)));
#if MAEMO_VERSION_MAJOR == 5
      fueltype=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->combofueltype), 0);
#else
      fueltype=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->combofueltype));
#endif
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 1, 
					 mark->str,
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 2,
					    model->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtAddCar, 3,
					   atoi(year->str))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 4,
					    regist->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 5,
					    notes->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 6,
					   fueltype)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddCar)) {
	PDEBUG("Problem saving the new car");
      }
      else { /* New car added */
#if MAEMO_VERSION_MAJOR == 5
	update_car_combo(pui->carbutton);
	update_car_combo(pui->stb_carbutton);
#else
	update_car_combo(pui->carcombo);
	update_car_combo(pui->stb_carcombo);
#endif
      }
      sqlite3_reset(ppStmtAddCar);
      g_string_free(mark, TRUE);
      g_string_free(model, TRUE);
      g_string_free(year, TRUE);
      g_string_free(regist, TRUE);
      g_string_free(notes, TRUE);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void edit_car_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for edit car dialog
 *
 * This callback changes the entered driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
static
void edit_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *mark;
  GString *model;
  GString *year;
  GString *regist;
  GString *notes;
  gint fueltype;

  mark=g_string_new(NULL);
  model=g_string_new(NULL);
  year=g_string_new(NULL);
  regist=g_string_new(NULL);
  notes=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(mark,gtk_entry_get_text(GTK_ENTRY(pui->entrymark)));
      g_string_assign(model,gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)));
      g_string_assign(year,gtk_entry_get_text(GTK_ENTRY(pui->entryyear)));
      g_string_assign(regist,gtk_entry_get_text(GTK_ENTRY(pui->entryregister)));
      g_string_assign(notes,gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)));
#if MAEMO_VERSION_MAJOR == 5
      fueltype=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->combofueltype), 0);
#else
      fueltype=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->combofueltype));
#endif
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 1,
					 mark->str,
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 2,
					    model->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 3,
					   atoi(year->str))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 4,
					    regist->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 5,
					    notes->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 6,
					   fueltype)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 7,
					   currentcar)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateCar)) {
	PDEBUG("Problem udpating car properties");
      }
      else {
	PDEBUG("Car data updated\n");
      }
      sqlite3_reset(ppStmtUpdateCar);
#if MAEMO_VERSION_MAJOR == 5
      update_car_combo(pui->carbutton);
      update_car_combo(pui->stb_carbutton);
#else
      update_car_combo(pui->carcombo);
      update_car_combo(pui->stb_carcombo);
#endif
      g_string_free(mark, TRUE);
      g_string_free(model, TRUE);
      g_string_free(year, TRUE);
      g_string_free(regist, TRUE);
      g_string_free(notes, TRUE);
    }
    else {
      PDEBUG("Can't update driver properties - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void add_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * add_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
static
void add_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *name;
  GString *nick;

  name=g_string_new(NULL);
  nick=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(name,gtk_entry_get_text(GTK_ENTRY(pui->entryname)));
      g_string_assign(nick,gtk_entry_get_text(GTK_ENTRY(pui->entrynick)));
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 1, 
					 name->str, 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 2, 
					    nick->str, 
					    -1, SQLITE_STATIC)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddDriver)) {
	PDEBUG("Problem saving the new driver");
      }
      else { /* New driver added */
#if MAEMO_VERSION_MAJOR == 5
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (pui->drivercombo), nick->str);
#else
	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo), nick->str);
#endif
      }
      sqlite3_reset(ppStmtAddDriver);
      g_string_free(nick, TRUE);
      g_string_free(name, TRUE);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void edit_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback editing driver data
 *
 * This callback updates the driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
static
void edit_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *name;
  GString *nick;

  name=g_string_new(NULL);
  nick=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(name,gtk_entry_get_text(GTK_ENTRY(pui->entryname)));
      g_string_assign(nick,gtk_entry_get_text(GTK_ENTRY(pui->entrynick)));
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 1, 
					 name->str, 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 2, 
					    nick->str, 
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateDriver, 3,
					   currentdriver)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateDriver)) {
	PDEBUG("Problem updatig the driver data");
      }
      else { /* New driver added */
/* 	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo),gtk_entry_get_text(GTK_ENTRY(pui->entrynick))); */
	PDEBUG("Driver data updated\n");
      }
      sqlite3_reset(ppStmtUpdateDriver);
      update_driver_combo(pui->drivercombo);
      g_string_free(nick, TRUE);
      g_string_free(name, TRUE);
    }
    else {
      PDEBUG("Can't update driver data - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}


/**
 * \fn void add_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new car to the database
 *
 * This callback opens a dialog which allows to add a new car to the
 * database.
 *
 */
static
void add_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new car"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void edit_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for edit car properties
 *
 * This callback opens a dialog which allows to edit car properties.
 *
 */
void edit_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit car properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneCar, 1, currentcar)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneCar)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entrymark),sqlite3_column_text(ppStmtOneCar,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrymodel),sqlite3_column_text(ppStmtOneCar,1));
      gtk_entry_set_text(GTK_ENTRY(pui->entryyear),sqlite3_column_text(ppStmtOneCar,2));
      gtk_entry_set_text(GTK_ENTRY(pui->entryregister),sqlite3_column_text(ppStmtOneCar,3));
      gtk_entry_set_text(GTK_ENTRY(pui->entrycarnotes),sqlite3_column_text(ppStmtOneCar,4));
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (pui->combofueltype), 0, sqlite3_column_int(ppStmtOneCar,5));
#else
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->combofueltype),sqlite3_column_int(ppStmtOneCar,5));
#endif
    }
    sqlite3_reset(ppStmtOneCar);
  }

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void delete_car_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a car from the database
 *
 * This callback deletes a car from the database. All the records and
 * events for this car are also deleted. The user is asked for a
 * confirmation before deleting.
 *
 */
static
void delete_car_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char *car;

  if (db_open()) { /* delete */
    if (db_get_numof_cars() <= 1) {
      message=g_string_new(_("The car can't be deleted. There is only one car left and the database must contain at least one car."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_carrecords(currentcar);
      numevents=db_get_numof_carevents(currentcar);
      db_get_current_car(&car);
      message=g_string_new(_("Are you sure you want to delete car"));
      g_string_append(message, " ");
      g_string_append(message, car);
      if (car != NULL)
	free(car);
      g_string_append(message, ". ");
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this car."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this car, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this car, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this car, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_car(currentcar)) {
	  PDEBUG("Problem deleting the car");
	}
	else { /* remove the record if succesfully removed from db */
#if MAEMO_VERSION_MAJOR == 5
      update_car_combo(pui->carbutton);
      update_car_combo(pui->stb_carbutton);
#else
	  update_car_combo(pui->carcombo);
	  update_car_combo(pui->stb_carcombo);
#endif
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void edit_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
static
void edit_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit driver properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneDriver, 1, currentdriver)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneDriver)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entryname),sqlite3_column_text(ppStmtOneDriver,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrynick),sqlite3_column_text(ppStmtOneDriver,1));
    }
    sqlite3_reset(ppStmtOneDriver);
  }

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void add_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
static
void add_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new driver"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void delete_driver_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a driver from the database
 *
 * This callback deletes a driver from the database. All the records
 * and events, which have this driver as a driver are also deleted. The
 * user is asked for a confirmation before deleting.
 *
 */
static
void delete_driver_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char *nickname;

  if (db_open()) { /* delete */
    if (db_get_numof_drivers() <= 1) {
      message=g_string_new(_("The driver can't be deleted. There is only one driver left and the database must contain at least one driver."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_records(currentdriver);
      numevents=db_get_numof_events(currentdriver);
      db_get_current_driver(&nickname);
      message=g_string_new(_("Are you sure you want to delete driver"));
      g_string_append(message, " ");
      g_string_append(message, nickname);
      g_string_append(message, ". ");
      if (nickname != NULL)
	free(nickname);
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this driver."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this driver, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this driver, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this driver, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_driver(currentdriver)) {
	  PDEBUG("Problem deleting the driver");
	}
	else { /* remove the record if succesfully removed from db */
	  update_driver_combo(pui->drivercombo);
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void unit_check_cb (GtkWidget *widget, AppUIData *pui )
 * \brief Callback for check button for selecting the unit individually
 *
 * This function either enables or disables the unit selection
 * comboboxes and sets the global variable individualunit according to
 * the state of the check button.
 *
 */
static
void unit_check_cb (GtkWidget *widget, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
if (hildon_check_button_get_active (HILDON_CHECK_BUTTON (widget))) 
#else
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget))) 
#endif
  {
    /* If control reaches here, the toggle button is down */
    curunit.individualunit = TRUE;
#if MAEMO_VERSION_MAJOR == 5
       gtk_widget_set_sensitive(pui->unitsystembutton, FALSE);
       gtk_widget_set_sensitive(pui->lengthunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->volumeunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->consumeunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->massunitbutton, TRUE);
       gtk_widget_show(pui->lengthunitbutton);
       gtk_widget_show(pui->volumeunitbutton);
       gtk_widget_show(pui->consumeunitbutton);
       gtk_widget_show(pui->massunitbutton);
#else
    gtk_widget_set_sensitive(pui->unitcombo, FALSE);
    gtk_widget_set_sensitive(pui->lengthcombo, TRUE);
    gtk_widget_set_sensitive(pui->volumecombo, TRUE);
    gtk_widget_set_sensitive(pui->consumecombo, TRUE);
    gtk_widget_set_sensitive(pui->masscombo, TRUE);
#endif
  } else {
  /* If control reaches here, the toggle button is up */
    curunit.individualunit = FALSE;
#if MAEMO_VERSION_MAJOR == 5
       gtk_widget_set_sensitive(pui->unitsystembutton, TRUE);
       gtk_widget_set_sensitive(pui->lengthunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->volumeunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->consumeunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->massunitbutton, FALSE);
       gtk_widget_hide(pui->lengthunitbutton);
       gtk_widget_hide(pui->volumeunitbutton);
       gtk_widget_hide(pui->consumeunitbutton);
       gtk_widget_hide(pui->massunitbutton);
       unit_changed_cb(pui->unitsystembutton, pui);
#else
    gtk_widget_set_sensitive(pui->unitcombo, TRUE);
    gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
    gtk_widget_set_sensitive(pui->volumecombo, FALSE);
    gtk_widget_set_sensitive(pui->consumecombo, FALSE);
    gtk_widget_set_sensitive(pui->masscombo, FALSE);
    unit_changed_cb(pui->unitcombo, pui);
#endif
 }
}

void callback_settings( GtkAction * action, AppUIData *pui )
{
  enum
  {
    SETTINGS_TITLE=0,
    SETTINGS_DATABASE,
    SETTINGS_OPEN,
    SETTINGS_IMPORT,
    SETTINGS_EXPORT,
    SETTINGS_DRIVER,
    SETTINGS_EDIT,
    SETTINGS_NEW,
    SETTINGS_DELETE,
    SETTINGS_CAR,
    SETTINGS_DATEFORMAT,
    SETTINGS_WARN,
    SETTINGS_REMIND,
    SETTINGS_CURRENCY,
    SETTINGS_UNITSYSTEM,
    SETTINGS_INDIVIDUAL,
    SETTINGS_LENGTHUNIT,
    SETTINGS_VOLUMEUNIT,
    SETTINGS_CONSUMEUNIT,
    SETTINGS_MASSUNIT,
    SETTINGS_UISTYLE,
    SETTINGS_SELECTCOLUMNS,
    SETTINGS_PAPER,
    SETTINGS_AUTOMATIC_NETWORK,
    SETTINGS_AUTOMATIC_GPS,
    SETTINGS_LOCATIONS_RADIUS
  };

  enum
  {
    SETTINGS_HDR_DATABASE=0,
    SETTINGS_HDR_CARDRIVER,
    SETTINGS_HDR_REMINDERS,
    SETTINGS_HDR_UNITS,
    SETTINGS_HDR_USERINTERFACE,
    SETTINGS_HDR_REPORT,
    SETTINGS_HDR_DRIVINGLOG,
    SETTINGS_HDR_GPS,
    SETTINGS_HDR_LOCATIONS
  };


#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *scrollwin;
  GtkWidget *hbox;
  GtkWidget *svbox;
  GtkWidget *uvbox;
  GtkWidget *frame;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *unitcheck;
  GtkWidget *unitscrollwin;
  GtkWidget *check;
  gint       i;
  gint       row;
  GString   *buf;
  gchar *labels[] = {gettext_noop("Settings"),
		   gettext_noop("Database"),
		   gettext_noop("Open"),
		   gettext_noop("Import"),
		   gettext_noop("Export"),
		   gettext_noop("Driver"),
		   gettext_noop("Edit"),
		   gettext_noop("New"),
		   gettext_noop("Delete"),
		   gettext_noop("Car"),
		   gettext_noop("Date format"),
		   gettext_noop("Warn"),
		   gettext_noop("Remind before"),
		   gettext_noop("Currency"),
		   gettext_noop("Unit system"),
		   gettext_noop("Select individually"),
		   gettext_noop("Length unit"),
		   gettext_noop("Volume unit"),
		   gettext_noop("Fuel consumption unit"),
		   gettext_noop("Mass unit"),
		   gettext_noop("Table view style"),
		   gettext_noop("Select columns"),
		   gettext_noop("Paper"),
		   gettext_noop("Start network automatically"),
		   gettext_noop("Start automatically"),
		   gettext_noop("Recognition radius"),
  };
  gchar *headers[] = {gettext_noop("Database"),
		      gettext_noop("Driver and car"),
		      gettext_noop("Reminders"),
		      gettext_noop("Units"),
		      gettext_noop("User interface"),
		      gettext_noop("Report"),
		      gettext_noop("Driving log"),
		      gettext_noop("GPS"),
		      gettext_noop("Locations")
  };


#if MAEMO_VERSION_MAJOR == 5
  dialog = gtk_dialog_new_with_buttons(gettext(labels[SETTINGS_TITLE]),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       _("About"),
				       GTK_RESPONSE_HELP,
/* 				       _("Select columns"), */
/* 				       RESPONSE_SELECT_COLUMNS, */
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       NULL);
#else
  dialog = gtk_dialog_new_with_buttons(gettext(labels[SETTINGS_TITLE]),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       NULL);
#endif

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);
#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  svbox = gtk_vbox_new (FALSE, 0);
  row=0;
#else
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show (notebook);
#endif

  /** Driver and car notebook page **/

#if MAEMO_VERSION_MAJOR == 5

/*   table = gtk_table_new(16, 3, FALSE); */

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_CARDRIVER]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  row++;
  /* First row */

  table = gtk_table_new(4, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(frame), table);

  pui->driverbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->driverbutton), gettext(labels[SETTINGS_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), pui->driverbutton, 0, 1, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->driverbutton);

  pui->drivercombo = create_driver_combo(pui, pui->driverbutton);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_driver_cb), &ui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_driver_cb), &ui);
  gtk_widget_show(button);

/*   gtk_widget_show(table); */

  row++;
  /* Second row */

/*   table = gtk_table_new(4, 1, FALSE); */
/*   gtk_box_pack_start (GTK_BOX(svbox), table, TRUE, TRUE, 0); */

  pui->carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->carbutton), gettext(labels[SETTINGS_CAR]));
  gtk_table_attach(GTK_TABLE(table), pui->carbutton, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->carbutton);

  pui->carcombo = create_car_combo(pui, pui->carbutton);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_car_cb), pui);
  gtk_widget_show(button);

  row++;
  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_car_cb), pui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_car_cb), pui);
  gtk_widget_show(button);

  gtk_widget_show(table);

  row++;
  /* Third row */

/*   gtk_widget_show(table); */

  row++;

  /* First row */

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_REMINDERS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), hbox);

  label=gtk_label_new(gettext(labels[SETTINGS_WARN]));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  pui->entrywarnkm=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrywarnkm), MAXWARNLEN);
  update_warn_entry(pui->entrywarnkm, warnkm);
  gtk_box_pack_start (GTK_BOX(hbox), pui->entrywarnkm, TRUE, TRUE, 0);
  gtk_widget_show(pui->entrywarnkm);

  pui->labelentrywarn=gtk_label_new(NULL);
  update_warn_label(pui->labelentrywarn);
  gtk_box_pack_start (GTK_BOX(hbox), pui->labelentrywarn, TRUE, TRUE, 0);
  gtk_widget_show(pui->labelentrywarn);

  gtk_widget_show(hbox);

  row++;
  /* Second row */
  pui->remindbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->remindbutton), gettext(labels[SETTINGS_REMIND]));
  gtk_box_pack_start (GTK_BOX(svbox), pui->remindbutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->remindbutton);

  pui->warntimecombo=create_warntime_combo(pui);
  
  //  gtk_widget_show(table);

  row++;

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_UNITS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);

  /* First row */
  pui->datebutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->datebutton), gettext(labels[SETTINGS_DATEFORMAT]));
  gtk_box_pack_start (GTK_BOX(uvbox), pui->datebutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->datebutton);

  pui->datecombo = create_dateformat_combo(pui);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(uvbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(gettext(labels[SETTINGS_CURRENCY]));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  pui->entrycurrency=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycurrency), 30);
  gtk_entry_set_text(GTK_ENTRY(pui->entrycurrency),curunit.currency->str);
  gtk_box_pack_start (GTK_BOX(hbox), pui->entrycurrency, TRUE, TRUE, 0);
  gtk_widget_show(pui->entrycurrency);

  gtk_widget_show(hbox);

  row++;
  pui->unitsystembutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->unitsystembutton), gettext(labels[SETTINGS_UNITSYSTEM]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->unitsystembutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->unitsystembutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->unitsystembutton);

  unitcheck = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (unitcheck), gettext(labels[SETTINGS_INDIVIDUAL]));
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  gtk_box_pack_start (GTK_BOX(uvbox), unitcheck, TRUE, TRUE, 0);
  gtk_widget_show(unitcheck);

  row=row+2;
  pui->lengthunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->lengthunitbutton), gettext(labels[SETTINGS_LENGTHUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->lengthunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->lengthunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->lengthunitbutton);

  pui->lengthcombo = create_unit_combo(LENGTHCOMBO, pui, pui->lengthunitbutton);
  g_signal_connect(G_OBJECT(pui->lengthunitbutton), "value-changed",
		   G_CALLBACK(l_unit_changed_cb), pui);

  row++;
  pui->volumeunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->volumeunitbutton), gettext(labels[SETTINGS_VOLUMEUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->volumeunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->volumeunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->volumeunitbutton);

  pui->volumecombo = create_unit_combo(VOLUMECOMBO, pui, pui->volumeunitbutton);
  g_signal_connect(G_OBJECT(pui->volumeunitbutton), "value-changed",
		   G_CALLBACK(v_unit_changed_cb), pui);

  row++;
  pui->consumeunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->consumeunitbutton), gettext(labels[SETTINGS_CONSUMEUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->consumeunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->consumeunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->consumeunitbutton);

  pui->consumecombo = create_unit_combo(CONSUMECOMBO, pui, pui->consumeunitbutton);
  g_signal_connect(G_OBJECT(pui->consumeunitbutton), "value-changed",
		   G_CALLBACK(c_unit_changed_cb), pui);

  row++;
  pui->massunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->massunitbutton), gettext(labels[SETTINGS_MASSUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->massunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->massunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->massunitbutton);

  pui->masscombo = create_unit_combo(MASSCOMBO, pui, pui->massunitbutton);
  g_signal_connect(G_OBJECT(pui->massunitbutton), "value-changed",
		   G_CALLBACK(m_unit_changed_cb), pui);

  row=row-5;
  /* Must be the last combo since the toggle callback uses the other combos */
  pui->unitcombo = create_unit_combo(UNITCOMBO, pui, pui->unitsystembutton);
  g_signal_connect(G_OBJECT(pui->unitsystembutton), "value-changed",
		   G_CALLBACK(unit_changed_cb), pui);

  row++;
  /* Must be the last since the toggle callback uses the combos */
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON (unitcheck), curunit.individualunit);
  unit_check_cb (unitcheck, pui);

  row=row+5;

  gtk_widget_show(uvbox);

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_REPORT]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  /* First row */
  pui->paperbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->paperbutton), gettext(labels[SETTINGS_PAPER]));
  gtk_container_add (GTK_CONTAINER(frame), pui->paperbutton);
  gtk_widget_show(pui->paperbutton);

  pui->papercombo = create_paper_combo(pui);

  /* User interface */
  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_USERINTERFACE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox=gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);
  gtk_widget_show(uvbox);

  /* Need to create this before uibutton */
  pui->uibutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->uibutton), gettext(labels[SETTINGS_UISTYLE]));
/*   gtk_container_add (GTK_CONTAINER(frame), pui->uibutton); */
  gtk_box_pack_start (GTK_BOX(uvbox), pui->uibutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->uibutton);

  pui->uicombo = create_uistyle_combo(pui);

  pui->selectcolumnsbutton = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(pui->selectcolumnsbutton), gettext(labels[SETTINGS_SELECTCOLUMNS]));
  gtk_box_pack_start (GTK_BOX(uvbox), pui->selectcolumnsbutton, TRUE, TRUE, 0);
  gtk_widget_set_sensitive(pui->selectcolumnsbutton, uistyle==UICLASSIC);
  g_signal_connect (GTK_OBJECT (pui->selectcolumnsbutton), "clicked",
		    G_CALLBACK(callback_select_columns), pui);
  if (uistyle==UICLASSIC) {
    gtk_widget_show(pui->selectcolumnsbutton);
  }

  g_signal_connect (GTK_OBJECT (pui->uibutton), "value-changed",
		    G_CALLBACK(uistyle_changed_cb), pui);

  /* Database */
  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_DATABASE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(svbox), hbox, TRUE, TRUE, 0);

  pui->databasebutton = hildon_button_new_with_text (HILDON_SIZE_AUTO, 
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						     gettext(labels[SETTINGS_DATABASE]),
						     db_getfilename());
/*   gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,  */
/* 		   GTK_EXPAND|GTK_FILL, */
/* 		   0, 0, 5); */
  gtk_box_pack_start (GTK_BOX(hbox), pui->databasebutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->databasebutton);

  gtk_widget_show(hbox);

  /* Temporarily disabled */
/*   button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT); */
/*   gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_OPEN])); */
/*   gtk_box_pack_start (GTK_BOX(svbox), button, TRUE, TRUE, 0); */
/*   g_signal_connect(G_OBJECT(button), "clicked", */
/* 		   G_CALLBACK(callback_open_db), pui); */
/*   gtk_widget_show(button); */

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(svbox), hbox, TRUE, TRUE, 0);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_IMPORT]));
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(button), "clicked",
		   G_CALLBACK(callback_import), pui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EXPORT]));
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
  g_signal_connect(GTK_OBJECT(button), "clicked",
		   G_CALLBACK(callback_export), pui);
  gtk_widget_show(button);

  gtk_widget_show(hbox);

  /* Driving log */
  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_DRIVINGLOG]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox=gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);
  gtk_widget_show(uvbox);

  pui->autonetcheck = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (pui->autonetcheck), gettext(labels[SETTINGS_AUTOMATIC_NETWORK]));
  hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->autonetcheck), autostartnetwork);
/*   g_signal_connect (GTK_OBJECT (pui->autonetcheck), "toggled", */
/* 		    G_CALLBACK(network_check_cb), pui); */
  gtk_box_pack_start (GTK_BOX(uvbox), pui->autonetcheck, TRUE, TRUE, 0);
  gtk_widget_show(pui->autonetcheck);

  /* GPS  */
  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_GPS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox=gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);
  gtk_widget_show(uvbox);

  pui->autogpscheck = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (pui->autogpscheck), gettext(labels[SETTINGS_AUTOMATIC_GPS]));
  hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->autogpscheck), autostartGPS);
/*   g_signal_connect (GTK_OBJECT (pui->autogpscheck), "toggled", */
/* 		    G_CALLBACK(gps_check_cb), pui); */
  gtk_box_pack_start (GTK_BOX(uvbox), pui->autogpscheck, TRUE, TRUE, 0);
  gtk_widget_show(pui->autogpscheck);

  /* Locations */
  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_LOCATIONS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox=gtk_vbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);
  gtk_widget_show(uvbox);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(uvbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(gettext(labels[SETTINGS_LOCATIONS_RADIUS]));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  pui->entrylocationsradius=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrylocationsradius), 10);
  buf=g_string_new(NULL);
  g_string_printf(buf, "%.0f", locationaliasradius);
  gtk_entry_set_text(GTK_ENTRY(pui->entrylocationsradius),buf->str);
  g_string_free(buf, TRUE);
  gtk_box_pack_start (GTK_BOX(hbox), pui->entrylocationsradius, TRUE, TRUE, 0);
  gtk_widget_show(pui->entrylocationsradius);

  gtk_widget_show(hbox);

  /* Done, show the whole thing */

/*   gtk_box_pack_start (GTK_BOX(scrollwin), svbox, TRUE, TRUE, 0); */
  gtk_widget_show(svbox);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),svbox);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_SETTINGS,
				 pui->app->osso);
#endif

#else // MAEMO_VERSION_MAJOR < 5
  /** Driver and car notebook page **/

  table = gtk_table_new(6, 4, FALSE);
/*     gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5); */

  label=gtk_label_new(gettext(labels[SETTINGS_DATABASE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  label=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(label), db_getfilename());
  gtk_editable_set_editable(GTK_EDITABLE(label), FALSE);
  gtk_table_attach(GTK_TABLE(table), label, 1, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  row++;
  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->drivercombo = create_driver_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->drivercombo);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  button = gtk_button_new_with_label(gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_driver_cb), &ui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  /* Second row */
  label=gtk_label_new(gettext(labels[SETTINGS_CAR]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->carcombo = create_car_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);

  gtk_widget_show(pui->carcombo);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_car_cb), pui);
  gtk_widget_show(button);

  row++;
  button = gtk_button_new_with_label(gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_car_cb), pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_car_cb), pui);
  gtk_widget_show(button);

  row++;
  /* Third row */
  label=gtk_label_new(gettext(labels[SETTINGS_DATEFORMAT]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->datecombo = create_dateformat_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->datecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->datecombo);

  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_DATABASE]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /** Reminders notebook page **/

  table = gtk_table_new(2, 3, FALSE);
  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_WARN]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrywarnkm=gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrywarnkm), MAXWARNLEN);
  update_warn_entry(pui->entrywarnkm, warnkm);
  gtk_table_attach(GTK_TABLE(table), pui->entrywarnkm, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrywarnkm);

  pui->labelentrywarn=gtk_label_new(NULL);
  update_warn_label(pui->labelentrywarn);
  gtk_table_attach(GTK_TABLE(table), pui->labelentrywarn, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->labelentrywarn);

  row++;
  /* Second row */
  label=gtk_label_new(gettext(labels[SETTINGS_REMIND]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->warntimecombo=create_warntime_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->warntimecombo, 1, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->warntimecombo);

/*   label=gtk_label_new(_("before")); */
/*   gtk_table_attach(GTK_TABLE(table), label, 2, 3, row, row+1,  */
/* 		   GTK_EXPAND|GTK_FILL, */
/* 		   0, 0, 5); */
/*   gtk_widget_show(label); */

  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_REMINDERS]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /** Units notebook page **/

  unitscrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(unitscrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  table = gtk_table_new(7, 3, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(unitscrollwin),table);

  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_CURRENCY]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrycurrency = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycurrency), 30);
  gtk_entry_set_text(GTK_ENTRY(pui->entrycurrency),curunit.currency->str);
  gtk_table_attach(GTK_TABLE(table), pui->entrycurrency, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycurrency);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_UNITSYSTEM]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  row=row+2;
  label=gtk_label_new(gettext(labels[SETTINGS_LENGTHUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->lengthcombo = create_unit_combo(LENGTHCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->lengthcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
  gtk_widget_show(pui->lengthcombo);

  row++;
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->volumecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->volumecombo, FALSE);
  gtk_widget_show(pui->volumecombo);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_CONSUMEUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->consumecombo = create_unit_combo(CONSUMECOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->consumecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->consumecombo, FALSE);
  gtk_widget_show(pui->consumecombo);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_MASSUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->masscombo = create_unit_combo(MASSCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->masscombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->masscombo, FALSE);
  gtk_widget_show(pui->masscombo);

  row=row-5;
  /* Must be the last combo since the toggle callback uses the other combos */
  pui->unitcombo = create_unit_combo(UNITCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->unitcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->unitcombo);

  row++;
  /* Must be the last since the toggle callback uses the combos */
  unitcheck = gtk_check_button_new_with_label(gettext(labels[SETTINGS_INDIVIDUAL]));
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), unitcheck, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), unitcheck, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(unitcheck);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (unitcheck), curunit.individualunit);

  gtk_widget_show(table);
  gtk_widget_show(unitscrollwin);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_UNITS]));
  /*     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label); */
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), unitscrollwin, label);

  /** Reports notebook page **/

  table = gtk_table_new(1, 2, FALSE);
  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_PAPER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->papercombo = create_paper_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->papercombo, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->papercombo);

  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_REPORT]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /* Notebooks done */


#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_SETTINGS,
				 pui->app->osso);
#endif
#endif

/*   g_signal_connect_swapped (dialog, "response", */
/* 			    G_CALLBACK (settings_response), */
/* 			    dialog); */
  g_signal_connect (dialog, "response",
		    G_CALLBACK (settings_response),
		    pui);

  gtk_widget_show(dialog);
}
