/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <gconf/gconf-client.h>

#include "uiconfig.h"
#include "ui.h"
#include "uisettings.h"
#include "fuelpad.h"
#include "util.h"

/*******************************************************************
 *
 * Private defines
 *
 *******************************************************************/

#define GCONF_KEY_PREFIX "/apps/fuelpad"
#define GCONF_KEY_CURRENT_DRIVER GCONF_KEY_PREFIX"/current_driver"
#define GCONF_KEY_CURRENT_CAR GCONF_KEY_PREFIX"/current_car"
#define GCONF_KEY_CURRENT_UNIT GCONF_KEY_PREFIX"/current_unit"
#define GCONF_KEY_INDIVIDUAL_UNIT GCONF_KEY_PREFIX"/individual_unit"
#define GCONF_KEY_CURRENT_LENGTHUNIT GCONF_KEY_PREFIX"/current_lengthunit"
#define GCONF_KEY_CURRENT_VOLUMEUNIT GCONF_KEY_PREFIX"/current_volumeunit"
#define GCONF_KEY_CURRENT_CONSUMEUNIT GCONF_KEY_PREFIX"/current_consumeunit"
#define GCONF_KEY_CURRENT_MASSUNIT GCONF_KEY_PREFIX"/current_massunit"
#define GCONF_KEY_DATE_FORMAT GCONF_KEY_PREFIX"/date_format"
#define GCONF_KEY_CURRENCY GCONF_KEY_PREFIX"/currency"
#define GCONF_KEY_DATABASE GCONF_KEY_PREFIX"/database"
#define GCONF_KEY_WARNING_KM GCONF_KEY_PREFIX"/warningkm"
#define GCONF_KEY_WARNING_DAYS GCONF_KEY_PREFIX"/warningdays"
#define GCONF_KEY_REPORT_PAPER GCONF_KEY_PREFIX"/reportpaper"
#define GCONF_KEY_SELECT_COLUMNS GCONF_KEY_PREFIX"/mainviewcolumns"
#define GCONF_KEY_MAINVIEW_FONTSIZE GCONF_KEY_PREFIX"/mainviewfontsize"
#define GCONF_KEY_MAINTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/maintoolbar_notvisible"
#define GCONF_KEY_SECTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/secondarytoolbar_notvisible"
#define GCONF_MAINTABLE_SORTED GCONF_KEY_PREFIX"/maintablesorted"
#define GCONF_MAINTABLE_SORTCOLUMN GCONF_KEY_PREFIX"/maintablesortcolumn"
#define GCONF_MAINTABLE_SORTORDER GCONF_KEY_PREFIX"/maintablesortorder"
#define GCONF_MAINTABLE_UISTYLE GCONF_KEY_PREFIX"/uistyle"
#define GCONF_KEY_AUTOSTARTGPS GCONF_KEY_PREFIX"/autostart_gps"
#define GCONF_KEY_AUTOSTARTNET GCONF_KEY_PREFIX"/autostart_network"
#define GCONF_KEY_ALIASRADIUS GCONF_KEY_PREFIX"/aliasradius"

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private function prototypes
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

/**
 * \fn void config_save(void)
 * \brief Save all configuration data to gconf.
 *
 * Saves the current car, current driver and current units to gconf.
 *
 */
void config_save(void)
{
  GConfClient *gconf_client = gconf_client_get_default();

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings were not saved."));
      return;
    }
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CAR,
		       currentcar, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_DRIVER, 
		       currentdriver, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_UNIT,
		       curunit.mainunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_LENGTHUNIT,
		       curunit.lengthunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_VOLUMEUNIT,
		       curunit.volumeunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CONSUMEUNIT,
		       curunit.consumeunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_MASSUNIT,
		       curunit.massunit, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_INDIVIDUAL_UNIT,
			curunit.individualunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_DATE_FORMAT,
		       curunit.dateformat, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_CURRENCY,
			   curunit.currency->str, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_DATABASE,
			  db_getfilename(), NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_KM,
		       warnkm, NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_DAYS,
		       warndays, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_REPORT_PAPER,
		       reportpaper, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_SELECT_COLUMNS,
		       dispcol, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_MAINVIEW_FONTSIZE,
		       (gint)fontsize, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
			!main_toolbar_visible, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
			!secondary_toolbar_visible, NULL);
  gconf_client_set_bool(gconf_client,GCONF_MAINTABLE_SORTED,
			maintablesorted, NULL);
  gconf_client_set_int(gconf_client,GCONF_MAINTABLE_SORTCOLUMN,
		       (gint)maintablesortcol, NULL);
  gconf_client_set_int(gconf_client,GCONF_MAINTABLE_SORTORDER,
		       (gint)maintablesortorder, NULL);
  gconf_client_set_int(gconf_client,GCONF_MAINTABLE_UISTYLE,
		       (gint)uistyle, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_AUTOSTARTGPS,
			autostartGPS, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_AUTOSTARTNET,
			autostartnetwork, NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_ALIASRADIUS,
			 locationaliasradius, NULL);

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);
}

/**
 * \fn void config_init(void)
 * \brief Initialize all configuration from gconf.
 */
void config_init(void)
{
  GConfClient *gconf_client = gconf_client_get_default();
  struct lconv *locdata;
  gchar *tmpcurrency;
  gchar *tmpdatabase;

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings not read."));
      return;
    }
  currentdriver = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_DRIVER,
				       NULL);
  currentcar = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_CAR,
				    NULL);
  curunit.mainunit = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_UNIT,
				     NULL);
  curunit.lengthunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_LENGTHUNIT, NULL);
  curunit.volumeunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_VOLUMEUNIT, NULL);
  curunit.consumeunit = gconf_client_get_int(gconf_client,
					    GCONF_KEY_CURRENT_CONSUMEUNIT, NULL);
  curunit.massunit = gconf_client_get_int(gconf_client,
					    GCONF_KEY_CURRENT_MASSUNIT, NULL);
  curunit.individualunit = gconf_client_get_bool(gconf_client,
					 GCONF_KEY_INDIVIDUAL_UNIT, NULL);

  curunit.dateformat = gconf_client_get_int(gconf_client,
					    GCONF_KEY_DATE_FORMAT, NULL);
  tmpcurrency = gconf_client_get_string(gconf_client,
				     GCONF_KEY_CURRENCY, NULL);
  tmpdatabase = gconf_client_get_string(gconf_client,
					GCONF_KEY_DATABASE, NULL);
  warnkm = gconf_client_get_float(gconf_client,
				  GCONF_KEY_WARNING_KM, NULL);
  warndays = gconf_client_get_float(gconf_client,
				    GCONF_KEY_WARNING_DAYS, NULL);
  reportpaper=gconf_client_get_int(gconf_client,
				   GCONF_KEY_REPORT_PAPER, NULL);
  dispcol=gconf_client_get_int(gconf_client,
				   GCONF_KEY_SELECT_COLUMNS, NULL);
  fontsize=gconf_client_get_int(gconf_client,
				   GCONF_KEY_MAINVIEW_FONTSIZE, NULL);
  main_toolbar_visible=!gconf_client_get_bool(gconf_client,
					      GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
					      NULL);
  secondary_toolbar_visible=!gconf_client_get_bool(gconf_client,
						   GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
						   NULL);
  maintablesorted = gconf_client_get_bool(gconf_client,
					  GCONF_MAINTABLE_SORTED,
					  NULL);
  maintablesortcol = gconf_client_get_int(gconf_client,
					  GCONF_MAINTABLE_SORTCOLUMN,
					  NULL);
  maintablesortorder = (GtkSortType)gconf_client_get_int(gconf_client,
							 GCONF_MAINTABLE_SORTORDER,
							 NULL);
  uistyle = (enum userinterfacestyle)gconf_client_get_int(gconf_client,
							  GCONF_MAINTABLE_UISTYLE,
							  NULL);
  autostartGPS = gconf_client_get_bool(gconf_client,
				       GCONF_KEY_AUTOSTARTGPS,
				       NULL);
  autostartnetwork = gconf_client_get_bool(gconf_client,
					   GCONF_KEY_AUTOSTARTNET,
					   NULL);
  locationaliasradius = gconf_client_get_float(gconf_client,
					       GCONF_KEY_ALIASRADIUS, NULL);

#if MAEMO_VERSION_MAJOR == 5
  main_toolbar_visible=FALSE;
  secondary_toolbar_visible=TRUE;
#endif

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);  

  if (tmpcurrency == NULL) {
    locdata=localeconv();
    curunit.currency=g_string_new(locdata->int_curr_symbol);
  }
  else
    curunit.currency=g_string_new(tmpcurrency);

  db_setfilename(tmpdatabase);

  /* Current car and driver are zero if this is the first time 
   * this program is run
   */
  if (currentcar==0) currentcar=1;
  if (currentdriver==0) currentdriver=1;
  if (curunit.mainunit>IMPERIAL || curunit.mainunit < SI) 
    curunit.mainunit=SI;
  if (curunit.lengthunit>IMPERIAL || curunit.lengthunit < SI) 
    curunit.lengthunit=SI;
  if (curunit.volumeunit>IMPERIAL || curunit.volumeunit < SI) 
    curunit.volumeunit=SI;
  if (curunit.consumeunit>IMPERIAL || curunit.consumeunit < SI) 
    curunit.consumeunit=SI;
  if (curunit.massunit>IMPERIAL || curunit.massunit < SI) 
    curunit.massunit=SI;
  if (curunit.dateformat<0 || curunit.dateformat>=sizeofdatefmtstr())
    curunit.dateformat=0;
  if (warnkm==0.0)
    warnkm=1000;
  if (warndays==0.0)
    warndays=7.0;
  if (locationaliasradius == 0.0)
    locationaliasradius = 50.0;

  /* Limit reportpaper between 0 and 1 */ 
  if (reportpaper<0) reportpaper=0;
  else if (reportpaper>1) reportpaper=1;

  /* Reset shown columns to default value */
  if (dispcol == 0)
    dispcol=DISPCOLDEFAULT;

  /* Reset fontsize to MEDIUM*/
  if (fontsize<XSMALL || fontsize>LARGE)
    fontsize=MEDIUM;

  /* Limit uistyle to allowable range */
  if (uistyle<UIFANCY) uistyle=UIFANCY;
  if (uistyle>UICLASSIC) uistyle=UICLASSIC;
}
