/**********************************************************************
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "graphics.h"
#include "mapview.h"
#include "maptools.h"

#define MAX_TOOL_BAR_BUTTONS 64

#define TOOL_BAR_BUTTON_SIZE 26

struct _MapTools
{
  MapToolBar *map_tool_bar;
};

struct _MapToolBar
{
  MapTools *map_tools;
  GdkPixmap *pixmap;
  gint x;
  gint y;
  gint width;
  gint height;
  gint button_count;
  gboolean needs_redraw;
  MapToolBarButton *buttons[MAX_TOOL_BAR_BUTTONS];
};

struct _MapToolBarButton
{
  MapToolBar *map_tool_bar;
  MapToolBarButtonCallback callback;
  gpointer data;
  GdkPixmap *pixmap_normal;
  GdkPixmap *pixmap_highlighted;
  GdkPixmap *pixmap_disabled;
  guint disabled : 1;
  guint highlighted : 1;
};

static void map_tools_dirty_rect(int x, int y, int w, int h);

static void map_tool_bar_draw(MapToolBar *tb,
			      GdkDrawable *drawable,
			      GdkGC *gc);

static gboolean map_tool_bar_down(MapToolBar *tb,
				  GdkEventButton *ev,
				  gpointer data);

static gboolean map_tool_bar_release(MapToolBar *tb,
				     GdkEventButton *ev,
				     gpointer data);

static gboolean map_tool_bar_button_down(MapToolBarButton *bb,
					 GdkEventButton *ev,
					 gpointer data);

static gboolean map_tool_bar_button_release(MapToolBarButton *bb,
					    GdkEventButton *ev,
					    gpointer data);

static gboolean map_tool_bar_clip_out(MapToolBar *tb,
				      GdkDrawable *drawable,
				      GdkGC *gc,
				      gint x,
				      gint y,
				      gint width,
				      gint height);

static GdkPixmap *map_tool_bar_button_get_pixmap(MapToolBarButton *bb);

MapTools *map_tools_new(void)
{
  MapTools *mt = g_malloc0(sizeof(MapTools));
  memset(mt,0,sizeof(MapTools));
  return mt;
}

void map_tools_set_tool_bar(MapTools *mt, MapToolBar *tb)
{
  if (!mt)
    return;
  mt->map_tool_bar = tb;
  if (tb)
    tb->map_tools = mt;
}

/* Prevent the map tools being overwritten */
/* Return false if there is no overlap anyway */
gboolean map_tools_clip_out(MapTools *mt,
			    GdkDrawable *drawable,
			    GdkGC *gc,
			    gint x,
			    gint y,
			    gint width,
			    gint height)
{
  gboolean clipped = FALSE;

  if (map_tool_bar_clip_out(mt->map_tool_bar, drawable, gc, x, y, width, height))
    clipped = TRUE;

  return clipped;
}

/* Remove the clipping from map_tools_clip_out and draw all the tools */
void map_tools_unclip_and_draw(MapTools *mt,
			       GdkDrawable *drawable,
			       GdkGC *gc,
			       gint x,
			       gint y,
			       gint width,
			       gint height)
{
  if (!mt)
    return;
  map_tool_bar_draw(mt->map_tool_bar, drawable, gc);
}

gboolean map_tools_button_down(MapTools *mt, GdkEventButton *ev, gpointer data)
{
  if (!mt)
    return FALSE;
  if (mt->map_tool_bar)
    if (map_tool_bar_down(mt->map_tool_bar, ev, data))
      return TRUE;
  return FALSE;
}

gboolean map_tools_button_release(MapTools *mt,
				  GdkEventButton *ev,
				  gpointer data)
{
  if (!mt)
    return FALSE;
  if (mt->map_tool_bar)
    if (map_tool_bar_release(mt->map_tool_bar, ev, data))
      return TRUE;
  return FALSE;
}

static void map_tools_dirty_rect(int x, int y, int w, int h)
{
  dirty_rect(x, y, w, h);
}

MapToolBar *map_tool_bar_new(void)
{
  MapToolBar *tb = g_malloc0(sizeof(MapToolBar));
  return tb;
}

void map_tool_bar_add_button(MapToolBar *tb, MapToolBarButton *bb)
{
  if (!tb || !bb)
    return;
  if (tb->button_count >= MAX_TOOL_BAR_BUTTONS)
    return;
  tb->buttons[(tb->button_count)++] = bb;
  bb->map_tool_bar = tb;
}

static void map_tool_bar_update_pixmap(MapToolBar *tb, GdkDrawable *drawable)
{
  gint w, h;

  if (!tb)
    return;
  if (!tb->pixmap) {
    tb->width = tb->button_count * TOOL_BAR_BUTTON_SIZE;
    tb->height = TOOL_BAR_BUTTON_SIZE;
    tb->pixmap = gdk_pixmap_new(drawable, tb->width, tb->height, -1);
    tb->needs_redraw = TRUE;
  }
  gdk_drawable_get_size(drawable, &w, &h);
  tb->x = w - tb->width;
  tb->y = 0;
  if (tb->needs_redraw) {
    GdkGC *gc = gdk_gc_new(GDK_DRAWABLE(tb->pixmap));
    GdkColormap *cm = gdk_drawable_get_colormap(GDK_DRAWABLE(tb->pixmap));
    GdkColor c;

    gint i, pos;
    pos = 0;

    c.red = 0;
    c.green = 0;
    c.blue = 0;

    gdk_colormap_alloc_color(cm,&c,FALSE,TRUE);
    gdk_gc_set_foreground(gc, &c);
    gdk_draw_rectangle(GDK_DRAWABLE(tb->pixmap), gc, TRUE, 0, 0, tb->width,
				    tb->height);

    for (i = 0; i < tb->button_count; ++i) {
      MapToolBarButton *bb = tb->buttons[tb->button_count - i - 1];
      if (bb) {
	GdkPixmap *pm = map_tool_bar_button_get_pixmap(bb);
	if (pm) {
	  gdk_draw_drawable(GDK_DRAWABLE(tb->pixmap), gc, pm, 0, 0, pos, 0,
			    TOOL_BAR_BUTTON_SIZE, TOOL_BAR_BUTTON_SIZE);
	}
      }
      pos += TOOL_BAR_BUTTON_SIZE;
    }

    tb->needs_redraw = FALSE;
  }
}

static void map_tool_bar_draw(MapToolBar *tb,
			      GdkDrawable *drawable,
			      GdkGC *gc)
{
  if (!tb || !drawable)
    return;
  map_tool_bar_update_pixmap(tb, drawable);
  gdk_draw_drawable(drawable, gc, tb->pixmap, 0, 0, tb->x, tb->y,
		    tb->width, tb->height);
}

static gboolean map_tool_bar_clip_out(MapToolBar *tb,
				      GdkDrawable *drawable,
				      GdkGC *gc,
				      gint x,
				      gint y,
				      gint width,
				      gint height)
{
  map_tool_bar_update_pixmap(tb, drawable);
  if (x < tb->x + tb->width && x + width >= x &&
      y < tb->y + tb->height && y + height >= tb->y)
    return TRUE;
  return FALSE;
}

static gint map_tool_bar_get_button_no(MapToolBar *tb,
				       GdkEventButton *ev,
				       gpointer data)
{
  gint pos;

  if (!tb || !ev)
    return -1;
  if (ev->button != 1)
    return -1;
  if (ev->x < tb->x || ev->x >= tb->x + tb->width)
    return -1;
  if (ev->y < tb->y || ev->y >= tb->y + tb->height)
    return -1;

  pos = (ev->x - tb->x) / TOOL_BAR_BUTTON_SIZE;
  return tb->button_count - pos - 1;
}

static gboolean map_tool_bar_down(MapToolBar *tb,
				  GdkEventButton *ev,
				  gpointer data)
{
  gint bn = map_tool_bar_get_button_no(tb, ev, data);

  if (bn < 0)
    return FALSE;

  if (bn < tb->button_count)
    map_tool_bar_button_down(tb->buttons[bn], ev, data);

  return TRUE;
}

static gboolean map_tool_bar_release(MapToolBar *tb,
				     GdkEventButton *ev,
				     gpointer data)
{
  gint bn = map_tool_bar_get_button_no(tb, ev, data);

  if (bn < 0)
    return FALSE;

  if (bn < tb->button_count)
    map_tool_bar_button_release(tb->buttons[bn], ev, data);

  return TRUE;
}

static void map_tool_bar_needs_redraw(MapToolBar *tb, MapToolBarButton *bb)
{
  if (!tb)
    return;
  tb->needs_redraw = TRUE;
  map_tools_dirty_rect(tb->x, tb->y, tb->width, tb->height);
}

static GdkPixmap *get_tool_bar_button_pixmap(const gchar *file_stem,
					     const gchar *file_suffix)
{
  GdkPixmap *pm = NULL;
  GdkBitmap *bm = NULL;
  GdkPixbuf *pb;
  gchar *fn;

  fn = get_gfx_filename("toolbarbuttons", file_stem, file_suffix);
  if (!fn)
    return NULL;

  pb = gdk_pixbuf_new_from_file(fn, NULL);

  if (pb) {
    /* Workaround for MAEMO colormap problem */
    gdk_pixbuf_render_pixmap_and_mask_for_colormap(pb,
						   gdk_colormap_get_system(),
						   &pm,
						   &bm,
						   1);
    g_object_unref(pb);
  }

  if (bm)
    g_object_unref(bm);

  g_free(fn);

  return pm;
}

MapToolBarButton *map_tool_bar_button_new(const gchar *file_stem,
					  MapToolBarButtonCallback cb,
					  gpointer data)
{
  MapToolBarButton *bb;

  bb = g_malloc0(sizeof(MapToolBarButton));
  bb->callback = cb;
  bb->data = data;
  bb->pixmap_normal = get_tool_bar_button_pixmap(file_stem,"");
  bb->pixmap_highlighted = get_tool_bar_button_pixmap(file_stem,"-highlighted");
  bb->pixmap_disabled = get_tool_bar_button_pixmap(file_stem,"-disabled");

  return bb;
}

void map_tool_bar_button_enable(MapToolBarButton *bb)
{
  if (!bb)
    return;
  if (bb->disabled) {
    bb->disabled = FALSE;
    map_tool_bar_needs_redraw(bb->map_tool_bar, bb);
  }
}

void map_tool_bar_button_disable(MapToolBarButton *bb)
{
  if (!bb)
    return;
  if (!bb->disabled) {
    bb->disabled = TRUE;
    map_tool_bar_needs_redraw(bb->map_tool_bar, bb);
  }
}

void map_tool_bar_button_highlight_on(MapToolBarButton *bb)
{
  if (!bb)
    return;
  if (!bb->highlighted) {
    bb->highlighted = TRUE;
    map_tool_bar_needs_redraw(bb->map_tool_bar, bb);
  }
}

void map_tool_bar_button_highlight_off(MapToolBarButton *bb)
{
  if (!bb)
    return;
  if (bb->highlighted) {
    bb->highlighted = FALSE;
    map_tool_bar_needs_redraw(bb->map_tool_bar, bb);
  }
}

static gboolean map_tool_bar_button_down(MapToolBarButton *bb,
					 GdkEventButton *ev,
					 gpointer data)
{
  if (!bb || !bb->callback)
    return FALSE;
  bb->callback(bb, bb->data);
  return TRUE;
}

static gboolean map_tool_bar_button_release(MapToolBarButton *bb,
					    GdkEventButton *ev,
					    gpointer data)
{
  return FALSE;
}

static GdkPixmap *map_tool_bar_button_get_pixmap(MapToolBarButton *bb)
{
  if (!bb)
    return NULL;
  if (bb->disabled && bb->pixmap_disabled)
    return bb->pixmap_disabled;
  if (bb->highlighted && bb->pixmap_highlighted)
    return bb->pixmap_highlighted;
  return bb->pixmap_normal;
}
