/* This file is part of "modern computer flatboat", a pdf viewer.
 * Copyright (C) 2012 Frank Fuhlbrück 
 * License: GPLv3 (or any later version, at your option)
 * See the file "LICENSE".
 */

#include "qpdfimage.h"
#include <QPainter>
#include <QtGui/QFileDialog>
#include <QPaintEngine>
#include <QDebug>
#include <QSettings>



typedef unsigned char t_rgba[4];
#define swap(x,y) { x = x + y; y = x - y; x = x - y; }
void rgba2bgra(unsigned char *dat, int width, int height) {
	t_rgba *data = (t_rgba*) dat;
	for (int i = 0; i < width*height; i++) 
		swap(data[i][0],data[i][2])
}
#undef swap
void invertSamples(unsigned char *dat, int width, int height) {
	t_rgba *data = (t_rgba*) dat;
	for (int i = 0; i < width*height; i++) {
		data[i][0] = 255 - data[i][0];
		data[i][1] = 255 - data[i][1];
		data[i][2] = 255 - data[i][2];
	}       
}

static QRect fz_rectToQRect(fz_rect rect){
	fz_bbox b = fz_bbox_covering_rect(rect);
	return QRect(b.x0,b.y0,b.x1-b.x0,b.y1-b.y0);
}


static QString textChar_p2string(fz_text_char *text,int len,fz_rect** rectsOfChars=NULL){
	//MuPDF's internal format seems to be ucs4, so double conversion(->utf8->utf16) is unnecessary:
	//     int reallen = 0;
	//     char utf8[len*4];
	//     char chars[4];
	fz_rect *rects = new fz_rect[len];
	unsigned int ucs4[len];
	for(int i=0;i<len;i++){
		//         int lchars = fz_runetochar(chars, text[i].c);
		//         for(int j = 0; j < lchars; j++)
		//             utf8[reallen+j] = chars[j];
		ucs4[i] = text[i].c;
		rects[i] = text[i].bbox;
		//         reallen += lchars;
	}
	if(rectsOfChars)
		*rectsOfChars = rects;
	return QString::fromUcs4(ucs4,len);
}

static QString page2string(fz_text_page *page){
	fz_text_block *block;
	fz_text_line *line;
	fz_text_span *span;
	QString str("");
	for (block = page->blocks; block < page->blocks + page->len; block++){
		for (line = block->lines; line < block->lines + block->len; line++){
			for (span = line->spans; span < line->spans + line->len; span++)
				str += textChar_p2string(span->text,span->len) + "\n";// pseudo-newline
		}
	}
	return str;
}

static QList<QRect> searchInPage(fz_text_page *page,QString search,bool caseSensitive){
	fz_text_block *block;
	fz_text_line *line;
	fz_text_span *span;
	fz_rect* rects;
	QList<QRect> highlight;
	QString str;
	int fpos;
	int lpos;
	QRect uni;
	if(search == "")
		return highlight;
	for (block = page->blocks; block < page->blocks + page->len; block++){
		for (line = block->lines; line < block->lines + block->len; line++){
			for (span = line->spans; span < line->spans + line->len; span++){
				str = textChar_p2string(span->text,span->len,&rects);
				fpos = -1;
				while((fpos = str.indexOf(search,fpos+1,(Qt::CaseSensitivity)caseSensitive)) > -1){
					lpos = fpos + search.length();
					uni = fz_rectToQRect(rects[fpos]);
					for(int i=fpos+1;i<lpos;i++)
						uni |= fz_rectToQRect(rects[i]);
					highlight += uni;
				}
			}
		}
	}
	return highlight;	
}

static inline int commonWidth(QRect a,QRect b){
	int x0 = std::max(a.left(),b.left());
	int x1 = std::min(a.right(),b.right())+1;//+1: see Qt doc (right())
	return std::max(x1 - x0,0);	
}

static inline uint qHash(const QRect & r){
	return qHash(QString("%1,%2,%3,%4").arg(r.x()).arg(r.y()).arg(r.width()).arg(r.height()));
}

static inline bool operator< ( const QRect & r1, const QRect & r2 ){
	return r1.left() < r2.left();
}

static QList<QRect> textAreas(fz_text_page *page,int zoom){
	fz_text_block *block;
	QList<QRect> textAreas;
	QRect uni;
	QRect nextR;
	for (block = page->blocks; block < page->blocks + page->len; block++){
		nextR = fz_rectToQRect(block->bbox);
		if(commonWidth(uni,nextR) < 0.3*std::min(uni.width(),nextR.width())){//new column
			textAreas += uni;
			uni = nextR;
		} else //still the same column
			uni |= nextR;
	}
	if(uni.isValid())
		textAreas += uni;
	
	for(int i=0;i<textAreas.length();i++){
		for(int j=0;j<textAreas.length();j++){
			if(i!=j && commonWidth(textAreas[i],textAreas[j]) > 0.8*std::min(textAreas[i].width(),textAreas[j].width())){
				textAreas[j] |= textAreas[i];
			}
		}
	}
	
	//page numbers:
	for(int i=0;i<textAreas.length();)
		if(textAreas[i].width() < 30*zoom/100)
			textAreas.removeAt(i);
		else 
			i++;
	
	if(textAreas.isEmpty())
		textAreas += fz_rectToQRect(page->mediabox);
	textAreas = textAreas.toSet().toList();
	qSort(textAreas);
	return textAreas;
}


QPdfImage::QPdfImage(QDeclarativeItem *parent) :
QDeclarativeItem(parent) {
	setFlag (QGraphicsItem::ItemHasNoContents,false);
	QSettings stg;
	QString lastf = stg.value("lastFile","").toString();
	int lastp = stg.value("lastPage",1).toInt();
	int lastz = stg.value("lastZoom",100).toInt();
	context = NULL;
	document = NULL;
	textMode_prv = false;
	pagenum_prv = 1;
	zoom_prv = 100;
	totalpages_prv = 0;
	highlightPhrase_prv = "";
	caseSensitive_prv = false;
	hitOnPage = -1;
	
	textDocument = QList<QString>();
	hitCount = QList<int>();
	
	image = new QImage(800, 800, QImage::Format_RGB32);
	QRgb value;
	value = qRgb(189, 149, 39); // 0xffbd9527
	image->fill(value);
	setSize(image->size());
	loadFile(lastf);
	gotoPage(lastp);
	setZoom(lastz);
}

bool QPdfImage::loadFile(QString filename){
	if(!QFile::exists(filename))
		return false;
	filepath_prv = filename;
	QSettings().setValue("lastFile",filename);
	if(image){
		delete image;
		image = NULL;
	}
	if(pixmap){
		fz_drop_pixmap(context, pixmap);
		pixmap = NULL;
	}
	if(document){
		fz_close_document(document);
		document = NULL;
	}
	if(!context)
		context = fz_new_context(NULL, NULL, FZ_STORE_UNLIMITED);
	textDocument.clear();
	
	document = fz_open_document(context, filename.toLocal8Bit().data());
	totalpages_prv = fz_count_pages(document);
	return true;
}

bool QPdfImage::gotoPage(int pagenum){
	if(!document || pagenum > totalpages_prv || pagenum < 1)
		return false;
	QSettings().setValue("lastPage",pagenum);
	//save the pointer to the old pixmap
	fz_pixmap* oldpm = pixmap;
	
	if(pagenum_prv != pagenum){
		pagenum_prv = pagenum;
		emit pagenumChanged(pagenum_prv);
	}
	fz_page *page = fz_load_page(document, pagenum_prv - 1);
	
	//render page to pixmap
	fz_matrix transform = fz_scale(zoom_prv/100.0,zoom_prv/100.0);
	transform = fz_concat(transform, fz_rotate(0));
	fz_rect rect = fz_bound_page(document, page);//transformations are calculated using float
	rect = fz_transform_rect(transform, rect);
	fz_bbox bbox = fz_round_rect(rect); /*bounding box is int,
	fz_round_rect may be smaller than fz_bbox_covering_rect */
	pixmap = fz_new_pixmap_with_bbox(context, fz_device_bgr, bbox);//bgr matches Qt's rgb!
	fz_clear_pixmap_with_value(context, pixmap, 255);//draw on white background
	fz_device *device = fz_new_draw_device(context, pixmap);
	fz_run_page(document, page, device, transform, NULL);
	fz_free_device(device);
	
	//extract text
	fz_text_sheet *text_sheet = fz_new_text_sheet(context);//stores styles
	text_page = fz_new_text_page(context, rect);//stores "plain" text
	fz_device *text_device = fz_new_text_device(context, text_sheet, text_page);
	fz_run_page(document, page, text_device, transform, NULL);
	fz_free_device(text_device);
	
	
	
	//prepare the raw image
	unsigned char *samples = fz_pixmap_samples(context, pixmap);
	int width = fz_pixmap_width(context, pixmap);
	int height = fz_pixmap_height(context, pixmap);
	
	if(image)
		delete image;
	image = new QImage(samples, width, height, QImage::Format_ARGB32);
	setSize(image->size());
	highlight.clear();
	highlight += searchInPage(text_page,highlightPhrase_prv,caseSensitive_prv);
	regions.clear();
	regions += textAreas(text_page,zoom_prv);
	
	update(boundingRect());
	
	//delete old pixmap
	fz_drop_pixmap(context, oldpm);
	//delete the page reference
	fz_free_page(document, page);
	return true;
}

void QPdfImage::paint(QPainter * painter, const QStyleOptionGraphicsItem *, QWidget *) {
	if(textMode_prv){
		if(textDocument.isEmpty())
			textDocument += document2stringList();
		painter->drawText(QRect(0,0,0.8*width(),3*height()),
			Qt::AlignLeft|Qt::TextDontClip|Qt::TextSingleLine|Qt::TextWordWrap,
			textDocument[pagenum_prv-1]);
		return;
	}
	if(image)
		painter->drawImage (QPoint(0,0),*image);
	painter->setPen(QColor("transparent"));
	painter->setBrush(QBrush(QColor ( 230, 230, 30, 120)));
	for(int i=0;i < highlight.length();i++)
		painter->drawRect(highlight[i]);
	painter->setBrush(QBrush(QColor ( 255, 0, 0, 80)));
	if(hitOnPage > -1 && hitOnPage < highlight.length() && lastSearchPhrase == highlightPhrase_prv)
		painter->drawRect(highlight[hitOnPage]);

// // 	debug the column finder code
// 	painter->setPen(QColor ( 255, 0, 0, 255));
// 	painter->setBrush(QBrush(QColor ("transparent")));
// 	for(int i=0;i < regions.length();i++){
// 		painter->drawRect(regions[i]);
// 		painter->drawText(regions[i].regions(),QString::number(i));
// 	}
}

QStringList QPdfImage::document2stringList(){
	QStringList textDocument;
	fz_page *page;
	fz_text_page *text_page;
	fz_text_sheet *text_sheet;
	fz_device *text_device;
	if(!document)
		return textDocument;
	for(int i=0;i<totalpages_prv;i++){
		page = fz_load_page(document,i);
		text_sheet = fz_new_text_sheet(context);//stores styles
		text_page = fz_new_text_page(context, fz_infinite_rect);//stores "plain" text
		text_device = fz_new_text_device(context, text_sheet, text_page);
		fz_run_page(document, page, text_device, fz_identity, NULL);
		textDocument << page2string(text_page);
		fz_free_device(text_device);
		fz_free_text_page(context,text_page);
		fz_free_text_sheet(context,text_sheet);
		fz_free_page(document,page);
	}
	return textDocument;
}

bool QPdfImage::searchNext(QString str){
	if(textDocument.isEmpty())
		textDocument += document2stringList(); 
	if(str != lastSearchPhrase || caseSensitive_prv != lastSearchCaseSensitive){
		hitCount.clear();
		hitOnPage = -1;
		for(int i=0;i<totalpages_prv;i++)
			hitCount.append(textDocument[i].count(str, (Qt::CaseSensitivity) caseSensitive_prv));
	}
	lastSearchCaseSensitive = caseSensitive_prv;
	lastSearchPhrase = str;
	if(hitOnPage < hitCount[pagenum_prv-1]-1){
		hitOnPage++;	
		update(boundingRect());
	} else {
		int nextPage = pagenum_prv == totalpages_prv ? 1 : pagenum_prv+1;
		while(nextPage != pagenum_prv && hitCount[nextPage-1] == 0)
			nextPage = nextPage == totalpages_prv ? 1 : nextPage+1;
		if(hitCount[nextPage-1] != 0)
			hitOnPage = 0;
		else
			hitOnPage = -1;
		if(nextPage != pagenum_prv)
			gotoPage(nextPage);
		else
			update(boundingRect());
	}
	//set the corresponding region if phrase is found
	if(hitOnPage >= 0){
		QRect hitRect = highlight[hitOnPage];
		int i = 0;
		for(;i < regions.length() && regions[i].intersects(hitRect);i++);
		if(i < regions.length())
			curRegionIdx = i;
		else
			curRegionIdx = 0;
	}
	return (bool) (hitOnPage+1);
}

bool QPdfImage::searchPrev(QString str){
	if(textDocument.isEmpty())
		textDocument += document2stringList(); 
	if(str != lastSearchPhrase || caseSensitive_prv != lastSearchCaseSensitive){
		hitCount.clear();
		hitOnPage = -1;
		for(int i=0;i<totalpages_prv;i++)
			hitCount.append(textDocument[i].count(str, (Qt::CaseSensitivity) caseSensitive_prv));
	}
	lastSearchCaseSensitive = caseSensitive_prv;
	lastSearchPhrase = str;
	if(hitOnPage > 0){
		hitOnPage--;	
		update(boundingRect());
	} else {
		int prevPage = pagenum_prv == 1 ? totalpages_prv : pagenum_prv-1;
		while(prevPage != pagenum_prv && hitCount[prevPage-1] == 0)
			prevPage = prevPage == 1 ? totalpages_prv : prevPage-1;
		if(hitCount[prevPage-1] != 0)
			hitOnPage = hitCount[prevPage-1]-1;//last hit
			else
				hitOnPage = -1;
			if(prevPage != pagenum_prv)
				gotoPage(prevPage);
			else
				update(boundingRect());
	}
	//set the corresponding region if phrase is found
	if(hitOnPage >= 0){
		QRect hitRect = highlight[hitOnPage];
		int i = 0;
		for(;i < regions.length() && regions[i].intersects(hitRect);i++);
		if(i < regions.length())
			curRegionIdx = i;
		else
			curRegionIdx = 0;
	}
	return (bool) (hitOnPage+1);
}

QRect QPdfImage::prevRegion(){
	if(curRegionIdx > 0)
		return regions[--curRegionIdx];
	if(pagenum_prv > 1)
		gotoPage(pagenum_prv-1);//sets regions
	else
		gotoPage(totalpages_prv);//sets regions
	curRegionIdx = regions.length()-1;
	return regions[curRegionIdx];
}

QRect QPdfImage::curRegion(){
	return regions[curRegionIdx];
}

QRect QPdfImage::nextRegion(){
	if(curRegionIdx < regions.length()-1)
		return regions[++curRegionIdx];
	if(pagenum_prv < totalpages_prv)
		gotoPage(pagenum_prv+1);//sets regions
	else
		gotoPage(1);//sets regions
	curRegionIdx = 0;
	return regions[0];
}



int QPdfImage::totalPages(){
	return totalpages_prv;
}

#ifdef Q_OS_ANDROID
#define DEFPATH "/mnt/sdcard"
#else
#define DEFPATH QDir::homePath()
#endif
bool QPdfImage::chooseFile(){
	QFileInfo lf(QSettings().value("lastFile").toString());
	QString dir = lf.dir().exists() ? lf.dir().path() : DEFPATH;
	QString filename = QFileDialog::getOpenFileName(0, "Select PDF file", dir, "PDF (*.pdf)");
	return loadFile(filename) && gotoPage(1);
}

QString QPdfImage::filepath(){
	return filepath_prv;
}

bool QPdfImage::textMode(){
	return textMode_prv;
}
void QPdfImage::setTextMode(bool tm){
	textMode_prv = tm;
	update(boundingRect());
}


int QPdfImage::pagenum(){
	return pagenum_prv;
}

int QPdfImage::zoom(){
	return zoom_prv;
}

bool QPdfImage::setZoom(int percent){
	zoom_prv = percent;
	QSettings().setValue("lastZoom",percent);
	int ret = gotoPage(pagenum_prv);
	emit zoomChanged(zoom_prv);
	return ret;
}


QString QPdfImage::highlightPhrase(){
	return highlightPhrase_prv;
}
void QPdfImage::setHighlightPhrase(QString phrase){
	highlightPhrase_prv = phrase;
	highlight.clear();
	if(!text_page)
		return;
	highlight += searchInPage(text_page,highlightPhrase_prv,caseSensitive_prv);
	update(boundingRect());
}

bool QPdfImage::caseSensitive(){
	return caseSensitive_prv;
}
void QPdfImage::setCaseSensitive(bool cs){
	caseSensitive_prv = cs;
	setHighlightPhrase(highlightPhrase_prv);//update highlighting
}
