/*
 *  Flashlight applet (widget) for Maemo.
 *  Copyright (C) 2009, 2010 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>
#include <libhal.h>
#include <dbus/dbus.h>

#include <libosso.h>

#include "flashlight_applet.h"
#include "flashlight_lib.h"

#define MSG_FLASHLIGHT_ON _("On")
#define MSG_FLASHLIGHT_OFF _("Off")

#define ICON_FLASHLIGHT_ON "statusarea_flashlight_on"
#define ICON_FLASHLIGHT_OFF "statusarea_flashlight_off"

#define CAM_COVER_UDI "/org/freedesktop/Hal/devices/platform_cam_shutter"
#define CAM_COVER_STATE "button.state.value"

#define FLASHLIGHT_APPLET_SERVICE "org.maemo.flashlight_applet"
#define FLASHLIGHT_APPLET_OBJECT "/org/maemo/flashlight_applet"
#define FLASHLIGHT_APPLET_IFACE "org.maemo.flashlight_applet"

#define FLASHLIGHT_APPLET_METHOD_ENABLE "enable"
#define FLASHLIGHT_APPLET_METHOD_DISABLE "disable"

#define _CAMERAUI(str) dgettext("osso-camera-ui",str)

#define FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,   \
                            TYPE_FLASHLIGHT_STATUS_PLUGIN, FlashlightPluginPrivate))

struct _FlashlightPluginPrivate
{
	GtkWidget *button;
	guint status_timer;

	FlashlightContext_t *flashlight;
	DBusConnection *dbus_connection;
	LibHalContext *hal;
	osso_context_t *osso_context;
};

HD_DEFINE_PLUGIN_MODULE (FlashlightPlugin, flashlight_status_plugin, HD_TYPE_STATUS_MENU_ITEM)

static gboolean flashlight_status_plugin_status (gpointer data);
static void flashlight_status_plugin_finalize (GObject *object);

static void
flashlight_status_plugin_show_notification (FlashlightPlugin *plugin,
					    const gchar *text)
{
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);
	GtkWidget *banner;

	g_return_if_fail (priv);
	g_return_if_fail (priv->button);

	banner = hildon_banner_show_information (GTK_WIDGET (priv->button), NULL, text);
	hildon_banner_set_timeout (HILDON_BANNER (banner), 3000);
}

static void
flashlight_status_plugin_enable (FlashlightPlugin *plugin,
				 gboolean enable)
{
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);

	g_return_if_fail (priv);
	g_return_if_fail (priv->button);

	if (enable) {
		if (flashlight_open (priv->flashlight, "/dev/video0") < 0) {
			flashlight_status_plugin_show_notification (plugin,
				_("Unable to initialize flashlight.\nCamera in use by another application."));
			flashlight_close (priv->flashlight);
			return;
		}

		if (flashlight_set_intensity (priv->flashlight, 1) < 0) {
			flashlight_status_plugin_show_notification (plugin,
				_("Unable to turn on flashlight."));
			flashlight_close (priv->flashlight);
			return;
		}

		hildon_button_set_value (HILDON_BUTTON (priv->button), MSG_FLASHLIGHT_ON);
		hildon_button_set_image (HILDON_BUTTON (priv->button),
					 gtk_image_new_from_icon_name (ICON_FLASHLIGHT_ON, -1));

		/* check status of controller every 1s */
		priv->status_timer = g_timeout_add (1000, flashlight_status_plugin_status, plugin);
	} else {
		/* cancel status timer */
		if (priv->status_timer) {
			g_source_remove (priv->status_timer);
			priv->status_timer = 0;
		}

		/* set intensity to 0 */
		if (flashlight_set_intensity (priv->flashlight, 0) < 0) {
			flashlight_status_plugin_show_notification (plugin,
				_("Unable to turn off flashlight."));
			return;
		}

		if (flashlight_close (priv->flashlight) < 0) {
			return;
		}

		hildon_button_set_value (HILDON_BUTTON (priv->button), MSG_FLASHLIGHT_OFF);
		hildon_button_set_image (HILDON_BUTTON (priv->button),
					 gtk_image_new_from_icon_name (ICON_FLASHLIGHT_OFF, -1));
	}
}

static gint
flashlight_applet_dbus_request_handler (const gchar *interface,
					const gchar *method,
					GArray *arguments,
					gpointer data,
					osso_rpc_t *retval)
{
	FlashlightPlugin *plugin = data;
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);
	gboolean is_open = FALSE;

	g_return_val_if_fail (interface, OSSO_ERROR);
	g_return_val_if_fail (method, OSSO_ERROR);

	g_return_val_if_fail (priv, OSSO_ERROR);
	g_return_val_if_fail (priv->hal, OSSO_ERROR);
	g_return_val_if_fail (priv->button, OSSO_ERROR);

	if (strcmp (interface, FLASHLIGHT_APPLET_IFACE))
		return OSSO_ERROR;

	is_open = !libhal_device_get_property_bool (priv->hal, CAM_COVER_UDI, CAM_COVER_STATE, NULL);

	if (!strcmp (method, FLASHLIGHT_APPLET_METHOD_ENABLE)) {
		/* check if cover is open */
		if (is_open) {
			/* try to enable flashlight, only if it's disabled */
			if (!strcmp (hildon_button_get_value (HILDON_BUTTON (priv->button)), MSG_FLASHLIGHT_OFF)) {
				flashlight_status_plugin_enable (plugin, TRUE);
			}
		} else {
			flashlight_status_plugin_show_notification (plugin, _CAMERAUI ("camera_ia_open_lens_cover"));
		}
	} else if (!strcmp (method, FLASHLIGHT_APPLET_METHOD_DISABLE)) {
		if (is_open) {
			/* try to disable flashlight, only if it's enabled */
			if (!strcmp (hildon_button_get_value (HILDON_BUTTON (priv->button)), MSG_FLASHLIGHT_ON)) {
				flashlight_status_plugin_enable (plugin, FALSE);
			}
		}
	} else {
		return OSSO_ERROR;
	}

	return OSSO_OK;
}

static void
flashlight_status_plugin_on_hal_property_modified (LibHalContext *ctx,
						   const char *udi,
						   const char *key,
						   dbus_bool_t is_removed,
						   dbus_bool_t is_added)
{
	FlashlightPlugin *plugin = libhal_ctx_get_user_data (ctx);
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);
	gboolean is_open = FALSE;
	int intensity = 0;

	g_return_if_fail (priv);

	if (strcmp (udi, CAM_COVER_UDI) != 0)
		return;

	if (strcmp (key, CAM_COVER_STATE) != 0)
		return;

	is_open = !libhal_device_get_property_bool (ctx, udi, key, NULL);

	if (is_open) {
		/* show widget */
		gtk_widget_show_all (GTK_WIDGET (plugin));
	} else {
		/* turn off flashlight if flashlight is enabled */
		if (flashlight_get_intensity (priv->flashlight, &intensity) == 0) {
			if (intensity > 0) {
				flashlight_status_plugin_enable (plugin, FALSE);
			}
		}

		/* hide widget */
		gtk_widget_hide_all (GTK_WIDGET (plugin));
	}
}

static gboolean
flashlight_status_plugin_status (gpointer data)
{
	FlashlightPlugin *plugin = data;
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);
	int status = 0;

	if (flashlight_get_status (priv->flashlight, &status) < 0) {
		flashlight_status_plugin_show_notification (plugin,
							    _("Unable to read status from driver."));
		return FALSE;
	}

	/* ops, something is wrong */
	if (status) {
		/* turn off flashlight */
		flashlight_status_plugin_enable (plugin, FALSE);

		if (status & FLASHLIGHT_STATUS_SHORT_CIRCUT_FAULT) {
			flashlight_status_plugin_show_notification (plugin,
				_("Short-circut fault detected!\nTurning off flashlight."));
		} else if (status & FLASHLIGHT_STATUS_OVERTEMPERATURE_FAULT) {
			flashlight_status_plugin_show_notification (plugin,
				_("Overtemperature fault detected!\nTurning off flashlight."));
		} else if (status & FLASHLIGHT_STATUS_TIMEOUT_FAULT) {
			flashlight_status_plugin_show_notification (plugin,
				_("Timeout fault detected!\nTurning off flashlight."));
		} else if (status & FLASHLIGHT_STATUS_OVERVOLTAGE_FAULT) {
			flashlight_status_plugin_show_notification (plugin,
				_("Overvoltage fault detected!\nTurning off flashlight."));
		}
	}

	return TRUE;
}

static void
flashlight_status_plugin_on_clicked (HildonButton *button,
				     gpointer data)
{
	FlashlightPlugin *plugin = data;
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);

	g_return_if_fail (priv);

	if (!strcmp (hildon_button_get_value (button), MSG_FLASHLIGHT_ON)) {
		flashlight_status_plugin_enable (plugin, FALSE);
	} else {
		flashlight_status_plugin_enable (plugin, TRUE);
	}
}

static GtkWidget *
flashlight_status_plugin_ui (FlashlightPlugin *plugin)
{
	GtkWidget *button;

	g_return_val_if_fail (plugin, NULL);

	button = hildon_button_new (HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	gtk_button_set_alignment (GTK_BUTTON (button), 0.0, 0.5);
	hildon_button_set_title (HILDON_BUTTON (button), _("Flashlight"));
	hildon_button_set_value (HILDON_BUTTON (button), MSG_FLASHLIGHT_OFF);
	hildon_button_set_image (HILDON_BUTTON (button),
				 gtk_image_new_from_icon_name (ICON_FLASHLIGHT_OFF, -1));
	hildon_button_set_image_position (HILDON_BUTTON (button), GTK_POS_LEFT);

	g_signal_connect (button, "clicked",
					G_CALLBACK (flashlight_status_plugin_on_clicked), plugin);

	return button;
}

static void
flashlight_status_plugin_class_init (FlashlightPluginClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);

	object_class->finalize = flashlight_status_plugin_finalize;

	g_type_class_add_private (class, sizeof (FlashlightPluginPrivate));
}

static void
flashlight_status_plugin_class_finalize (FlashlightPluginClass *class)
{
}

static void
flashlight_status_plugin_init (FlashlightPlugin *plugin)
{
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);
	DBusError error;

	/* initialize osso */
	priv->osso_context = osso_initialize (PACKAGE, VERSION, TRUE, NULL);
	if (!priv->osso_context) {
		g_critical ("flashlight_status_plugin_init: Could not initialize OSSO\n");
		return;
	}

	if (osso_rpc_set_cb_f (priv->osso_context,
			       FLASHLIGHT_APPLET_SERVICE,
			       FLASHLIGHT_APPLET_OBJECT,
			       FLASHLIGHT_APPLET_IFACE,
			       flashlight_applet_dbus_request_handler, plugin) != OSSO_OK) {
		g_critical ("flashlight_status_plugin_init: Unable to register D-BUS request handler\n");
		return;
	}

	/* initialize dbus */
	dbus_error_init (&error);
	priv->dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	if (dbus_error_is_set (&error)) {
		g_critical ("flashlight_status_plugin_init: Could not get the system DBus connection, %s",
			    error.message);
		dbus_error_free (&error);
		return;
	}

	/* initialize flashlight */
	if (flashlight_init (&priv->flashlight) < 0) {
		g_critical ("flashlight_status_plugin_init: Unable to create Flashlight context\n");
		return;
	}

	/* initialize hal */
	priv->hal = libhal_ctx_new ();
	if (!priv->hal) {
		g_critical ("flashlight_status_plugin_init: Unable to create HAL context\n");
		return;
	}

	libhal_ctx_set_dbus_connection (priv->hal, priv->dbus_connection);
	libhal_ctx_set_user_data (priv->hal, plugin);
	libhal_ctx_set_device_property_modified (priv->hal,
						 flashlight_status_plugin_on_hal_property_modified);

	if (!libhal_ctx_init (priv->hal, &error)) {
		if (dbus_error_is_set (&error)) {
			g_critical ("Could not initialize the HAL context, %s",
				    error.message);
			dbus_error_free (&error);
		} else {
			g_critical ("Could not initialize the HAL context, "
				    "no error, is hald running?");
		}
		libhal_ctx_set_user_data (priv->hal, NULL);
		libhal_ctx_free (priv->hal);
		priv->hal = NULL;
		return;
	}

	libhal_device_add_property_watch (priv->hal, CAM_COVER_UDI, NULL);

	/* create plugin ui */
	priv->button = flashlight_status_plugin_ui (plugin);
	gtk_container_add (GTK_CONTAINER (plugin), priv->button);

	/* show widget if camera cover is open */
	if ( !libhal_device_get_property_bool (priv->hal, CAM_COVER_UDI, CAM_COVER_STATE, NULL))
		gtk_widget_show_all (GTK_WIDGET (plugin));

}

static void
flashlight_status_plugin_finalize (GObject *object)
{
	FlashlightPlugin *plugin = FLASHLIGHT_STATUS_PLUGIN (object);
	FlashlightPluginPrivate *priv = FLASHLIGHT_STATUS_PLUGIN_GET_PRIVATE (plugin);

	g_return_if_fail (priv);

	/* deinitialize hal */
	if (priv->hal) {
		libhal_device_remove_property_watch (priv->hal, CAM_COVER_UDI, NULL);
		libhal_ctx_set_user_data (priv->hal, NULL);
		libhal_ctx_shutdown (priv->hal, NULL);
		libhal_ctx_free (priv->hal);
	}
	priv->hal = NULL;

	/* unreference dbus connection */
	if (priv->dbus_connection) {
		dbus_connection_unref (priv->dbus_connection);
	}
	priv->dbus_connection = NULL;

	/* cancel status timer */
	if (priv->status_timer) {
		g_source_remove (priv->status_timer);
	}
	priv->status_timer = 0;

	/* deinitialize flashlight */
	if (priv->flashlight) {
		flashlight_deinit (priv->flashlight);
	}
	priv->flashlight = NULL;

	/* deinitialize osso */
	if (priv->osso_context) {
		osso_deinitialize (priv->osso_context);
	}
	priv->osso_context = NULL;

	G_OBJECT_CLASS (flashlight_status_plugin_parent_class)->finalize (object);
}
