# coding: utf-8

import gobject
import gtk
import pango
from threading import Thread
from threading import Lock
import hildon
import uuid
import os
import ConfigParser
from datetime import date
from datetime import timedelta
from datetime import datetime
import urllib
import webbrowser
import glob
import shutil
import sys
import dbus
from BeautifulSoup import BeautifulStoneSoup
import feedcircuit
from config import FeedConfig
from config import FeedCircuitConfig

version = "0.8.4"


def init_osso_context():
    try:
        import osso
        osso_context = osso.Context("org.maemo.feedcircuit", version, False)
    except ImportError:
        pass


def show_file_chooser(title, parent, action, filename = None):
    try:
        dlg = hildon.FileChooserDialog(parent, action, hildon.FileSystemModel())
        if filename:
            dlg.set_filename(filename)
        result = None
        if dlg.run() == gtk.RESPONSE_OK:
            result = dlg.get_filename()
        return result
    finally:
        dlg.destroy()


def create_config_dir():
    path = os.path.expanduser("~/.config/feedcircuit")
    if not os.path.exists(path):
        os.makedirs(path)
    return path


def save_icon(filename, icon_data):
    f = open(filename, "w")
    try:
        f.write(icon_data)
    finally:
        f.close()


class BannerThread(Thread):


    def __init__(self, parent, text):
        Thread.__init__(self)
        self.__parent = parent
        hildon.hildon_gtk_window_set_progress_indicator(self.__parent, 1)
        self.__banner = hildon.hildon_banner_show_animation(parent, "qgn_indi_pball_a", text)


    def run(self):
        try:
            result = self.__callback(*self.__args)
            gobject.idle_add(self.done, result, None)
        except Exception, ex:
            gobject.idle_add(self.done, None, ex)


    def done(self, result, ex):
        self.__result = result
        self.__exception = ex
        gtk.main_quit()


    def execute(self, callback, *args):
        self.__callback = callback
        self.__args = args
        self.__result = None
        self.__exception = None
        self.start()
        gtk.main()
        self.__banner.destroy()
        hildon.hildon_gtk_window_set_progress_indicator(self.__parent, 0)
        if self.__exception:
            raise self.__exception
        return self.__result


class PreferencesWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Preferences"))
        hbox = gtk.HBox()
        self.path_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        hbox.pack_start(self.path_entry, expand = True)
        self.choose_button = gtk.ToolButton(stock_id = gtk.STOCK_DIRECTORY)
        self.choose_button.connect("clicked", self.choose_button_clicked)
        hbox.pack_start(self.choose_button, expand = False)
        caption = hildon.Caption(None, _("Cache path"), hbox, None, True)
        self.vbox.pack_start(caption)
        self.new_win_check = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.new_win_check.set_label(_("Open in new window"))
        self.vbox.pack_start(self.new_win_check)
        self.autorefresh_check = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.autorefresh_check.set_label(_("Refresh automatically"))
        self.autorefresh_check.connect("toggled", self.update)
        self.vbox.pack_start(self.autorefresh_check)
        time_hbox = gtk.HBox()
        time_mode_hbox = gtk.HBox(homogeneous=True)
        self.refresh_time = hildon.GtkToggleButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.refresh_time.set_label(_("Refresh time"))
        self.refresh_time.connect("toggled", self.update)
        time_mode_hbox.pack_start(self.refresh_time)
        self.refresh_interval = hildon.GtkToggleButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.refresh_interval.set_label(_("Refresh interval"))
        self.refresh_interval.connect("toggled", self.update)
        time_mode_hbox.pack_start(self.refresh_interval)
        time_hbox.pack_start(time_mode_hbox)
        self.time_selector = hildon.TimeButton(gtk.HILDON_SIZE_FINGER_HEIGHT,hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.time_selector.set_title("")
        time_hbox.pack_start(self.time_selector)
        self.vbox.pack_start(time_hbox)
        self.show_all()


    def choose_button_clicked(self, source):
        filename = show_file_chooser(_("Select folder"), self,
            gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER, self.path_entry.get_text())
        if filename:
            self.path_entry.set_text(filename)


    def update(self, source):
        self.refresh_interval.set_sensitive(self.autorefresh_check.get_active())
        self.refresh_time.set_sensitive(self.autorefresh_check.get_active())
        self.time_selector.set_sensitive(self.autorefresh_check.get_active())
        if source is self.refresh_time:
            self.refresh_interval.set_active(not self.refresh_time.get_active())
        else:
            self.refresh_time.set_active(not self.refresh_interval.get_active())


class ContentWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Content settings"))
        self.content_tag_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.content_tag_entry.connect("changed", self.something_changed)
        caption = hildon.Caption(None, _("Content tag"), self.content_tag_entry, None, True)
        self.vbox.pack_start(caption)
        self.reformatter_url_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.reformatter_url_entry.connect("changed", self.something_changed)
        caption = hildon.Caption(None, _("Reformatting service url"), self.reformatter_url_entry, None, True)
        self.vbox.pack_start(caption)
        self.print_version_keywords_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        caption = hildon.Caption(None, _("Printable version link"), self.print_version_keywords_entry, None, True)
        self.vbox.pack_start(caption)
        self.single_page_keywords_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        caption = hildon.Caption(None, _("Single page link"), self.single_page_keywords_entry, None, True)
        self.vbox.pack_start(caption)
        self.next_page_keywords_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        caption = hildon.Caption(None, _("Next page link"), self.next_page_keywords_entry, None, True)
        self.vbox.pack_start(caption)
        self.show_all()


    def something_changed(self, source):
        self.content_tag_entry.set_sensitive(len(self.reformatter_url_entry.get_text()) == 0)
        self.reformatter_url_entry.set_sensitive(len(self.content_tag_entry.get_text()) == 0)


class UrlsWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.resize(*parent.get_size())
        self.set_title(_("Feed urls"))
        self.vbox.pack_start(gtk.Label(_("One url per line")), expand = False)
        self.text_buffer = gtk.TextBuffer()
        pa = hildon.PannableArea()
        text_view = hildon.TextView()
        text_view.set_buffer(self.text_buffer)
        pa.add_with_viewport(text_view)
        self.vbox.pack_start(pa, expand = True)
        self.show_all()


class EditWindow(gtk.Dialog):


    def __init__(self, parent, feed_config):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Edit feed"))
        self.feed_config = feed_config
        self.content_tag = feed_config.content_tag
        self.reformatter_url = feed_config.reformatter_url
        self.print_version_keywords = feed_config.get_print_version_keywords_string()
        self.single_page_keywords = feed_config.get_single_page_keywords_string()
        self.next_page_keywords = feed_config.get_next_page_keywords_string()
        pa = hildon.PannableArea()
        pa.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
        self.vbox.pack_start(pa)
        vbox = gtk.VBox()
        hbox = gtk.HBox()
        self.url_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.url_entry.set_text(feed_config.url)
        self.edit_button = gtk.ToolButton(stock_id = gtk.STOCK_EDIT)
        self.edit_button.connect("clicked", self.edit_button_clicked)
        hbox.pack_start(self.url_entry, expand = True)
        hbox.pack_start(self.edit_button, expand = False)
        caption = hildon.Caption(None, _("Url"), hbox, None, True)
        vbox.pack_start(caption)
        self.name_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.name_entry.set_text(feed_config.name)
        caption = hildon.Caption(None, _("Name"), self.name_entry, None, True)
        vbox.pack_start(caption)
        hbox = gtk.HBox(spacing = 3)
        self.content_button = gtk.ToolButton(gtk.STOCK_PROPERTIES)
        self.content_button.connect("clicked", self.content_clicked)
        model = gtk.ListStore(gobject.TYPE_STRING)
        self.mode_combo = hildon.PickerButton(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        selector = hildon.TouchSelector()
        text_col = selector.append_text_column(model, False)
        text_col.set_text_column(0)
        selector.connect("changed", self.mode_changed)
        selector.append_text(_("Do nothing"))
        selector.append_text(_("Cache"))
        selector.append_text(_("Inline"))
        self.mode_combo.set_selector(selector)
        self.mode_combo.set_active(feed_config.mode)
        hbox.pack_start(self.mode_combo)
        hbox.pack_start(self.content_button, expand = False)
        caption = hildon.Caption(None, _("Content"), hbox, None, True)
        vbox.pack_start(caption)
        self.days_editor = hildon.NumberEditor(0, 365)
        self.days_editor.set_value(feed_config.days_to_keep)
        caption = hildon.Caption(None, _("Days to keep"), self.days_editor, None, True)
        vbox.pack_start(caption)
        self.include_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.include_entry.set_text(feed_config.include)
        caption = hildon.Caption(None, _("Include urls"), self.include_entry, None, True)
        vbox.pack_start(caption)
        self.exclude_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        self.exclude_entry.set_text(feed_config.exclude)
        caption = hildon.Caption(None, _("Exclude urls"), self.exclude_entry, None, True)
        vbox.pack_start(caption)
        self.ignore_images_check = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.ignore_images_check.set_active(feed_config.ignore_images)
        self.ignore_images_check.set_label(_("Ignore images"))
        vbox.pack_start(self.ignore_images_check)
        pa.add_with_viewport(vbox)
        self.show_all()


    def mode_changed(self, column, user_data):
        self.content_button.set_sensitive(self.mode_combo.get_active() != FeedConfig.mode_ommit)


    def update_feed(self):
        self.feed_config.url = self.url_entry.get_text()
        self.feed_config.name = self.name_entry.get_text()
        self.feed_config.mode = self.mode_combo.get_active()
        self.feed_config.days_to_keep = self.days_editor.get_value()
        self.feed_config.include = self.include_entry.get_text()
        self.feed_config.exclude = self.exclude_entry.get_text()
        self.feed_config.content_tag = self.content_tag
        self.feed_config.reformatter_url = self.reformatter_url
        self.feed_config.set_print_version_keywords_string(self.print_version_keywords)
        self.feed_config.set_single_page_keywords_string(self.single_page_keywords)
        self.feed_config.set_next_page_keywords_string(self.next_page_keywords)
        self.feed_config.ignore_images = self.ignore_images_check.get_active()


    def content_clicked(self, source):
        cnt_window = ContentWindow(self)
        cnt_window.content_tag_entry.set_text(self.content_tag)
        cnt_window.reformatter_url_entry.set_text(self.reformatter_url)
        cnt_window.print_version_keywords_entry.set_text(self.print_version_keywords)
        cnt_window.single_page_keywords_entry.set_text(self.single_page_keywords)
        cnt_window.next_page_keywords_entry.set_text(self.next_page_keywords)
        if cnt_window.run() == gtk.RESPONSE_ACCEPT:
            self.content_tag = cnt_window.content_tag_entry.get_text()
            self.reformatter_url = cnt_window.reformatter_url_entry.get_text()
            self.print_version_keywords = cnt_window.print_version_keywords_entry.get_text()
            self.single_page_keywords = cnt_window.single_page_keywords_entry.get_text()
            self.next_page_keywords = cnt_window.next_page_keywords_entry.get_text()
        cnt_window.destroy()


    def edit_button_clicked(self, source):
        urls_window = UrlsWindow(self)
        tb = urls_window.text_buffer
        tb.set_text(self.url_entry.get_text().replace(" ", "\n"))
        if urls_window.run() == gtk.RESPONSE_ACCEPT:
            text = tb.get_text(tb.get_start_iter(), tb.get_end_iter(), include_hidden_chars = False)
            lines = filter(len, [line.strip().replace(" ", "%20") for line in text.split("\n")])
            self.url_entry.set_text(reduce(lambda s1, s2: s1 + " " + s2, lines))
        urls_window.destroy()


class AddWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("New feed"))
        self.url_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        caption = hildon.Caption(None, _("Feed url"), self.url_entry, None, True)
        self.vbox.add(caption)
        self.show_all()


class ImportWindow(gtk.Dialog):


    def __init__(self, parent):
        gtk.Dialog.__init__(self, None, parent,
            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        self.set_title(_("Import"))
        hbox = gtk.HBox()
        self.path_entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
        hbox.pack_start(self.path_entry, expand = True)
        self.choose_button = gtk.ToolButton(stock_id = gtk.STOCK_DIRECTORY)
        self.choose_button.connect("clicked", self.choose_button_clicked)
        hbox.pack_start(self.choose_button, expand = False)
        caption = hildon.Caption(None, _("From OPML file"), hbox, None, True)
        self.vbox.pack_start(caption)
        self.stock_check = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.stock_check.set_label(_("From default reader"))
        self.stock_check.connect("toggled", self.stock_check_changed)
        self.vbox.pack_start(self.stock_check)
        self.show_all()


    def choose_button_clicked(self, source):
        filename = None
        if self.path_entry.get_text():
            filename = os.path.abspath(self.path_entry.get_text())
        filename = show_file_chooser(_("Select opml file"), self, 
            gtk.FILE_CHOOSER_ACTION_OPEN, filename)
        if filename:
            self.path_entry.set_text(filename)


    def stock_check_changed(self, source):
        self.path_entry.set_sensitive(not self.stock_check.get_active())
        self.choose_button.set_sensitive(not self.stock_check.get_active())


class RefreshProgressWindow(hildon.Note):


    def __init__(self, parent):
        self.__progress_bar = gtk.ProgressBar()
        hildon.Note.__init__(self, "cancel", parent, "", progressbar = self.__progress_bar)
        self.__name = ""
        self.__detail = ""


    def update(self, fraction = None, name = None, detail = None):
        if fraction:
            self.__progress_bar.set_fraction(fraction)
        if name:
            self.__name = name
        if detail:
            self.__detail = detail
        self.set_property("description", self.__name + "\n" + self.__detail)


class EditFeedsWindow(hildon.StackableWindow):


    def __init__(self, model):
        hildon.StackableWindow.__init__(self)
        
        self.set_title(_("Edit feeds"))
        
        self.model = model
        
        vbox = gtk.VBox()
        self.add(vbox)

        hbox = gtk.HBox()
        self.edit_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.edit_button.set_image(gtk.image_new_from_stock(gtk.STOCK_EDIT, gtk.ICON_SIZE_BUTTON))
        self.edit_button.connect("clicked", self.edit)
        hbox.pack_start(self.edit_button)
        self.delete_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.delete_button.set_image(gtk.image_new_from_stock(gtk.STOCK_DELETE, gtk.ICON_SIZE_BUTTON))
        self.delete_button.connect("clicked", self.delete)
        hbox.pack_start(self.delete_button)
        self.clear_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.clear_button.set_image(gtk.image_new_from_stock(gtk.STOCK_CLEAR, gtk.ICON_SIZE_BUTTON))
        self.clear_button.connect("clicked", self.clear)
        hbox.pack_start(self.clear_button)
        self.up_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.up_button.set_image(gtk.image_new_from_stock(gtk.STOCK_GO_UP, gtk.ICON_SIZE_BUTTON))
        self.up_button.connect("clicked", self.up)
        hbox.pack_start(self.up_button)
        self.down_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        self.down_button.set_image(gtk.image_new_from_stock(gtk.STOCK_GO_DOWN, gtk.ICON_SIZE_BUTTON))
        self.down_button.connect("clicked", self.down)
        hbox.pack_start(self.down_button)
        vbox.pack_start(hbox, expand = False)

        pa = hildon.PannableArea()
        vbox.pack_start(pa)

        self.view = hildon.GtkTreeView(gtk.HILDON_UI_MODE_EDIT, self.model)
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.view.get_selection().connect("changed", self.selection_changed)
        main_column = gtk.TreeViewColumn()
        render_pixbuf = gtk.CellRendererPixbuf()
        main_column.pack_start(render_pixbuf, expand=False)
        main_column.add_attribute(render_pixbuf, 'pixbuf', 0)
        render_text = gtk.CellRendererText()
        render_text.set_property("ellipsize", pango.ELLIPSIZE_MIDDLE)
        main_column.pack_start(render_text, expand=True)
        main_column.add_attribute(render_text, 'markup', 1)
        main_column.set_expand(True)
        self.view.append_column(main_column)
        pa.add(self.view)

        self.selection_changed(self.view)
        self.show_all()


    def selection_changed(self, source):
        pathes = self.view.get_selection().get_selected_rows()[1]
        self.edit_button.set_sensitive(len(pathes) == 1)
        self.delete_button.set_sensitive(len(pathes) > 0)
        self.clear_button.set_sensitive(len(pathes) > 0)
        self.up_button.set_sensitive(len(pathes) == 1 and pathes[0][0] > 0)
        self.down_button.set_sensitive(len(pathes) == 1 and (pathes[0][0] + 1) < len(self.model.config.list))


    def edit(self, source):
        pathes = self.view.get_selection().get_selected_rows()[1]
        if len(pathes) > 0:
            feed_config = self.model.config.list[pathes[0][0]]
            edit_window = EditWindow(self, feed_config)
            result = edit_window.run()
            if result == gtk.RESPONSE_ACCEPT:
                edit_window.update_feed()
                self.model.update(feed_config.id)
                edit_window.destroy()


    def remove_feeds(self, feed_configs, delete):
        for feed_config in feed_configs:
            feed_config.message_count = 0
            feed = feedcircuit.Feed(None, path = self.model.config.cache_path,
                cache_path = ".cache", filename = feed_config.get_filename())
            feed.delete()
            gobject.idle_add(self.model.update, feed_config.id)
            if delete:
                if feed_config.icon_filename:
                    icon_filepath = os.path.join(self.model.config.cache_path, feed_config.icon_filename)
                    if os.path.exists(icon_filepath):
                        os.remove(icon_filepath)
                gobject.idle_add(self.model.delete, feed_config.id)


    def remove_feeds_async(self, delete):
        pathes = self.view.get_selection().get_selected_rows()[1]
        if len(pathes):
            feed_configs = map(lambda path: self.model.config.list[path[0]], pathes)
            if delete:
                message = _("Delete feeds?")
            else:
                message = _("Clear feeds?")
            note = hildon.Note("confirmation", self, message)
            note_result = note.run()
            note.destroy()
            if note_result == gtk.RESPONSE_OK:
                remove_thread = BannerThread(self, _("clearing feeds"))
                remove_thread.execute(self.remove_feeds, feed_configs, delete)


    def delete(self, source):
        self.remove_feeds_async(True)


    def clear(self, source):
        self.remove_feeds_async(False)


    def up(self, source):
        pathes = self.view.get_selection().get_selected_rows()[1]
        if len(pathes) > 0:
            index = pathes[0][0]
            if index > 0:
                self.model.move(self.model.config.list[index].id, index - 1)


    def down(self, source):
        pathes = self.view.get_selection().get_selected_rows()[1]
        if len(pathes) > 0:
            index = pathes[0][0]
            self.model.move(self.model.config.list[index].id, index + 2)


class PickFeedsDialog(hildon.PickerDialog):


    def __init__(self, parent, model):
        hildon.PickerDialog.__init__(self, parent)
        self.set_title(_("Choose feeds to refresh"))
        self.__model = model
        selector = hildon.TouchSelector()
        #this stuff is done in classic way because "text-column" of TouchSelectorColumn doesn't work properly
        renderer = gtk.CellRendererText()
        text_col = selector.append_column(self.__model, renderer)
        text_col.add_attribute(renderer, 'text', 4)
        selector.set_column_selection_mode(hildon.TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE)
        self.set_selector(selector)
        


class Program:


    def __init__(self):
        self.__prog = hildon.Program.get_instance()
        self.__window = hildon.StackableWindow()
        self.__window.set_title("Feed circuit")
        self.__window.connect("destroy", self.stop)
        self.__prog.add_window(self.__window)
        
        self.__update_lock = Lock()

        self.__model = FeedListModel(create_config_dir())
        
        if not os.path.exists(self.__model.get_config_file_name()):
            self.import_opml()

        if self.__model.config.browser:
            webbrowser.register("FeedCircuitCustom", None, webbrowser.BackgroundBrowser(self.__model.config.browser.split()))


        vbox = gtk.VBox()
        self.__window.add(vbox)

        hbox = gtk.HBox()
        refresh_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        refresh_button.set_image(gtk.image_new_from_stock(gtk.STOCK_REFRESH, gtk.ICON_SIZE_BUTTON))
        refresh_button.connect("clicked", self.refresh)
        hbox.pack_start(refresh_button)
        add_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        add_button.set_image(gtk.image_new_from_stock(gtk.STOCK_ADD, gtk.ICON_SIZE_BUTTON))
        add_button.connect("clicked", self.add)
        hbox.pack_start(add_button)
        vbox.pack_start(hbox, expand = False)
        
        menu = hildon.AppMenu()
        edit_mode_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        edit_mode_button.set_title(_("Edit feeds"))
        edit_mode_button.connect("clicked", self.edit_feeds)
        menu.append(edit_mode_button)
        refresh_selected_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        refresh_selected_button.set_title(_("Refresh selected"))
        refresh_selected_button.connect("clicked", self.refresh_selected)
        menu.append(refresh_selected_button)
        import_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        import_button.set_title(_("Import"))
        import_button.connect("clicked", self.import_feeds)
        menu.append(import_button)
        export_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        export_button.set_title(_("Export"))
        export_button.connect("clicked", self.export_feeds)
        menu.append(export_button)
        pref_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        pref_button.set_title(_("Preferences"))
        pref_button.connect("clicked", self.preferences)
        menu.append(pref_button)
        about_button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_HORIZONTAL)
        about_button.set_title(_("About"))
        about_button.connect("clicked", self.about)
        menu.append(about_button)
        menu.show_all()
        self.__window.set_app_menu(menu)

        self.__view = hildon.GtkTreeView(gtk.HILDON_UI_MODE_NORMAL, self.__model)
        self.__main_column = gtk.TreeViewColumn()
        render_pixbuf = gtk.CellRendererPixbuf()
        self.__main_column.pack_start(render_pixbuf, expand=False)
        self.__main_column.add_attribute(render_pixbuf, 'pixbuf', 0)
        render_text = gtk.CellRendererText()
        render_text.set_property("ellipsize", pango.ELLIPSIZE_MIDDLE)
        self.__main_column.pack_start(render_text, expand=True)
        self.__main_column.add_attribute(render_text, 'markup', 1)
        self.__main_column.set_expand(True)
        self.__view.append_column(self.__main_column)
        self.__view.connect("row_activated", self.row_activated)

        pannable_area = hildon.PannableArea()
        pannable_area.add(self.__view)
        vbox.pack_start(pannable_area, expand = True)

        self.__timer_id = 0
        self.schedule_next_update()

        init_osso_context()

    def run(self):
        self.__window.show_all()
        gtk.main()


    def stop(self, source):
        gtk.main_quit()


    def open_url(self, url, new_win):
        if self.__model.config.browser:
            webbrowser.get("FeedCircuitCustom").open("file://" + url, new_win)
        else:
            #webbrowser.open(url, new_win)
            browser = bus.get_object("com.nokia.osso_browser", "/com/nokia/osso_browser/request")
            if new_win:
                browser.open_new_window(url, dbus_interface = "com.nokia.osso_browser")
            else:
                browser.load_url(url, dbus_interface = "com.nokia.osso_browser")


    def open_feed(self, feed_config):
        path = os.path.join(self.__model.config.cache_path, feed_config.get_filename())
        if os.path.exists(path):
            new_win = 0
            if self.__model.config.use_new_window:
                new_win = 2
            self.open_url(path, new_win)
            feed_config.message_count = 0
            self.__model.update(feed_config.id)
        else:
            note = hildon.Note("information", self.__window, _("The feed is empty"))
            note.run()
            note.destroy()


    def move_cache(self, from_path, to_path):
        if not os.path.exists(to_path):
            os.makedirs(to_path)
        for feed_config in self.__model.config.list:
            filename = os.path.join(from_path, feed_config.get_filename())
            if os.path.exists(filename):
                shutil.move(filename, os.path.join(to_path, feed_config.get_filename()))
            if feed_config.icon_filename:
                icon_path = os.path.join(from_path, feed_config.icon_filename)
                if os.path.exists(icon_path):
                    shutil.move(icon_path, os.path.join(to_path, feed_config.icon_filename))
        dot_cache_path = os.path.join(from_path, ".cache")
        if os.path.exists(dot_cache_path):
            shutil.move(dot_cache_path, os.path.join(to_path, ".cache"))


    def row_activated(self, source, path, view_column):
        feed_config = self.__model.config.list[path[0]]
        self.open_feed(feed_config)


    def add(self, source):
        add_window = AddWindow(self.__window)
        response = add_window.run()
        add_window.hide()
        if response == gtk.RESPONSE_ACCEPT:
            url = add_window.url_entry.get_text()
            title = ""
            icon_filename = ""
            try:
                discovery_thread = BannerThread(self.__window, _("Discovering feed"))
                (url, title, icon_filename, icon_data) = \
                    discovery_thread.execute(feedcircuit.discover_feed, url)
                if icon_filename:
                    icon_filename = uuid.uuid1().hex + icon_filename
                    save_icon(os.path.join(self.__model.config.cache_path, icon_filename), icon_data)
            except Exception, ex:
                error_note = hildon.Note("information", self.__window,
                    _("Cannot discover feed at given url") + "\n" + str(ex)) 
                error_note.run()
                error_note.destroy()
            feed_config = FeedConfig(url, title, FeedConfig.mode_ommit, icon_filename)
            edit_window = EditWindow(self.__window, feed_config)
            if edit_window.run() == gtk.RESPONSE_ACCEPT:
                edit_window.update_feed()
                self.__model.append(feed_config)
            elif icon_filename:
                os.remove(os.path.join(self.__model.config.cache_path, icon_filename))
            edit_window.destroy()
        add_window.destroy()


    def preferences(self, source):
        pref_window = PreferencesWindow(self.__window)
        try:
            pref_window.path_entry.set_text(self.__model.config.cache_path)
            pref_window.new_win_check.set_active(self.__model.config.use_new_window)
            pref_window.refresh_interval.set_active(self.__model.config.is_interval)
            time = self.__model.get_autorefresh_time()
            if time:
                pref_window.autorefresh_check.set_active(True)
                pref_window.time_selector.set_time(*time)
            pref_window.update(None)
            if pref_window.run() == gtk.RESPONSE_ACCEPT:
                new_path = os.path.abspath(pref_window.path_entry.get_text())
                if new_path != self.__model.config.cache_path:
                    move_thread = BannerThread(self.__window, _("moving cache"))
                    move_thread.execute(self.move_cache, self.__model.config.cache_path, new_path)
                    self.__model.update_cache_path(new_path)
                self.__model.config.use_new_window = pref_window.new_win_check.get_active()
                self.__model.set_autorefresh_time(None)
                self.__model.config.is_interval = pref_window.refresh_interval.get_active()
                if pref_window.autorefresh_check.get_active():
                    self.__model.set_autorefresh_time(pref_window.time_selector.get_time())
                self.schedule_next_update()
                self.__model.save_config()
        except Exception, ex:
            error_note = hildon.Note("information", self.__window, str(ex)) 
            error_note.run()
            error_note.destroy()
        finally:
            pref_window.destroy()


    def import_opml(self, filename = None):
        fav_dir = None
        if not filename:
            filename = os.path.expanduser("~/.osso_rss_feed_reader/feedlist.opml")
            fav_dir = os.path.expanduser("~/.osso_rss_feed_reader/cache/favicons")
        if os.path.exists(filename):
            f = open(filename, "r")
            try:
                opml = BeautifulStoneSoup(f)
                outlines = filter(
                    lambda o: o.has_key("title") and o.has_key("xmlurl"),
                    opml.findAll("outline"))
                for outline in outlines:
                    icon_file_name = ""
                    if fav_dir:
                        if outline.has_key("id"):
                            icon_file = outline["id"] + ".png"
                            icon_file_src = os.path.join(fav_dir, icon_file)
                            if os.path.exists(icon_file_src):
                                icon_file_name = uuid.uuid1().hex + icon_file
                                shutil.copy(icon_file_src,
                                    os.path.join(self.__model.config.cache_path, icon_file_name))
                    else:
                        try:
                            discovery_thread = BannerThread(self.__window, _("Importing feeds"))
                            (url, title, icon_file_name, icon_data) = \
                                discovery_thread.execute(feedcircuit.get_feed_details, outline["xmlurl"])
                            if icon_file_name:
                                icon_file_name = uuid.uuid1().hex + icon_file_name
                                save_icon(os.path.join(self.__model.config.cache_path, icon_file_name), icon_data)
                        except Exception, ex:
                            error_note = hildon.Note("information", self.__window,
                                _("Cannot discover feed at given url") + "\n" + str(ex)) 
                            error_note.run()
                            error_note.destroy()
                    feed_config = FeedConfig(outline["xmlurl"], outline["title"], FeedConfig.mode_ommit, icon_file_name)
                    self.__model.append(feed_config)
            finally:
                f.close()


    def import_feeds(self, source):
        imp_window = ImportWindow(self.__window)
        try:
            result = imp_window.run()
            imp_window.hide()
            if result == gtk.RESPONSE_ACCEPT:
                filename = imp_window.path_entry.get_text()
                if imp_window.stock_check.get_active():
                    self.import_opml()
                elif filename:
                    self.import_opml(filename)
        except Exception, ex:
            error_note = hildon.Note("information", self.__window, str(ex)) 
            error_note.run()
            error_note.destroy()
        finally:
            imp_window.destroy()


    def export_feeds(self, source):
        filename = show_file_chooser(_("Select opml file"), self.__window, gtk.FILE_CHOOSER_ACTION_SAVE)
        if filename:
            if not filename.endswith(".opml"):
                filename += ".opml"
            f = open(filename, "w")
            try:
                f.write('<?xml version="1.0" encoding="utf-8"?>\n')
                f.write('<opml version="1.0">\n')
                f.write('<head/>\n')
                f.write('<body>\n')
                for feed_config in self.__model.config.list:
                    title = feed_config.name.replace("<", "&lt;").replace(">", "&gt;")
                    for url in feed_config.url.split(" "):
                        f.write('<outline title="%s" description="%s" xmlUrl="%s" /> \n' %
                            (title, title, url))
                f.write('</body>\n')
                f.write('</opml>\n')
            finally:
                f.close()


    def about(self, source):
        dialog = gtk.AboutDialog()
        dialog.set_name("FeedCircuit")
        dialog.set_version(version)
        dialog.set_website("http://feedcircuit.garage.maemo.org")
        dialog.set_authors([_("George Kibardin")])
        dialog.show_all()
        dialog.run()
        dialog.destroy()


    def save_log(self):
        try:
            f = open(os.path.join(self.__model.config_path, "refresh.log"), "w")
            try:
                f.writelines(self.__log)
            finally:
                f.close()
        except:
            pass


    def log(self, text):
        self.__log.append("%s %s\n" % (datetime.today().isoformat(), text))


    def update_started(self):
        self.__log = []


    def update_status(self, feed_config, action, item, i, feed_count, percent, j, url_count, details):
        if self.__progress_window:
            fraction = None
            if percent:
                fraction = (j * 100.0 + percent) / (url_count * 100.0)
            self.__progress_window.update(
                fraction,
                "%s (%u/%u)" % (feed_config.name, i, feed_count),
                "%s %s" % (_(action), item))
        self.log("(%u/%u) %s %s" % (i, feed_count, _(action), item))
        if details:
            self.log(details)


    def update_feed_complete(self, feed_config, new_item_count):
        feed_config.message_count += new_item_count
        self.__model.update(feed_config.id)
        if self.__progress_window:
            self.__progress_window.update(fraction = 0.0)


    def update_complete(self):
        if self.__progress_window:
            self.__progress_window.response(gtk.RESPONSE_NONE)
        self.save_log()


    def callback(self, item):
        abort = self.__aborted
        self.__aborted = False
        if abort:
            sys.exit()
        if item[0]:
            gobject.idle_add(self.update_status, self.__progress_info[0], item[0], item[1],
                self.__progress_info[1], self.__progress_info[2], item[2],
                self.__progress_info[3], self.__progress_info[4], item[3])


    def update_feeds(self, feed_configs):
        self.__update_lock.acquire()
        self.__aborted = False
        gobject.idle_add(self.update_started)
        try:
            i = 0
            for feed_config in feed_configs:
                i += 1
                new_item_count = 0
                urls = feed_config.get_urls()
                url_count = len(urls)
                feed_count = len(feed_configs)
                (item_t, update_t, feed_t, page_t) = feed_config.get_templates(self.__model.config_path)
                title = None
                if url_count == 1:
                    title = feed_config.name
                j = 0
                try:
                    for url in urls:
                        feed = None
                        self.__progress_info = (feed_config, i, feed_count, j, url_count)
                        del_before = None
                        clear_cache = True
                        try:
                            if feed_config.days_to_keep:
                                del_before = date.today() - timedelta(feed_config.days_to_keep)
                                clear_cache = False
                            feed = feedcircuit.Feed(url = url, path = self.__model.config.cache_path, cache_path = ".cache",
                                title = title, filename = feed_config.get_filename(),
                                callback = self.callback,
                                delete_before_date = del_before, always_clear_cache = clear_cache,
                                inline_items = feed_config.mode == FeedConfig.mode_inline,
                                cache_items = feed_config.mode == FeedConfig.mode_cache,
                                include = feed_config.include, exclude = feed_config.exclude,
                                print_version_keywords = feed_config.print_version_keywords,
                                single_page_keywords = feed_config.single_page_keywords,
                                next_page_keywords = feed_config.next_page_keywords,
                                reformatter_url = feed_config.reformatter_url,
                                content_kwargs = feed_config.get_content_kwargs(),
                                ignore_images = feed_config.ignore_images,
                                new_window = self.__model.config.use_new_window,
                                feed_item_template = item_t, feed_update_template = update_t,
                                page_template = page_t, feed_template = feed_t)
                            feed.update()
                            j += 1
                        except Exception:
                            pass
                        finally:
                            if feed:
                                new_item_count += feed.new_item_count
                finally:
                    gobject.idle_add(self.update_feed_complete, feed_config, new_item_count)
        except BaseException:
            pass
        finally:
            gobject.idle_add(self.update_complete)
            self.__update_lock.release()


    def update_feeds_async(self, feeds_configs):
        self.__progress_window = RefreshProgressWindow(self.__window)
        self.__thread = Thread(target = self.update_feeds, args = (feeds_configs,))
        self.__thread.start()
        try:
            gtk.gdk.threads_enter()
            if self.__progress_window.run() == gtk.RESPONSE_CANCEL:
                self.__aborted = True
        finally:
            self.__progress_window.hide()
            self.__progress_window.destroy()
            self.__progress_window = None
            gtk.gdk.threads_leave()


    def refresh(self, source):
        connect_and_exec(False, None, self.update_feeds_async, list(self.__model.config.list))


    def refresh_selected(self, source):
        picker_dialog = PickFeedsDialog(self.__window, self.__model)
        try:
            result = picker_dialog.run()
            picker_dialog.hide()
            if result == gtk.RESPONSE_OK:
                pathes = picker_dialog.get_selector().get_selected_rows(0)
                connect_and_exec(False, None, self.update_feeds_async,
                    map(lambda path: self.__model.config.list[path[0]], pathes))
        finally:
            picker_dialog.destroy()


    def schedule_next_update(self):
        if self.__timer_id:
            gobject.source_remove(self.__timer_id)
            self.__timer_id = 0
        time = self.__model.get_autorefresh_time()
        if time:
            now = datetime.now()
            if self.__model.config.is_interval:
                delta = timedelta(hours = time[0], minutes = time[1])
            else:
                next_time = datetime(now.year, now.month, now.day, time[0], time[1], 0)
                if next_time < (now + timedelta(seconds = 10)):
                    next_time = next_time + timedelta(1)
                delta = next_time - now
            self.__timer_id = gobject.timeout_add(
                (delta.days * 24 * 3600 + delta.seconds) * 1000, self.auto_refresh)


    def auto_refresh(self):
        try:
            connect_and_exec(True, self.__model.config.connection_id,
                self.update_feeds_async, list(self.__model.config.list))
        finally:
            self.schedule_next_update()


    def edit_feeds(self, source):
        win = EditFeedsWindow(self.__model)


class FeedListModel(gtk.GenericTreeModel):


    def __init__(self, config_path):
        gtk.GenericTreeModel.__init__(self)
        self.column_types = (gtk.gdk.Pixbuf, str, str, gobject.TYPE_BOOLEAN,
            str, str, int)
        self.map = {}
        self.icons = {}
        self.config_path = config_path
        self.config = FeedCircuitConfig(self.get_config_file_name())
        self.rebuild_map()


    def get_feed_icon(self, feed_config):
        icon = self.icons.get(feed_config.id)
        if not icon:
            if feed_config.icon_filename:
                icon_filepath = os.path.join(self.config.cache_path, feed_config.icon_filename)
                if os.path.exists(icon_filepath):
                    icon = gtk.gdk.pixbuf_new_from_file(icon_filepath)
                    if icon.get_height() > 48:
                        icon = icon.scale_simple(
                            icon.get_width() * 48 / icon.get_height(), 48, gtk.gdk.INTERP_BILINEAR)
                    self.icons[feed_config.id] = icon
        return icon


    def get_feed_time(self, feed_config):
        filename = os.path.join(self.config.cache_path, feed_config.get_filename())
        if os.path.exists(filename):
            return datetime.fromtimestamp(os.path.getmtime(filename)).strftime("%c")
        return ""


    def get_feed_markup(self, feed_config):
        result = "\n" + self.get_feed_time(feed_config)
        if feed_config.message_count:
            result = "<b>" + feed_config.name.replace("&", "&amp;") + " (" + str(feed_config.message_count) + ")</b>" + result
        else:
            result = feed_config.name.replace("&", "&amp;") + result
        return result


    def on_get_flags(self):
        return gtk.TREE_MODEL_LIST_ONLY|gtk.TREE_MODEL_ITERS_PERSIST


    def on_get_n_columns(self):
        return len(self.column_types)


    def on_get_column_type(self, n):
        return self.column_types[n]


    def on_get_iter(self, path):
        if path[0] < len(self.config.list):
            return self.config.list[path[0]].id
        return None


    def on_get_path(self, rowref):
        return self.map[rowref]


    def on_get_value(self, rowref, column):
        if self.map.has_key(rowref):
            if column is 0:
                return self.get_feed_icon(self.config.list[self.map[rowref]])
            elif column is 1:
                return self.get_feed_markup(self.config.list[self.map[rowref]])
            elif column is 2:
                return self.get_feed_time(self.config.list[self.map[rowref]])
            elif column is 3:
                return self.config.list[self.map[rowref]].checked
            elif column is 4:
                return self.config.list[self.map[rowref]].name
            elif column is 5:
                return gtk.STOCK_EDIT # edit feed icon
            elif column is 6:
                return gtk.ICON_SIZE_BUTTON # edit feed icon size


    def on_iter_next(self, rowref):
        index = self.map[rowref] + 1
        if index < len(self.config.list):
            return self.config.list[index].id
        return None


    def on_iter_children(self, rowref):
        if rowref:
            return None
        return self.config.list[0].id


    def on_iter_has_child(self, rowref):
        return False


    def on_iter_parent(self, child):
        return None


    def on_iter_n_children(self, rowref):
        if rowref:
            return 0
        return len(self.config.list)


    def on_iter_nth_child(self, rowref, n):
        if not rowref and n < len(self.config.list):
            return self.config.list[n].id
        return None


    def get_config_file_name(self):
        return os.path.join(self.config_path, "config")


    def get_autorefresh_time(self):
        if self.config.autorefresh_time:
            tuple = self.config.autorefresh_time.split(":")
            return (int(tuple[0]), int(tuple[1]))


    def set_autorefresh_time(self, time):
        self.config.autorefresh_time = ""
        if (time):
            self.config.autorefresh_time = "%d:%d" % (time[0], time[1])


    def rebuild_map(self):
        self.map.clear()
        for i in range(len(self.config.list)):
            self.map[self.config.list[i].id] = i

    def update(self, rowref):
        index = self.map[rowref]
        self.row_changed((index,), self.get_iter((index,)))
        self.save_config()


    def refresh(self, rowref):
        index = self.map[rowref]
        self.row_changed((index,), self.get_iter((index,)))


    def delete(self, rowref):
        index = self.map[rowref]
        del self.config.list[index]
        self.rebuild_map()
        self.row_deleted((index,))
        self.save_config()


    def append(self, feed_config):
        index = len(self.config.list)
        self.map[feed_config.id] = index
        self.config.list.append(feed_config)
        self.row_inserted((index,), self.get_iter((index,)))
        self.save_config()


    def update_cache_path(self, cache_path):
        self.config.cache_path = cache_path


    def move(self, id, pos):
        before_id = None
        if pos >= 0 and pos < len(self.config.list):
            before_id = self.config.list[pos].id
        if before_id == id:
            return
        old_order = [f.id for f in self.config.list]
        index = self.map[id]
        feed_config = self.config.list[index]
        del self.config.list[index]
        self.rebuild_map()
        if before_id:
            self.config.list.insert(self.map[before_id], feed_config)
        else:
            self.config.list.append(feed_config)
        self.rebuild_map()
        self.rows_reordered(None, None, [self.map[id] for id in old_order])
        self.save_config()


    def toggle(self, path):
        feed_config = self.config.list[path[0]]
        feed_config.checked = not feed_config.checked
        self.row_changed(path, self.get_iter(path))
        self.save_config()


    def save_config(self):
        self.config.save()


gtk.gdk.threads_init()
hildon.hildon_init()

global bus
bus = dbus.SessionBus()

#This is also to be able to start Feed Circuit under the scratchbox
try:
    import conic


    func_to_exec = None
    args_to_use = None


    def connection_cb(connection, event, magic):
        global func_to_exec, args_to_use
        if event.get_status() == conic.STATUS_CONNECTED:
            if func_to_exec:
                gobject.idle_add(func_to_exec, *args_to_use)
                func_to_exec = None
        elif event.get_status() == conic.STATUS_DISCONNECTED:
                func_to_exec = None


    connection = conic.Connection()
    connection.connect("connection-event", connection_cb, None)


    def connect_and_exec(auto, id, func, *args):
        global func_to_exec, args_to_use
        func_to_exec = func
        args_to_use = args
        if auto:
            if id:
                connection.request_connection_by_id(id, conic.CONNECT_FLAG_NONE)
            else:
                connection.request_connection(conic.CONNECT_FLAG_AUTOMATICALLY_TRIGGERED)
        else:
            connection.request_connection(conic.CONNECT_FLAG_NONE)


    def get_connections():
        return [ (iap.get_id(), iap.get_name()) for iap in connection.get_all_iaps() ]


except ImportError:
    
    def connect_and_exec(auto, id, func, *args):
        gobject.idle_add(func, *args)


    def get_connections():
        return [("test1id", "test1"), ("test2id", "test2")]
