/* $Id: eyes.c 2397 2007-01-17 17:46:35Z nick $
 * 
 * Copyright (c) Benedikt Meurer <benedikt.meurer@unix-ag.uni-siegen.de>>
 * Copyright (c) Danny Milosavljevic <danny_milo@gmx.net>
 * Copyright (c) Dave Camp
 * Copyright (c) Davyd Madeley  <davyd@madeley.id.au>
 * Copyright (c) Nick Schermer <nick@xfce.org>
 * Copyright (c) Mikko Vartiainen <mvartiainen@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MATH_H
#include <math.h>
#endif

#include <sys/types.h>
#include <dirent.h>
#include <string.h>

#include <libhildondesktop/libhildondesktop.h>

#include "eyes.h"
#include "themes.h"
#include "accelerometer.h"

/* for xml: */
#define EYES_ROOT      "Eyes"
#define DEFAULTTHEME   "Default"
#define UPDATE_TIMEOUT 100

#define EYES_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
							                         EYES_TYPE_HOME_PLUGIN,\
							                         EyesPluginContent))

HD_DEFINE_PLUGIN_MODULE (EyesPlugin, eyes_plugin,      HD_TYPE_HOME_PLUGIN_ITEM)

/*****************************
 *** Eyes Plugin Functions ***
 *****************************/
static void
calculate_pupil_xy (EyesPluginContent *eyes_applet,
		    gint x, gint y,
		    gint *pupil_x, gint *pupil_y, GtkWidget* widget)
{
	double sina;
	double cosa;
	double h;
	double temp;
	double nx, ny;

	gfloat xalign, yalign;
	gint width, height;

	width = GTK_WIDGET(widget)->allocation.width;
	height = GTK_WIDGET(widget)->allocation.height;
	gtk_misc_get_alignment(GTK_MISC(widget),  &xalign, &yalign);

    nx = width*x/1000.0;
    ny = height*y/1000.0;

	h = hypot (nx, ny);

    if ( nx*nx/((eyes_applet->eye_width/2.0 - eyes_applet->pupil_width / 2)*(eyes_applet->eye_width/2.0 - eyes_applet->pupil_width/2)) + ny*ny/((eyes_applet->eye_height/2 - eyes_applet->pupil_height/2)*(eyes_applet->eye_height/2 - eyes_applet->pupil_height/2)) < 1 )
	{
			*pupil_x = nx + eyes_applet->eye_width / 2;
			*pupil_y = ny + eyes_applet->eye_height / 2;
			return;
	}

	sina = nx / h;
	cosa = ny / h;

	temp = hypot ((eyes_applet->eye_width / 2) * sina, (eyes_applet->eye_height / 2) * cosa);
	temp -= hypot ((eyes_applet->pupil_width / 2) * sina, (eyes_applet->pupil_height / 2) * cosa);
	temp -= hypot ((eyes_applet->wall_thickness / 2) * sina, (eyes_applet->wall_thickness / 2) * cosa);

	*pupil_x = temp * sina + (eyes_applet->eye_width / 2);
	*pupil_y = temp * cosa + (eyes_applet->eye_height / 2);
}



static void
draw_eye (EyesPluginContent *eyes,
          gint    eye_num,
          gint    pupil_x,
          gint    pupil_y)
{
    GdkPixbuf *pixbuf;
    GdkRectangle rect, r1, r2;

    pixbuf = gdk_pixbuf_copy (eyes->eye_image);
    r1.x = pupil_x - eyes->pupil_width / 2;
    r1.y = pupil_y - eyes->pupil_height / 2;
    r1.width = eyes->pupil_width;
    r1.height = eyes->pupil_height;
    r2.x = 0;
    r2.y = 0;
    r2.width = eyes->eye_width;
    r2.height = eyes->eye_height;
    if (gdk_rectangle_intersect (&r1, &r2, &rect))
    {
        gdk_pixbuf_composite (eyes->pupil_image, pixbuf,
                           rect.x,
                           rect.y,
                           rect.width,
                                 rect.height,
                                 pupil_x - eyes->pupil_width / 2,
                           pupil_y - eyes->pupil_height / 2, 1.0, 1.0,
                                 GDK_INTERP_BILINEAR,
                               255);
        gtk_image_set_from_pixbuf (GTK_IMAGE (eyes->eyes[eye_num]),
                              pixbuf);
    }
    g_object_unref (G_OBJECT (pixbuf));
}



static gint
timer_cb(EyesPluginContent *eyes)
{
    gint x, y, z;
    gint pupil_x, pupil_y;
    gint i;

    for (i = 0; i < eyes->num_eyes; i++)
    {
        if (GTK_WIDGET_REALIZED(eyes->eyes[i]))
        {
            accel_read(&x, &y, &z);
            x = -x;
            y = -y;

            if ((x != eyes->pointer_last_x[i]) || (y != eyes->pointer_last_y[i]))
            {

                calculate_pupil_xy (eyes, x, y, &pupil_x, &pupil_y, eyes->eyes[i]);
                draw_eye (eyes, i, pupil_x, pupil_y);

                eyes->pointer_last_x[i] = x;
                eyes->pointer_last_y[i] = y;
            }
        }
    }

    return TRUE;
}



static void
properties_load(EyesPluginContent *eyes)
{
    gchar *path;

    if (eyes->active_theme)
        path = g_build_filename(THEMESDIR, eyes->active_theme, NULL);
    else
        path = g_build_filename(THEMESDIR, DEFAULTTHEME, NULL);

    load_theme(eyes, path);

    g_free(path);
}



static void
setup_eyes(EyesPluginContent *eyes)
{
    g_warning ("setup_eyes");
    int i;

    if (eyes->hbox != NULL)
    {
        gtk_widget_destroy(eyes->hbox);
        eyes->hbox = NULL;
    }

    eyes->hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(eyes->align), GTK_WIDGET(eyes->hbox));

    eyes->eyes = g_new0 (GtkWidget *, eyes->num_eyes);
	eyes->pointer_last_x = g_new0 (gint, eyes->num_eyes);
	eyes->pointer_last_y = g_new0 (gint, eyes->num_eyes);

    for (i = 0; i < eyes->num_eyes; i++)
    {
        eyes->eyes[i] = gtk_image_new ();

        gtk_widget_set_size_request(GTK_WIDGET(eyes->eyes[i]),
                                    eyes->eye_width,
                                    eyes->eye_height);

        gtk_widget_show(eyes->eyes[i]);

        gtk_box_pack_start(GTK_BOX(eyes->hbox), eyes->eyes[i],
                           FALSE, FALSE, 0);

		if ((eyes->num_eyes != 1) && (i == 0))
            gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 1.0, 0.5);
		else if ((eyes->num_eyes != 1) && (i == eyes->num_eyes - 1))
			gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 0.0, 0.5);
		else
			gtk_misc_set_alignment (GTK_MISC (eyes->eyes[i]), 0.5, 0.5);

		eyes->pointer_last_x[i] = G_MAXINT;
		eyes->pointer_last_y[i] = G_MAXINT;

		draw_eye (eyes, i,
			      eyes->eye_width / 2,
                  eyes->eye_height / 2);
    }

    gtk_widget_show(eyes->hbox);
}



static gboolean
eyes_applet_fill(EyesPluginContent *eyes)
{
    gtk_widget_show_all(GTK_WIDGET(eyes->align));

    if (eyes->timeout_id == 0)
    {
        eyes->timeout_id = g_timeout_add (UPDATE_TIMEOUT,
                                          (GtkFunction)timer_cb, eyes);
    }

    return TRUE;
}




/*
static void
combobox_changed (GtkComboBox    *combobox,
                  EyesPluginContent     *eyes)
{
	gchar *selected = gtk_combo_box_get_active_text (combobox);

	if (eyes->active_theme)
		g_free (eyes->active_theme);

	eyes->active_theme = g_strdup (selected);
	g_free (selected);

	properties_load(eyes);
    setup_eyes(eyes);
    eyes_applet_fill(eyes);
}
*/

/*
static void
eyes_properties_dialog (XfcePanelPlugin *plugin,
                        EyesPluginContent      *eyes)
{
	GtkWidget   *dlg, *hbox, *label, *combobox;
	GDir        *dir;
	gint         i;
	gchar       *current;
	const gchar *entry;

	xfce_panel_plugin_block_menu (plugin);

	dlg = xfce_titled_dialog_new_with_buttons (_("Eyes"),
                                                  GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
                                                  GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
                                                  GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
                                                  NULL);

    gtk_window_set_position   (GTK_WINDOW (dlg), GTK_WIN_POS_CENTER);
    gtk_window_set_icon_name  (GTK_WINDOW (dlg), "xfce4-settings");

    g_signal_connect (dlg, "response", G_CALLBACK (eyes_properties_dialog_response),
                      eyes);

	hbox = gtk_hbox_new(FALSE, 6);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

	label = gtk_label_new_with_mnemonic (_("_Select a theme:"));
	gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);

	combobox = gtk_combo_box_new_text ();
    gtk_box_pack_start (GTK_BOX (hbox), combobox, FALSE, TRUE, 0);

    if (eyes->active_theme)
		current = g_strdup (eyes->active_theme);
	else
		current = g_strdup (DEFAULTTHEME);

    if ((dir = g_dir_open(THEMESDIR, 0, NULL)) == NULL)
    {
        gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), current);
    }
    else
    {
        for (i = 0; (entry = g_dir_read_name(dir)) != NULL; i++)
        {
            gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), entry);

            if (strcmp (entry, current) == 0)
				gtk_combo_box_set_active (GTK_COMBO_BOX(combobox), i);
		}

        g_dir_close(dir);
	}

	g_free (current);

	gtk_label_set_mnemonic_widget (GTK_LABEL (label), combobox);

    g_signal_connect(G_OBJECT(combobox), "changed",
            G_CALLBACK(combobox_changed), eyes);

    gtk_widget_show_all (dlg);
}
*/


/******************************
 *** Panel Plugin Functions ***
 ******************************/
static void
eyes_free_data(EyesPluginContent      *eyes)
{
    g_warning("eyes_free_data");
    g_return_if_fail(eyes != NULL);

    if (eyes->timeout_id != 0)
        g_source_remove (eyes->timeout_id);

    g_free (eyes->eyes);
	g_free (eyes->pointer_last_x);
	g_free (eyes->pointer_last_y);

	if (eyes->active_theme != NULL)
		g_free (eyes->active_theme);

    if (eyes->eye_image != NULL)
        g_object_unref (G_OBJECT (eyes->eye_image));

    if (eyes->pupil_image != NULL)
        g_object_unref (G_OBJECT (eyes->pupil_image));

    if (eyes->theme_dir != NULL)
        g_free(eyes->theme_dir);

    if (eyes->theme_name != NULL)
        g_free(eyes->theme_name);

    if (eyes->eye_filename != NULL)
        g_free(eyes->eye_filename);

    if (eyes->pupil_filename != NULL)
        g_free(eyes->pupil_filename);

    g_free(eyes);
}

static void
eyes_read_rc_file (EyesPluginContent      *eyes)
{
    g_warning ("eyes_read_rc");
/*    XfceRc      *rc;
    gchar       *file;
    gchar const *tmp;

    if (eyes->active_theme != NULL)
    {
        g_free (eyes->active_theme);
        eyes->active_theme = NULL;
    }

    if ((file = xfce_panel_plugin_lookup_rc_file (plugin)) != NULL)
    {
        rc = xfce_rc_simple_open (file, TRUE);
        g_free (file);

        if (rc != NULL)
        {
            tmp = xfce_rc_read_entry (rc, "theme", DEFAULTTHEME);

            if (tmp != NULL)
                eyes->active_theme = g_strdup (tmp);

            xfce_rc_close (rc);
        }
    }
*/
    if (eyes->active_theme == NULL)
		eyes->active_theme = g_strdup (DEFAULTTHEME);
}

/*
static void
eyes_write_rc_file (XfcePanelPlugin *plugin,
                    EyesPluginContent      *eyes)
{
    gchar  *file;
    XfceRc *rc;

    if (!(file = xfce_panel_plugin_save_location (plugin, TRUE)))
        return;

    rc = xfce_rc_simple_open (file, FALSE);
    g_free (file);

    if (!rc)
        return;

    if (eyes->active_theme != NULL)
        xfce_rc_write_entry (rc, "theme", eyes->active_theme);

    xfce_rc_close (rc);
}
*/

static EyesPluginContent *
eyes_plugin_new ()
{
    g_warning ("eyes_plugin_new");
    EyesPluginContent *eyes;

    eyes = g_new0(EyesPluginContent, 1);

    eyes->align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);

    gtk_widget_show(GTK_WIDGET(eyes->align));

    eyes_read_rc_file (eyes);

    properties_load(eyes);
    setup_eyes(eyes);
    eyes_applet_fill(eyes);

    return eyes;
}

static void eyes_check_desktop (GObject *gobject, GParamSpec *pspec, EyesPluginContent *eyes)
{
	g_warning ("eyes_check_desktop");
	gchar *name = pspec->name;
	gboolean status;
	g_object_get (gobject, name, &status, NULL);
	if (status) {
	    eyes_applet_fill(eyes);
	} else if (eyes->timeout_id != 0) {
        g_source_remove (eyes->timeout_id);
		eyes->timeout_id = 0;
	}
}

static void eyes_plugin_dispose (GObject *object)
{
	g_warning ("eyes_plugin_dispose");

	G_OBJECT_CLASS (eyes_plugin_parent_class)->dispose (object);
}

static void eyes_plugin_finalize (GObject *object)
{
	g_warning ("eyes_plugin_finalize");
	EyesPlugin *self = EYES_HOME_PLUGIN (object);

    eyes_free_data(self->priv);

	G_OBJECT_CLASS (eyes_plugin_parent_class)->finalize (object);
}

static void
eyes_plugin_realize (GtkWidget *widget)
{
    g_warning ("eyes_plugin_realize");
    GdkScreen *screen = gtk_widget_get_screen (widget);
    gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));
    gtk_widget_set_app_paintable (widget, TRUE);

    GTK_WIDGET_CLASS (eyes_plugin_parent_class)->realize (widget);
}

static gboolean
eyes_plugin_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
    g_warning ("eyes_plugin_expose_event");
    cairo_t *cr;

    cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
    gdk_cairo_region (cr, event->region);
    cairo_clip (cr);

    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
    cairo_paint (cr);
  
    cairo_destroy (cr);

    return GTK_WIDGET_CLASS (eyes_plugin_parent_class)->expose_event (widget, event);
}

static void
eyes_plugin_init (EyesPlugin *desktop_plugin)
{
    g_warning ("eyes_plugin_init");
    EyesPluginContent *eyes;

    eyes = eyes_plugin_new (desktop_plugin);
    desktop_plugin->priv = eyes;

    g_signal_connect (desktop_plugin, "notify::is-on-current-desktop",
        G_CALLBACK (eyes_check_desktop), eyes);

    gtk_container_add (GTK_CONTAINER (desktop_plugin), eyes->align);
} 

static void
eyes_plugin_class_init (EyesPluginClass *klass) {
    g_warning ("eyes_plugin_class_init");
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->dispose = eyes_plugin_dispose;
	object_class->finalize = eyes_plugin_finalize;

    widget_class->realize = eyes_plugin_realize;
    widget_class->expose_event = eyes_plugin_expose_event;
    g_type_class_add_private (klass, sizeof (EyesPluginContent));
} 

static void
eyes_plugin_class_finalize (EyesPluginClass *class) {} 

