// Copyright (C) 2010 Jaakko Kyro <jkyro@korjaussarja.net>
// This file is licenced under GPL, see COPYING
// for full licence information
#include "eveskilltraining.h"
#include "evemodel.h"
#include "eveaccount.h"

#include <QXmlStreamReader>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QtDebug>
EveSkillTraining::EveSkillTraining(QObject *parent) :
    QObject(parent),
    training(false),
    m_character(NULL),
    m_account(NULL),
    m_reply(NULL),
    m_loading(false)
{
}

bool EveSkillTraining::fromXml(QXmlStreamReader &xml)
{
    xml.readNextStartElement();
    if (xml.name() != "eveapi")
        return false;
    xml.readNextStartElement();
    if (xml.name() != "currentTime") {
        return false;
    }

    currentTime = QDateTime::fromString(xml.readElementText(),"yyyy-MM-dd hh:mm:ss");
    currentTime.setTimeSpec(Qt::UTC);
    //xml.readNextStartElement(); // end currentTime element
    bool inResult = xml.readNextStartElement(); // start result element
    if (xml.name() != "result") {
        qDebug() << "Wrong element:" << xml.name();
        return false;
    }
    inResult = xml.readNextStartElement();
    if (!inResult || xml.name() != "rowset" ) {
        qDebug() << "No rowset element";
        return false;
    }
    inResult = xml.readNextStartElement();
    if (xml.name() == "row") {
        training = true;
        m_trainingQueue.clear();
    }
    while (xml.name() == "row") {

        TrainingEntry entry;
        entry.typeId = xml.attributes().value("","typeID").toString().toInt();
        entry.destSkillpoints = xml.attributes().value("","endSP").toString().toInt();
        entry.startSkillpoints = xml.attributes().value("","startSP").toString().toInt();
        entry.level = xml.attributes().value("","level").toString().toInt();
        if (xml.attributes().hasAttribute("","startTime")) {
            entry.startTime = QDateTime::fromString(xml.attributes().value("","startTime").toString(),
                                                    "yyyy-MM-dd hh:mm:ss");
            entry.startTime.setTimeSpec(Qt::UTC);
        }
        if (xml.attributes().hasAttribute("","endTime")) {
            entry.endTime = QDateTime::fromString(xml.attributes().value("","endTime").toString(),
                                                  "yyyy-MM-dd hh:mm:ss");
            entry.endTime.setTimeSpec(Qt::UTC);
        }
        qDebug() << entry;
        m_trainingQueue.append(entry);
        if (!xml.readNextStartElement() ) // end row element
            xml.readNextStartElement(); // start next row element
    }
    xml.readNextStartElement(); // end rowset
    xml.readNextStartElement(); // end result

    if (xml.name() == "cachedUntil" ) {
        QString cacheDate = xml.readElementText();
        qDebug() << "Cache date: " << cacheDate;
        cachedUntil = QDateTime::fromString(cacheDate,"yyyy-MM-dd hh:mm:ss");

        cachedUntil.setTimeSpec(Qt::UTC);
        qDebug() << "Cached until " << cachedUntil.toString(Qt::SystemLocaleShortDate);
    }
    qDebug() << "Parsing finished";
    qDebug() << firstTraining();
    return true;
}

void EveSkillTraining::fetchInfo()
{
    if (m_character == NULL || m_account == NULL)
        return;
    if (cachedUntil.isValid()) {
        if (cachedUntil > QDateTime::currentDateTime().toUTC() ) {
            qDebug() << "Cached until " << cachedUntil.toString(Qt::SystemLocaleShortDate);
            return;
        }
    }
    m_loading = true;
    QNetworkRequest req(QUrl(QString("http://api.eveonline.com/char/SkillQueue.xml.aspx?apiKey=%1&userID=%2&characterID=%3")
                             .arg(m_account->apiKey())
                             .arg(m_account->userId())
                             .arg(m_character->characterId)));
    m_reply = m_mgr.get(req);
    connect(m_reply,SIGNAL(finished()),this,SLOT(infoReady()));
}

void EveSkillTraining::infoReady()
{
    if (m_reply->error()) {
        qDebug() << "Failed! " << m_reply->errorString();
        return;
    }
    qDebug() << "Skill reply";
    QByteArray reply = m_reply->readAll();
    qDebug() << "Reply ready";
    qDebug() << reply;
    QXmlStreamReader reader(reply);
    fromXml(reader);
    m_reply->deleteLater();
    m_loading = false;
    emit finished();
}

QDebug operator<<(QDebug dbg, const TrainingEntry &e)
{
    dbg.nospace() << e.typeId << " to "
                << e.level << " "
                << e.startTime.toString(Qt::SystemLocaleShortDate) << " -> "
                << e.endTime.toString(Qt::SystemLocaleShortDate);
    return dbg.space();
};
