// Copyright (C) 2010 Jaakko Kyro <jkyro@korjaussarja.net>
// This file is licenced under GPL, see COPYING
// for full licence information
#include "widget.h"
#include <QPainter>
#include "eveaccount.h"
#include "evesettingsdialog.h"
#include "eveskilltraining.h"
#include "skilltree.h"
#include <QtDebug>
#include <QNetworkConfigurationManager>
#include <QSettings>
#include <QTimer>
#include <QDateTime>

QTM_USE_NAMESPACE

Widget::Widget(QWidget *parent)
    : QLabel(parent),
    m_settings(new EveSettingsDialog(this)),
    m_training(NULL),
    m_net(new QNetworkConfigurationManager(this)),
    m_skills(new SkillTree(this)),
    m_model(new EveModel),
    m_timer(new QTimer(this))
{
    setAlignment(Qt::AlignCenter);
    setAttribute(Qt::WA_TranslucentBackground);
    setGeometry(0,0,150,180);
    connect(m_net,SIGNAL(onlineStateChanged(bool)),this,SLOT(onlineStateChanged(bool)));
    connect(m_skills,SIGNAL(skillsLoaded()),this,SLOT(update()));
    m_skills->loadSkills();
    m_model->loadSettings();
    m_settings->setModel(m_model);
    EveCharacter *c = m_model->selectedCharacter();
    if (c != NULL) {
        loadTraining();
    }
    m_timer->setSingleShot(true);
    connect(m_timer,SIGNAL(timeout()),this,SLOT(loadTraining()));
}

Widget::~Widget()
{

}


QSize Widget::sizeHint() const
{
    return 2 * QLabel::sizeHint();
}

void Widget::paintEvent(QPaintEvent *event)
{
    QPainter p(this);
    p.setBrush(QColor(0, 0, 0, 128));
    p.setPen(Qt::NoPen);
    p.drawRoundedRect(rect(), 25, 25);
    p.setPen(Qt::white);
    QFont dFont(p.font());
    dFont.setPixelSize(15);
    p.setFont(dFont);
    EveCharacter *character = m_model->selectedCharacter();
    if (character != NULL) {
        QPoint iconLoc((150-64)/2,(150-64)/2);
        if (character->characterIcon) {
            p.drawPixmap(iconLoc,*(character->characterIcon));
        }
        QRect nameLoc(0,10,150,20);
        p.drawText(nameLoc,Qt::AlignCenter,character->name);
        if (m_training && m_training->training) {
            QRect skillTitle(0,110,150,50);
            p.drawText(skillTitle,Qt::AlignCenter|Qt::TextWordWrap,
                       QString("%1 %2").arg(m_skills->skillName(m_training->typeId))
                                       .arg(m_training->level));
            QRect skillLoc(0,155,150,20);
            p.drawText(skillLoc,Qt::AlignCenter,m_training->endTime.toString(Qt::SystemLocaleShortDate));
        } else {

            QRect skillTitle(0,110,150,50);
            p.drawText(skillTitle,Qt::AlignCenter|Qt::TextWordWrap,
                       "No skill in training");
        }

    }
    p.end();

    QLabel::paintEvent(event);
}

void Widget::showSettingsDialog()
{

    int result = m_settings->exec();
    if (result == QDialog::Accepted) {

        m_model->setSelected(m_settings->selectedCharacter());
        EveCharacter *character = m_model->selectedCharacter();
        if (character != NULL) {
            qDebug() << "Got character " << character->name;
            connect(character,SIGNAL(imageLoaded()),this,SLOT(skillReady()));
            character->fetchImage();

            loadTraining();
        }
    }
}

void Widget::loadTraining()
{
    EveCharacter *character = m_model->selectedCharacter();
    if (character != NULL) {

        qDebug() << "Fetch skills";
        if (m_training == NULL) {
            m_training = new EveSkillTraining(this);
            connect(m_training,SIGNAL(finished()),this,SLOT(trainingLoaded()));
        }
        m_training->setAccount(m_settings->model().data());
        m_training->setCharacter(character);

        m_training->fetchInfo();
    }
}

void Widget::skillReady()
{
    update();
    m_training->fetchInfo();
    m_model->saveSettings();
}

void Widget::onlineStateChanged(bool online)
{
    qDebug() << "Online status changed, reloading info";
    if (online) {
        m_training->fetchInfo();
    }
}

void Widget::trainingLoaded()
{
    qDebug() << "Training loaded";
    if (m_training != NULL && m_training->training) {
        QDateTime limit = (m_training->cachedUntil > m_training->endTime) ?
                          m_training->cachedUntil : m_training->endTime;
        qDebug() << "Next refresh:" << limit.toString(Qt::SystemLocaleShortDate);
        int interval = 1000 * (m_training->currentTime.secsTo(limit) + 15);
        //qDebug() << "Timer interval" << interval;
        if (m_timer->isActive()) {
            m_timer->stop();
        }
        m_timer->setInterval( interval );
        m_timer->start();
    }
    update();
}
