/**
 * METHODS to test
 *
 * StarDictInfo* loadDictionary(gchar* path);
 * char* getDefinition(StarDictInfo* starDictInfo, gchar* word);
 * void freeData(StarDictInfo* starDictInfo);
 * GArray* getWordList(StarDictInfo* starDictInfo, gchar* prefix, gint maxNumSuggestedWords);
 */
#include "../engine/estardict-engine.h"
#include "../engine/estardict-engine-parsing.h"

// from estardict-gui-logic.h
#define MAX_NUMBER_SUGGESTED_WORDS 2

#include <stdlib.h>
#include <time.h>

#undef G_LOG_DOMAIN
#define G_LOG_DOMAIN "estardict-engine-tests"

// Words
#define WORD_DOG "dog"
#define WORD_CANE "cane"
#define WORD_DOG_NOUN "dog noun"
#define WORD_DOG_NOUN_1 "dog  1, noun" // dog - definition with image
#define WORD_TIME_NOUN "time  1, noun" // long definition
#define WORD_NOT_EXISTING "zzzzz"
#define WORD_CIVILTA "civilta" //diacritics
#define WORD_CIVILTA_DIACRITICS "civiltà" //diacritics

#define WORD_LONGMAN_WORST "zygote"
#define WORD_BABYLON_EN_IT_WORST "zzz"
#define WORD_ENGLISH_CZECH_WORST "Zyrian"
#define WORD_LANGDAO_ENGLISH_CHINESE_WORST "Zz."
#define WORD_OXFORD_COLLOCATIONS_WORST "zoo noun"
#define WORD_QUICK_ENG_SPA_WORST "zucchini"
#define WORD_ENG_ITA_WORST "zzz"
#define WORD_SINONIMI_WORST "zuzzurellone"
#define WORD_BRITANNICA_CONCISE_WORST "zymogen"
#define WORD_LONGMAN_IMAGE_WORST "time  1, noun" // long definition

#define WORD_LONGMAN_WORST_LENGHT 141
#define WORD_BABYLON_EN_IT_WORST_LENGHT 59
#define WORD_ENGLISH_CZECH_WORST_LENGHT 62
#define WORD_LANGDAO_ENGLISH_CHINESE_WORST_LENGHT 13
#define WORD_OXFORD_COLLOCATIONS_WORST_LENGHT 207
#define WORD_QUICK_ENG_SPA_WORST_LENGHT 10
#define WORD_ENG_ITA_WORST_LENGHT 49
#define WORD_SINONIMI_WORST_LENGHT 104
#define WORD_BRITANNICA_CONCISE_WORST_LENGHT 784
#define WORD_LONGMAN_IMAGE_WORST_LENGHT 30876

// Dictionaries
#define LONGMAN "stardict-longman-2.4.2"
#define BABYLON_EN_IT "stardict-babylon-Babylon_English_Italian-2.4.2"
#define ENGLISH_CZECH "stardict-english-czech"
#define LANGDAO_ENGLISH_CHINESE "stardict-langdao-ec-gb-2.4.2"
#define OXFORD_COLLOCATIONS "stardict-oxford_collocations-2.4.2"
#define INGLESE_ITALIANO "Inglese-Italiano"
#define SINONIMI "Sinonimi"
#define BRITANNICA_CONCISE "stardict-BritannicaConcise-2.4.2"
#define ENGLISH_SPANISH "stardict-quick_eng-spa-2.4.2"
#define VOCABOLARIO_ITALIANO "stardict-vocabolario-italiano"
#define QUICK_ENG_SPA "stardict-quick_eng-spa-2.4.2"
#define ENG_ITA "Inglese-Italiano"
#define SINONIMI "Sinonimi"
#define VOCABOLARIO_ITA "Stardict vocabolario italiano"
#define BRITANNICA "stardict-BritannicaConcise-2.4.2"
#define OALD8 "En-En-OALD8"
#define LONGMAN_IMAGE "longman-img-dictionary"

// Lengh for word per dictionary
#define LONGMAN_DEFINITION_DOG_LENGHT 3184
#define BABYLON_EN_IT_DEFINITION_DOG_LENGHT 121
#define ENGLISH_CZECH_DEFINITION_DOG_LENGHT 456
#define LANGDAO_ENGLISH_CHINESE_DEFINITION_DOG_LENGHT 255
#define OXFORD_COLLOCATIONS_DEFINITION_DOG_NOUN_LENGHT 2149
#define QUICK_ENG_SPA_DEFINITION_DOG_LENGHT 5
#define ENG_ITA_DEFINITION_DOG_LENGHT 242
#define SINONIMI_DEFINITION_CANE_LENGHT 666
#define VOCABOLARIO_ITA_DEFINITION_CANE_LENGHT 2433
#define BRITANNICA_DEFINITION_DOG_LENGHT 825
#define OALD8_DEFINITION_DOG_LENGHT 16110
#define LONGMAN_IMAGE_DEFINITION_DOG_NOUN_1_LENGHT 4620
#define VOCABOLARIO_ITA_DEFINITION_CIVILTA_LENGHT 0
#define WORD_NOT_EXISTING_LENGHT 0

#define ARR_SIZE(x) (sizeof(x)/sizeof((x)[0]))
#define MAX_DICTIONARIES 10

clock_t startm, stopm;

/** Enumeration with use cases */
typedef enum {
    UC_M,
    UC_OXFORD,
    UC_PANGO,
    UC_QUICK_ESP,
    UC_LANGDAO,
    UC_HTML,
    UC_ENG_ITA,
    UC_SINONIMI,
    UC_VOCABOLARIO_ITA,
    UC_BRITANNICA,
    UC_OALD8,
    UC_LONGMAN_IMAGE,
    UC_VOCABOLARIO_ITA_DIACRITICS,
    UC_THREE_DICTS,
    UC_THREE_DICTS_WORSE_CASE,
    UC_FIVE_DICTS,
    UC_FIVE_DICTS_WORSE_CASE,
    UC_TEN_DICTS,
    UC_TEN_DICTS_WORSE_CASE,
    UC_UNDEFINED,
} EnumUseCase;

/** Menu item structure */
typedef struct DictionaryStruct {
    const char* dictPath;
    const char* word;
    int definitionDogLenght;
} Dictionary;

/** Signature of the test method */
typedef gboolean (*ItemMethodPtr)(void*);

/** Menu item structure */
typedef struct MenuItemStruct {
    EnumUseCase key;
    const char* name;
    ItemMethodPtr method;
    Dictionary dictionary[MAX_DICTIONARIES];
} MenuItem;

/** Test methods */
static gboolean testDict();

/** Use cases */
static const MenuItem tests[] = {
    {
        UC_M, 
        "Longman (samesequence m, 43.052 words)", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_OXFORD, 
        "Oxford English Collocation Dictionary (samesequence m, 8.338 words, Requires suggested words)", 
        testDict, 
        {
            {OXFORD_COLLOCATIONS, WORD_DOG_NOUN, OXFORD_COLLOCATIONS_DEFINITION_DOG_NOUN_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_PANGO, 
        "English-Czech (samesequence g, 86.072 words)", 
        testDict, 
        {
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_QUICK_ESP, 
        "Quick Eng-Spa (samesequence m, 20.737 words)", 
        testDict, 
        {
            {QUICK_ENG_SPA, WORD_DOG, QUICK_ENG_SPA_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_LANGDAO, 
        "Langdao English to Chinese (samesequence m, 435.468 words, Huge word index file)", 
        testDict, 
        {
            {LANGDAO_ENGLISH_CHINESE, WORD_DOG, LANGDAO_ENGLISH_CHINESE_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_HTML, 
        "Babylon English-Italian (samesequence h, 142.077 words, synonym)", 
        testDict, 
        {
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_ENG_ITA, 
        "English-Italian (samesequence m, 246.338 words)", 
        testDict, 
        {
            {ENG_ITA, WORD_DOG, ENG_ITA_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_SINONIMI, 
        "Sinonimi (samesequence m, 55.065 words, Uncompressed dict)", 
        testDict, 
        {
            {SINONIMI, WORD_CANE, SINONIMI_DEFINITION_CANE_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_VOCABOLARIO_ITA, 
        "Vocabolario italiano (samesequence m, 82.718 words)", 
        testDict, 
        {
            {VOCABOLARIO_ITA, WORD_CANE, VOCABOLARIO_ITA_DEFINITION_CANE_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_BRITANNICA, 
        "The Britannica Concise (samesequence m, 24.402 words)", 
        testDict, 
        {
            {BRITANNICA, WORD_DOG, BRITANNICA_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_OALD8, 
        "Oxford Advanced Learner's Dictionary, 8th Edition (samesequence , 65.685 words)", 
        testDict, 
        {
            {OALD8, WORD_DOG, OALD8_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_LONGMAN_IMAGE, 
        "Longman image dog (samesequence h, 47625 words)", 
        testDict, 
        {
            {LONGMAN_IMAGE, WORD_DOG_NOUN_1, LONGMAN_IMAGE_DEFINITION_DOG_NOUN_1_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_VOCABOLARIO_ITA_DIACRITICS, 
        "Vocabolario italiano diacritics (samesequence m, 82.718 words)", 
        testDict, 
        {
            {VOCABOLARIO_ITA, WORD_CIVILTA, VOCABOLARIO_ITA_DEFINITION_CIVILTA_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_THREE_DICTS, 
        "Three dictionaries (longman, Babylon English-Italian, English-Czech)", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_THREE_DICTS_WORSE_CASE, 
        "Three dictionaries worse case (longman, Babylon English-Italian, English-Czech)", 
        testDict, 
        {
            {LONGMAN, WORD_LONGMAN_WORST, WORD_LONGMAN_WORST_LENGHT},
            {BABYLON_EN_IT, WORD_BABYLON_EN_IT_WORST, WORD_BABYLON_EN_IT_WORST_LENGHT},
            {ENGLISH_CZECH, WORD_ENGLISH_CZECH_WORST, WORD_ENGLISH_CZECH_WORST_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_FIVE_DICTS, 
        "Five dictionaries (longman, Babylon English-Italian, English-Czech, Langdao english chinese, Oxford collocations )", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {LANGDAO_ENGLISH_CHINESE, WORD_DOG, LANGDAO_ENGLISH_CHINESE_DEFINITION_DOG_LENGHT},
            {OXFORD_COLLOCATIONS, WORD_DOG_NOUN, OXFORD_COLLOCATIONS_DEFINITION_DOG_NOUN_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_FIVE_DICTS_WORSE_CASE, 
        "Five dictionaries worse case (longman, Babylon English-Italian, English-Czech, Langdao english chinese, Oxford collocations )", 
        testDict, 
        {
            {LONGMAN, WORD_LONGMAN_WORST, WORD_LONGMAN_WORST_LENGHT},
            {BABYLON_EN_IT, WORD_BABYLON_EN_IT_WORST, WORD_BABYLON_EN_IT_WORST_LENGHT},
            {ENGLISH_CZECH, WORD_ENGLISH_CZECH_WORST, WORD_ENGLISH_CZECH_WORST_LENGHT},
            {LANGDAO_ENGLISH_CHINESE, WORD_LANGDAO_ENGLISH_CHINESE_WORST, WORD_LANGDAO_ENGLISH_CHINESE_WORST_LENGHT},
            {OXFORD_COLLOCATIONS, WORD_OXFORD_COLLOCATIONS_WORST, WORD_OXFORD_COLLOCATIONS_WORST_LENGHT},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
    {
        UC_TEN_DICTS, 
        "Ten dicts (Longman, Babylon, Eng-Czech, Langdao, Oxford, Quick, Eng-Ita, Sinonimi, Britannica, OALD8)", 
        testDict, 
        {
            {LONGMAN, WORD_DOG, LONGMAN_DEFINITION_DOG_LENGHT},
            {BABYLON_EN_IT, WORD_DOG, BABYLON_EN_IT_DEFINITION_DOG_LENGHT},
            {ENGLISH_CZECH, WORD_DOG, ENGLISH_CZECH_DEFINITION_DOG_LENGHT},
            {LANGDAO_ENGLISH_CHINESE, WORD_DOG, LANGDAO_ENGLISH_CHINESE_DEFINITION_DOG_LENGHT},
            {OXFORD_COLLOCATIONS, WORD_DOG_NOUN, OXFORD_COLLOCATIONS_DEFINITION_DOG_NOUN_LENGHT},
            {QUICK_ENG_SPA, WORD_DOG, QUICK_ENG_SPA_DEFINITION_DOG_LENGHT},
            {ENG_ITA, WORD_DOG, ENG_ITA_DEFINITION_DOG_LENGHT},
            {SINONIMI, WORD_CANE, SINONIMI_DEFINITION_CANE_LENGHT},
            {BRITANNICA, WORD_DOG, BRITANNICA_DEFINITION_DOG_LENGHT},
            {LONGMAN_IMAGE, WORD_DOG_NOUN_1, LONGMAN_IMAGE_DEFINITION_DOG_NOUN_1_LENGHT}
        }
    },
    {
        UC_TEN_DICTS_WORSE_CASE, 
        "Ten dicts worse case (Longman, Babylon, Eng-Czech, Langdao, Oxford, Quick, Eng-Ita, Sinonimi, Britannica, OALD8)", 
        testDict, 
        {
            {LONGMAN, WORD_LONGMAN_WORST, WORD_LONGMAN_WORST_LENGHT},
            {BABYLON_EN_IT, WORD_BABYLON_EN_IT_WORST, WORD_BABYLON_EN_IT_WORST_LENGHT},
            {ENGLISH_CZECH, WORD_ENGLISH_CZECH_WORST, WORD_ENGLISH_CZECH_WORST_LENGHT},
            {LANGDAO_ENGLISH_CHINESE, WORD_LANGDAO_ENGLISH_CHINESE_WORST, WORD_LANGDAO_ENGLISH_CHINESE_WORST_LENGHT},
            {OXFORD_COLLOCATIONS, WORD_OXFORD_COLLOCATIONS_WORST, WORD_OXFORD_COLLOCATIONS_WORST_LENGHT},
            {QUICK_ENG_SPA, WORD_QUICK_ENG_SPA_WORST, WORD_QUICK_ENG_SPA_WORST_LENGHT},
            {ENG_ITA, WORD_ENG_ITA_WORST, WORD_ENG_ITA_WORST_LENGHT},
            {SINONIMI, WORD_SINONIMI_WORST, WORD_SINONIMI_WORST_LENGHT},
            {BRITANNICA, WORD_BRITANNICA_CONCISE_WORST, WORD_BRITANNICA_CONCISE_WORST_LENGHT},
            {LONGMAN_IMAGE, WORD_LONGMAN_IMAGE_WORST, WORD_LONGMAN_IMAGE_WORST_LENGHT}
        }
    },
    {
        UC_UNDEFINED, 
        "Invalid test case", 
        NULL, 
        {
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0},
            {NULL, NULL, 0}
        }
    },
};

/** Dictionary path */
const gchar * path;

/** Selected use case */
MenuItem menuItem;


/**
 * Start for calculate the time execution
 */
static void start() {
    startm = clock();
    if ( startm == -1) {
        g_error("Error calling clock\n");
    }
}


/**
 * Stop for calculate the time execution
 */
static void stop() {
    stopm = clock();
    if ( stopm == -1) {
        g_error("Error calling clock\n");
    }
}


/**
 * Print execution time
 */
static void printTimeExecution() {
    g_message( " in %6.3f seconds\n", ((double)stopm-startm)/CLOCKS_PER_SEC);
}


/**
 * Append char to string
 */
void append(gchar* s, gchar c) {
        int len = strlen(s);
        s[len] = c;
        s[len+1] = '\0';
}


/**
 * Test the given dictionary
 * 
 * @return TRUE success, FALSE failed
 */
static gboolean testDict() {
    g_debug("-> %s %s()\n", __FILE__, __FUNCTION__);

    gboolean result = FALSE;
    gchar* definition = NULL;
    gchar* absDictPath = NULL;
    gchar* inputWord = NULL;
    gchar* wordToSearch = NULL;
    gchar* word = NULL;
    StarDictInfo* starDictInfo[MAX_DICTIONARIES];
    GArray* list = NULL;
    guint i = 0;
    guint j = 0;
    guint z = 0;
    guint numDict = 0;
    guint wordLenght = 0;
    guint definitionLenght = 0;
    
    for ( i = 0; i < MAX_DICTIONARIES; i++) {
        starDictInfo[i] = NULL;
    }
    
    if ( path != NULL && menuItem.dictionary[0].word != NULL ) {
        
        // Load dictionaries
        start();
        for ( i = 0; i < MAX_DICTIONARIES; i++) {
            if ( menuItem.dictionary[i].dictPath != NULL ) {
                absDictPath = g_strconcat(path, "/", menuItem.dictionary[i].dictPath, NULL);
                starDictInfo[i] = loadDictionary(absDictPath);
            } else {
                break;
            }
        }
        stop();
        
        numDict = i;
        g_message("Loaded %d dictionary/ies", numDict);
        printTimeExecution();
        
        // Show suggested words
        wordLenght = strlen(menuItem.dictionary[0].word);
        inputWord = (gchar *)malloc( wordLenght + 1 );
        inputWord[0] = '\0';
        
        for ( j = 0; j < wordLenght; j++) {
            g_debug("\n");
            start();
            append(inputWord, menuItem.dictionary[0].word[j]);
            g_message("Wordlist for %s", inputWord);
            g_debug("\n");
            for ( i = 0; i < numDict; i++) {
                if (starDictInfo[i] != NULL) {
                    list = NULL;

                    list = getWordList(
                        starDictInfo[i], 
                        inputWord, 
                        MAX_NUMBER_SUGGESTED_WORDS
                    );

                    for ( z = 0; z < list->len; z++) {
                        word = g_array_index(list, gchar*, z);
                        g_debug("Current item is %s\n", word);
                        
                        // Diacritics
                        if (
                            ( strcmp(menuItem.dictionary[i].dictPath, VOCABOLARIO_ITA) == 0 && strcmp(inputWord, WORD_CIVILTA) == 0 )
                        ) {
                            if ( strcmp(word, WORD_CIVILTA_DIACRITICS) == 0 ) {
                                g_message("\nDIACRITICS OK\n");
                            } else {
                                g_message("\nDIACRITICS FAILED\n");
                            }
                        }
                        
                    }
                    g_array_free(list, TRUE);
                }
            }
            stop();
            printTimeExecution();
        }
        
        
        
        g_free(inputWord);
        
        // Get definition
        start();
        for ( i = 0; i < numDict; i++) {
            if (starDictInfo[i] != NULL) {
                wordToSearch = (gchar*)menuItem.dictionary[i].word;
                definition = getDefinition(starDictInfo[i], wordToSearch);
                
                g_debug("Dictionary %s\n", menuItem.dictionary[i].dictPath);
                g_debug("type %s\n", starDictInfo[i]->sametypesequence);
                
                if ( definition != NULL ) {
                    
                    // Parse the definition using the stardict encoding value (for example h means html)
                    definition = parseDefinition(starDictInfo[i], definition);
                    
                    definitionLenght = strlen(definition);
                    
                    // Image dictionary remove the lenght of the relative path (ONE image)
                    // the dictionary path lenght can change, it is given as first parameter from the command line
                    if ( 
                        ( strcmp(menuItem.dictionary[i].dictPath, LONGMAN_IMAGE) == 0 && strcmp(wordToSearch, WORD_DOG_NOUN_1) == 0 )
                        || ( strcmp(menuItem.dictionary[i].dictPath, OALD8) == 0 && strcmp(wordToSearch, WORD_DOG) == 0 )
                    ) {
                        definitionLenght -= strlen(path);
                    }
                    
                    g_debug("definition for %s: %s\n", menuItem.dictionary[i].word, definition);
                    g_debug("lenght definition for %s: expected %d real %d\n", menuItem.dictionary[i].word, menuItem.dictionary[i].definitionDogLenght, definitionLenght);

                    if (
                        definitionLenght == menuItem.dictionary[i].definitionDogLenght
                    ) {
                        g_debug("\t OK\n");
                        result = TRUE;
                        
                    // ERROR    
                    } else {
                        g_message("\t FAILED - Definition\n");
                    }
                    
                } else {
                    g_message("No definition\n");
                    
                    // It's correct, you can have no definition for some dictionary
                    result = TRUE;
                }

            } else {
                g_message("\t FAILED - Load stardict\n");
            }
        }
        stop();
        
        g_message("Get definitions in %d dictionary/iesn", numDict);
        printTimeExecution();
        
        // Free
        for ( i = 0; i < numDict; i++) {
            freeData(starDictInfo[i]);
        }
        
    } else {
        g_message("\t FAILED - Path or word to search not defined\n");
    }
    return result;
}

void logMessage(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {

    // Enable message
    if ( log_domain != NULL && strcmp(log_domain, G_LOG_DOMAIN) == 0 && log_level == G_LOG_LEVEL_MESSAGE ) {
        printf("%s", message);
    }

    // Enable debug message
    if ( log_domain != NULL && strcmp(log_domain, G_LOG_DOMAIN) == 0 && log_level == G_LOG_LEVEL_DEBUG ) {
        //printf("%s", message);
    }
}

void disableMessage(const gchar *log_domain, GLogLevelFlags log_level, const gchar *message, gpointer user_data) {
}


/**
 * Print welcome screen with all the use cases
 */
void printUsage(void) {
    g_message("estardict-engine-tests <dictionary_path> <use_case>\n");
    g_message("\t where\n");
    g_message("<path with dictionaries> contains test dictionaries\n");
    g_message("<use_case> could be:\n");
    int i;
    
    for ( i = 0; i < ARR_SIZE(tests); i++ ) {
        g_message("\t %d : %s\n", (int)tests[i].key, tests[i].name );
    }
}

/**
 * Lookup use case from key value
 * 
 * @param key Use case key
 * @return Use case menu item
 */
MenuItem lookup(EnumUseCase key) {
    int i;
    MenuItem result;
    
    for ( i = 0; i < ARR_SIZE(tests); i++ ) {
        if ( tests[i].key == key ) {
            result = tests[i];
            break;
        }
    }
    
    return result;
}


/**
 * Main function
 */
int main(int argc, char **argv) {
    int i = 0;
    gboolean result = FALSE;
    
    g_log_set_handler(G_LOG_DOMAIN, G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_DEBUG, logMessage, NULL);
    g_log_set_handler(NULL, G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_DEBUG, disableMessage, NULL);
    
    /* argc 0 command name
     * argc 1 dictionary path
     * argc 2 test case
     */
    if ( argc <= 2 ) {
        printUsage();
        return 0;
    }
    path = argv[1];

    for ( i = 2; i < argc; i++ ) {
        int index = atoi( argv[i] );
        
        if ( index < ARR_SIZE( tests) ) {
            menuItem = lookup(index);
            g_message("\nTesting: %s\n", menuItem.name);
            
            ItemMethodPtr itemMethod = menuItem.method;
            if ( itemMethod ) {
                result = itemMethod(NULL);
                if ( result ) {
                    g_message("success");
                } else {
                    g_message("failed");
                }
            } else {
                g_message("Method not implemented!");
            }
        } else {
            g_message("Use case not defined!");
            printUsage();
        }
    }
    
    return 0;
}
