/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-engine.c
  * @author Luca Vaudano
  */
#include "estardict-engine.h"
#include "estardict-engine-strip.h"

// IFO
const guint IFO_FILE_MAX_LENGHT_LINE = 100;
const gchar* IFO_FILE_FIRST_LINE = "StarDict's dict ifo file";
const gchar* IFO_FILE_SECOND_LINE_OPT1 = "version=2.4.2";
const gchar* IFO_FILE_SECOND_LINE_OPT2 = "version=3.0.0";
const gchar* IFO_FILE_BOOKNAME = "bookname"; // (required)
const gchar* IFO_FILE_WORDCOUNT = "wordcount"; // (required)
const gchar* IFO_FILE_IDXFILESIZE = "idxfilesize"; // (required)
const gchar* IFO_FILE_IDXOFFSETBITS = "idxoffsetbits";
const gchar* IFO_FILE_SAMETYPESEQUENCE = "sametypesequence";


/**
 * @brief Parse ifo file.
 * Consider only interesting parameters:
 * <ul>
 *   <li>bookname
 *   <li>wordcount
 *   <li>idxfilesize
 *   <li>idxoffsetbits
 *   <li>sametypesequence
 * </ul>
 *
 * @param starDictInfo StarDict data struct
 * @return True if the requirements are satisfied, otherwise false
 */
gboolean parseIfoFile(StarDictInfo* starDictInfo) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    gboolean result = FALSE;
    FILE *ifoFile;
    gchar buffer[IFO_FILE_MAX_LENGHT_LINE];

    // Open the file for reading
    ifoFile = g_fopen(starDictInfo->ifoFileName, "r");

    // If we can open the file
    if (ifoFile) {

        // First line must be equals to IFO_FILE_FIRST_LINE
        if (
            fgets(buffer, IFO_FILE_MAX_LENGHT_LINE, ifoFile) == NULL ||
            g_ascii_strncasecmp(buffer, IFO_FILE_FIRST_LINE, strlen(IFO_FILE_FIRST_LINE)) != 0
        ) {
            g_critical("--->First line of the IFO file must be %s", IFO_FILE_FIRST_LINE);
            estardictEngineError = ESTARDICT_ENGINE_ERROR_FIRST_LINE_IFO;
            return result;
        }

        // Second line must be equals to IFO_FILE_SECOND_LINE_OPT1 or IFO_FILE_SECOND_LINE_OPT2
        if (
            fgets(buffer, IFO_FILE_MAX_LENGHT_LINE, ifoFile) == NULL ||
            (g_ascii_strncasecmp(buffer, IFO_FILE_SECOND_LINE_OPT1, strlen(IFO_FILE_SECOND_LINE_OPT1)) != 0 &&
            g_ascii_strncasecmp(buffer, IFO_FILE_SECOND_LINE_OPT2, strlen(IFO_FILE_SECOND_LINE_OPT2)) != 0)
        ) {
            g_critical("--->Second line of the IFO file must be %s or %s", IFO_FILE_SECOND_LINE_OPT1, IFO_FILE_SECOND_LINE_OPT2);
            estardictEngineError = ESTARDICT_ENGINE_ERROR_SECOND_LINE_IFO;
            return result;
        }

        /* Read the other optional lines
         * Consider only interesting parameters: bookname (required), wordcount (required),
         * idxfilesize (required), idxoffsetbits, sametypesequence
         */
        while(fgets(buffer, IFO_FILE_MAX_LENGHT_LINE, ifoFile) != NULL) {
            gchar* value = NULL;
            g_debug("--> %s", buffer);
            if (g_ascii_strncasecmp(buffer, IFO_FILE_BOOKNAME, strlen(IFO_FILE_BOOKNAME)) == 0) {
                value = g_strstr_len(buffer, IFO_FILE_MAX_LENGHT_LINE, "=");
                value = value + 1; // skip =
                char* bookname = NULL;
                
                if ( value[strlen(value) - 1] == '\n' ) {
                    // Windows OS used \r\n, unix only n
                    if ( value[strlen(value) - 2] == '\r' ) {
                        bookname = g_strndup(value, strlen(value) - 2);
                    } else {
                        bookname = g_strndup(value, strlen(value) - 1);
                    }
                } else {
                    bookname = g_strdup(value);
                }

                starDictInfo->bookname = bookname;
                g_message("--->Bookname: %s", starDictInfo->bookname);
            }
            if (g_ascii_strncasecmp(buffer, IFO_FILE_WORDCOUNT, strlen(IFO_FILE_WORDCOUNT)) == 0) {
                value = g_strstr_len(buffer, IFO_FILE_MAX_LENGHT_LINE, "=");
                value = value + 1; // skip =
                starDictInfo->wordcount = (glong)g_ascii_strtoull(value, NULL, 10);;
                g_message("--->Wordcount: %ld", starDictInfo->wordcount);
            }
            if (g_ascii_strncasecmp(buffer, IFO_FILE_IDXFILESIZE, strlen(IFO_FILE_IDXFILESIZE)) == 0) {
                value = g_strstr_len(buffer, IFO_FILE_MAX_LENGHT_LINE, "=");
                value = value + 1; // skip =
                starDictInfo->idxfilesize = (glong)g_ascii_strtoull(value, NULL, 10);
                g_message("--->Idxfilesize: %ld", starDictInfo->idxfilesize);
            }
            if (g_ascii_strncasecmp(buffer, IFO_FILE_IDXOFFSETBITS, strlen(IFO_FILE_IDXOFFSETBITS)) == 0) {
                value = g_strstr_len(buffer, IFO_FILE_MAX_LENGHT_LINE, "=");
                value = value + 1; // skip =
                starDictInfo->idxoffsetbits = (glong)g_ascii_strtoull(value, NULL, 10);
                g_message("--->Idxoffsetbits: %ld", starDictInfo->idxoffsetbits);
            }
            if (g_ascii_strncasecmp(buffer, IFO_FILE_SAMETYPESEQUENCE, strlen(IFO_FILE_SAMETYPESEQUENCE)) == 0) {
                value = g_strstr_len(buffer, IFO_FILE_MAX_LENGHT_LINE, "=");
                value = value + 1; // skip =
                char* sametypesequence = NULL;

                if ( value[strlen(value) - 1] == '\n' ) {
                    // Windows OS used \r\n, unix only n
                    if ( value[strlen(value) - 2] == '\r' ) {
                        sametypesequence = g_strndup(value, strlen(value) - 2);
                    } else {
                        sametypesequence = g_strndup(value, strlen(value) - 1);
                    }
                } else {
                    sametypesequence = g_strdup(value);
                }

                starDictInfo->sametypesequence = sametypesequence;
                g_message("--->sametypesequence: %s", starDictInfo->sametypesequence);
            }
        }

        g_debug("--> Close ifo file");
        fclose(ifoFile);
        result = TRUE;
    }

    return result;
}


/**
 * @brief Find StarDict files.
 * I don't care about syn file.
 *
 * @param starDictInfo StarDict data struct
 * @return True if the requirements are satisfied, otherwise false
 */
gboolean findFiles(StarDictInfo* starDictInfo) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    GError *error = NULL;
    int flag = 0;
    gchar* path = starDictInfo->path;

    GDir *dir = g_dir_open(path, flag, &error);

    if (!dir) {
        g_critical("---> Cannot open the directory %s", path);
        estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_DIRECTORY;
        return FALSE;
    }

    // Initialize boolean for mandatory files checking
    gboolean ifo = FALSE;
    gboolean idx = FALSE;
    gboolean dict = FALSE;
    gboolean check = FALSE;

    // Array with all the files in this directory
    const gchar* file = g_dir_read_name(dir);

    // Loop over all the files in the giver directory
    while (file != NULL) {
        if (g_str_has_suffix(file, ".ifo")) {
            starDictInfo->ifoFileName = g_strconcat(path,"/",file,NULL);
            g_debug("--->Found ifo: %s", file);
            ifo = TRUE;
        }
        if (g_str_has_suffix(file, ".idx")) {
            starDictInfo->idxFileName = g_strconcat(path,"/",file,NULL);
            starDictInfo->idxCompressed = FALSE;
            g_debug("--->Found idx: %s", file);
            idx = TRUE;
        }
        if (g_str_has_suffix(file, ".idx.gz") && idx == FALSE) {
            starDictInfo->idxFileName = g_strconcat(path,"/",file,NULL);
            starDictInfo->idxCompressed = TRUE;
            g_debug("--->Found idx.gz: %s", file);
            idx = TRUE;
        }
        if (g_str_has_suffix(file, ".dict")) {
            starDictInfo->dictFileName = g_strconcat(path,"/",file,NULL);
            starDictInfo->dictCompressed = FALSE;
            g_debug("--->Found dict: %s", file);
            dict = TRUE;
        }
        if (g_str_has_suffix(file, ".dict.dz")) {
            starDictInfo->dictFileName = g_strconcat(path,"/",file,NULL);
            starDictInfo->dictCompressed = TRUE;
            g_debug("--->Found dict: %s", file);
            dict = TRUE;
        }
        if (ifo && idx && dict) {
            check = TRUE;
            break;
        }
        file = g_dir_read_name(dir);
    }
    g_dir_close(dir);
    return check;
}


/**
 * @brief Free StarDictInfo struct data.
 *
 * @param starDictInfo StarDictInfo struct data to free
 */
void freeData(StarDictInfo* starDictInfo) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    if (starDictInfo == NULL) {
        return;
    }
    if (starDictInfo->path != NULL) {
        g_free(starDictInfo->path);
        starDictInfo->path = NULL;
    }
    if (starDictInfo->ifoFileName != NULL) {
        g_free(starDictInfo->ifoFileName);
        starDictInfo->ifoFileName = NULL;
    }
    if (starDictInfo->idxFileName != NULL) {
        g_free(starDictInfo->idxFileName);
        starDictInfo->idxFileName = NULL;
    }
    if (starDictInfo->dictFileName != NULL) {
        g_free(starDictInfo->dictFileName);
        starDictInfo->dictFileName = NULL;
    }
    if (starDictInfo->bookname != NULL) {
        g_free(starDictInfo->bookname);
        starDictInfo->bookname = NULL;
    }
    if (starDictInfo->sametypesequence != NULL) {
        g_free(starDictInfo->sametypesequence);
        starDictInfo->sametypesequence = NULL;
    }
    if (starDictInfo->warnings != NULL) {
        g_list_free(starDictInfo->warnings);
        starDictInfo->warnings = NULL;
    }
    if (starDictInfo->idxFile != NULL) {
        close(*((gint*)(starDictInfo->idxFile)));
        starDictInfo->idxFile = NULL;
    }
    if (starDictInfo->dictFile != NULL) {
        gzclose(starDictInfo->dictFile);
        starDictInfo->dictFile = NULL;
    }
    g_free(starDictInfo);
    starDictInfo = NULL;
}


/**
 * @brief Load idx file.
 * The idx file is a sorted word list
 * NOTE: Two or more entries may have the same 'word' parameter with
 * different offset and lenght.
 *
 * @param starDictInfo StardDict struct
 * @return True succeffully operation, false otherwise
 */
gboolean loadIdx(StarDictInfo* starDictInfo) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    // Open the file for reading
    starDictInfo->idxFile = g_fopen(starDictInfo->idxFileName, "r");

    // If we can open the file
    if (starDictInfo->idxFile) {
        g_debug("--->idx file opened");
        fseek(starDictInfo->idxFile, 0, SEEK_END); // seek to end of file
        gint size = ftell(starDictInfo->idxFile); // get current file pointer
        fseek(starDictInfo->idxFile, 0, SEEK_SET); // seek back to beginning of file
        g_debug("--->idx size: %d", size);

        // cross check file size with ifo info
        if (size == starDictInfo->idxfilesize) {
            g_message("-->Cross check file size with ifo information OK");
        } else {
            g_warning("--->The idx file size doesn't mach with the ifo information.");
            starDictInfo->warnings = g_list_prepend(starDictInfo->warnings, GINT_TO_POINTER(ESTARDICT_ENGINE_WARNINGS_IDXSIZE_NO_MATCH));
        }

    } else {
        g_critical("--->Cannot open idx file");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_IDX;
        return FALSE;
    }

    // Open the dict file for reading
    starDictInfo->dictFile = gzopen(starDictInfo->dictFileName, "rb9");

    // If we can open the file
    if (starDictInfo->dictFile == NULL) {
        //g_critical("Cannot open dict file: %s", gzerror(dictFile, &error));
        g_critical("Cannot open dict file");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_DICT;
        return FALSE;
    }

    return TRUE;
}


/**
 * @brief Check the EstarDict limitation
 *
 * @param starDictInfo StardDict struct
 * @return TRUE the process can continue, false otherwise
 */
gboolean checkEstardictLimit(StarDictInfo* starDictInfo) {
    gboolean result = TRUE;

    /*
     * If "idxoffsetbits=64", the file size of the .dict file will be bigger
     * than 4G. Because we often need to mmap this large file, and there is
     * a 4G maximum virtual memory space limit in a process on the 32 bits
     * computer, which will make we can get error, so "idxoffsetbits=64"
     * dictionary can't be loaded in 32 bits machine in fact, StarDict will
     * simply print a warning in this case when loading. 64-bits computers
     * should haven't this limit.
     */
    g_debug("--->idxoffsetbits %ld", starDictInfo->idxoffsetbits);
    if (starDictInfo->idxoffsetbits != 0 && starDictInfo->idxoffsetbits != 32) {
        g_critical("Error loading dictionary: eStarDict doesn't work with 64bit dictionary.");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_64BIT_DICTIONARY_NOT_SUPPORTED;
        result = FALSE;
    }

    gchar* sametypesequence = starDictInfo->sametypesequence;
    if (
        sametypesequence != NULL &&
        strlen(sametypesequence) == 1 &&
        (g_ascii_strcasecmp(sametypesequence, "m") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "l") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "t") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "y") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "h") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "g") == 0)
    ) {
        // I don't want to make a different condition than parseDefinition method
        // so why this empty if
    } else {
        g_critical("Dictionary format not supported yet.");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_DICTIONARY_FORMAT_NOT_SUPPORTED;
        result = FALSE;
    }

    return result;
}


/**
 * @brief Load dictionary
 * Load dictionary from concrete location.
 *
 * @param path The directory from where we load the StarDict dictionary.
 * The string format is the result of the GUI file selector
 * @return Return dictionary struct or NULL in case of failure
 */
StarDictInfo* loadDictionary(gchar* path) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    g_assert(path != NULL);

    gboolean result = FALSE;

    // Create StarDict struct
    StarDictInfo* starDictInfo = (StarDictInfo*)g_try_malloc0(sizeof(StarDictInfo));

    if ( starDictInfo != NULL ) {
            
        // Store the dictionary path
        starDictInfo->path = g_strdup(path);

        // Initialize error value
        estardictEngineError = ESTARDICT_ENGINE_ERROR_NO_ERROR;

        // Check and store the mandatory files
        result = findFiles(starDictInfo);

        if (result) {

            // Parse the ifo file
            result = parseIfoFile(starDictInfo);

            if ( result ) {
                result = checkEstardictLimit(starDictInfo);
            }

            if (result) {

                // Load idx file
                result = loadIdx(starDictInfo);
            }

        // Mandatory files not present
        } else {
            estardictEngineError = ESTARDICT_ENGINE_ERROR_MANDATORY_FILES_NOT_PRESENT;
        }

        if (!result) {
            freeData(starDictInfo);
            starDictInfo = NULL;
        }
        
    } else {
        g_critical("Could not allocate memory for dictionary information\n");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_INFO;
    }
    
    return starDictInfo;
}


/**
 * @brief Retrieve definition from dict file
 *
 * @param starDictInfo StarDict struct
 * @param coordinates Definition coordinate in the dict file
 * @return Definition or NULL in case of failure
 */
char* retrieveDefinition(StarDictInfo* starDictInfo, WordCoordinates* coordinates) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    g_assert(starDictInfo != NULL);
    g_assert(coordinates != NULL);
    g_assert(coordinates->size != 0);

    gint error;
    gint offset = coordinates->offset;
    gint size = coordinates->size;

    gchar* definition = g_try_malloc(size);
    if ( definition == NULL) {
        g_critical("Error allocating memory for the dict file definition");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_DEFINITION;
        return NULL;
    }

    // Rewind the file cursor position
    gzrewind(starDictInfo->dictFile);

    // If we can open the file
    if (starDictInfo->dictFile) {
        g_debug("--->Dict file opened");

        glong result = (glong)gzseek(starDictInfo->dictFile, offset, SEEK_SET);

        // Error during seeking on the dict file
        if (result == -1) {
            g_critical("Error during seeking on the dict file: %s", gzerror(starDictInfo->dictFile, &error));
            estardictEngineError = ESTARDICT_ENGINE_ERROR_SEEKING_DICT;
            return NULL;
        }

        gint readed = (gint)gzread(starDictInfo->dictFile, definition, size);

        // Error during reading the dict file
        if (readed != size) {
            g_critical("Error during reading the dict file: %s", gzerror(starDictInfo->dictFile, &error));
            estardictEngineError = ESTARDICT_ENGINE_ERROR_READING_DICT;
            return NULL;
        }

    } else {
        g_critical("Cannot open dict file: %s", gzerror(starDictInfo->dictFile, &error));
        estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_DICT;
        return NULL;
    }

    return g_strndup(definition, size);
}


/**
 * @brief Really simple untag method.
 * For converting from html to text.
 * No check for invalid tag format, no replace for HTML special
 * characters
 *
 * @param html HTML string
 * @return Text string
 */
gchar* untag(gchar* html) {
    int i;
    int j;
    gboolean insideTag = FALSE;
    gchar c;

    gchar* text = g_try_malloc(strlen(html));
    if ( text == NULL) {
        g_critical("Error allocating memory for the untag method");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_DEFINITION;
        return NULL;
    }

    for (i = 0, j = 0; i < strlen(html); i++ ) {
        c = html[i];

        if ( c == '<') {
            insideTag = TRUE;
        }

        if ( !insideTag ) {
            text[j] = c;
            j++;
        }

        if ( c == '>') {
            insideTag = FALSE;
        }
    }
    text[j] = '\0';

    g_debug("HTML %s", html);
    g_debug("TEXT %s", text);

    return text;
}


/**
 * @brief Parse the text from the dict file and returns the formatted
 * definition.
 *
 * @param starDictInfo Dictionary
 * @param dictText Definition text from dict file
 * @return Formatted definition
 */
gchar* parseDefinition(StarDictInfo* starDictInfo, gchar* dictText) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    gchar* sametypesequence = starDictInfo->sametypesequence;

    // Single sametypesequence pure text
    if (
        sametypesequence != NULL &&
        strlen(sametypesequence) == 1 &&
        (g_ascii_strcasecmp(sametypesequence, "m") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "l") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "t") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "y") == 0)
    ) {
        g_message("---> Sametypesequence pure text");
        return dictText + '\0';

    // Single sametypesequence HTML and pango
    } else if (
        sametypesequence != NULL &&
        strlen(sametypesequence) == 1 &&
        (
        g_ascii_strcasecmp(sametypesequence, "h") == 0 ||
        g_ascii_strcasecmp(sametypesequence, "g") == 0
        )
    ) {
        return untag(dictText + '\0');
    } else {
        estardictEngineError = ESTARDICT_ENGINE_ERROR_PARSING_DEFINITION;
    }

    return NULL;
}


/**
 * @brief Return the coordinates in the data file for the given word and
 * dictionary.
 *
 * @param starDictInfo Dictionary
 * @param word Word to search
 * @param diatrics Consider diatrics when search
 * @return Coordinates for the given word and dictionary, NULL if not
 * found or in case of error (in this case the estardictEngineError
 * is valued)
 */
WordCoordinates* getWordCoordinates(StarDictInfo* starDictInfo, gchar* word, gboolean diacritics) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    WordCoordinates* coordinates = NULL;

    if ( starDictInfo != NULL && starDictInfo->idxFile != NULL ) {

        rewind(starDictInfo->idxFile);

        // If we can open the file
        if (starDictInfo->idxFile) {

            // Copy a piece of the file content in the buffer
            gchar buffer[COORDINATES_BUFFER_LENGTH + 1];

            gchar* tmp = NULL;
            gchar* tmp1 = NULL;
            guint len = 0;
            guint wordLenght = strlen(word);
            gboolean continueRead = FALSE;
            gboolean firstCycle = TRUE;

            do {

                gint readed = (gint)fread(buffer, sizeof(gchar), COORDINATES_BUFFER_LENGTH, starDictInfo->idxFile);

                if (readed < 0) {
                    g_critical("---> Error while reading file for searching");
                    estardictEngineError = ESTARDICT_ENGINE_ERROR_READING_IDX;
                    break;

                }
    
                continueRead = (readed == (COORDINATES_BUFFER_LENGTH));

                tmp = buffer;

                guint pos = 0;

                // I want to be sure to have the coordinates value after
                // the word
                gint min = _MIN(readed, COORDINATES_BUFFER_LENGTH - COORDINATES_BLOCK);

                // Loop over buffer size
                while (pos < min) {

                    // Normalize the word without diatrics
                    // e.g. espátula
                    if (diacritics) {
                        tmp1 = strip(tmp);
                        //g_debug("Search stripped word %s", tmp1);
                    }

                    // we have found word in dictionary
                    if ( 
                     ( strlen(tmp) == wordLenght && g_ascii_strncasecmp(tmp, word, wordLenght) == 0 )
                     || ( diacritics && tmp1 != NULL && strlen(tmp1) == wordLenght && g_ascii_strncasecmp(tmp1, word, wordLenght) == 0 )
                    ) {
                        
                        if ( diacritics ) {
                            word = g_strdup(tmp);
                        }

                        // Ignore the first word inside the loop cycle
                        // it could be partial
                        if ( !firstCycle && pos < COORDINATES_BLOCK ) {

                            // calculate length of word
                            len = strlen(tmp) + 1;

                            // go to the coordinate field (offset, size)
                            tmp += len;

                            // go to the next word
                            tmp += 2 * sizeof(guint);

                            // update position value
                            pos = pos + len + 2 * sizeof(guint);

                            continue;
                        }
    
                        // temp variable for offset and size
                        guint val = 0;

                        // calculate length of word
                        gint len = strlen(word) + 1;

                        // go to the coordinate field (offset, size)
                        tmp += len;

                        // Coordinates struct for word
                        coordinates = (WordCoordinates*)malloc(sizeof(WordCoordinates));

                        // save offset
                        memcpy(&val,tmp,sizeof(guint));
                        guint offset = ntohl(val);
                        coordinates->offset = offset;

                        // save size
                        memcpy(&val,tmp + sizeof(guint),sizeof(guint));
                        guint size = ntohl(val);
                        coordinates->size = size;

                        continueRead = FALSE;
                        g_debug("--->%s %d %d", word, offset, size);
                        break;
                    }
    
                    // calculate length of word
                    len = strlen(tmp) + 1;

                    // go to the coordinate field (offset, size)
                    tmp += len;

                    // go to the next word
                    tmp += 2 * sizeof(guint);

                    // update position value
                    pos = pos + len + 2 * sizeof(guint);
                }

                if (!continueRead) break;

                // Create some overlap to be sure to not miss words
                gint result = fseek(starDictInfo->idxFile, COORDINATES_BLOCK * (-2), SEEK_CUR);
                if ( result != 0 ) {
                    g_critical("--->Cannot seek idx file");
                    estardictEngineError = ESTARDICT_ENGINE_ERROR_READING_IDX;
                }

                firstCycle = FALSE;

            } while(continueRead);

        } else {
            g_critical("--->Cannot open idx file");
            estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_IDX;
        }
    } else {
        g_critical("--->Internal error while opening idx file");
    }
    return coordinates;
}


/**
 * @brief Return the word list with the given prefix and dictionary.
 *
 * @param starDictInfo Dictionary
 * @param prefix Prefix to search
 * @param maxNumSuggestedWords Maximum number of suggested words
 * @return Array of words,
 * NULL if not found or in case of error (in this case the
 * estardictEngineError is valued)
 */
GArray* getWordList(StarDictInfo* starDictInfo, gchar* prefix, gint maxNumSuggestedWords) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    GArray* wordList = g_array_sized_new(TRUE, TRUE, sizeof(gchar*), maxNumSuggestedWords);

    rewind(starDictInfo->idxFile);

    // If we can open the file
    if (starDictInfo->idxFile) {

        // Copy a piece of the file content in the buffer
        gchar buffer[COORDINATES_BUFFER_LENGTH + 1];

        gchar* tmp = NULL;
        guint len = 0;
        guint prefixLenght = strlen(prefix);
        gboolean continueRead = FALSE;
        gboolean firstCycle = TRUE;

        do {

            gint readed = (gint)fread(buffer, sizeof(gchar), COORDINATES_BUFFER_LENGTH, starDictInfo->idxFile);

            if (readed < 0) {
                g_critical("---> Error while reading file for searching");
                estardictEngineError = ESTARDICT_ENGINE_ERROR_READING_IDX;
                break;

            }

            continueRead = (readed == (COORDINATES_BUFFER_LENGTH));

            tmp = buffer;

            guint pos = 0;

            // I want to be sure to have the words after the given prefix
            gint min = _MIN(readed, COORDINATES_BUFFER_LENGTH - (maxNumSuggestedWords * COORDINATES_BLOCK) );

            // Loop over buffer size
            while (pos < min) {
                gint i;

                // we have found a word that start with prefix in the dictionary
                if ( g_ascii_strncasecmp(tmp, prefix, prefixLenght) == 0 ) {

                    // Ignore the first word inside the loop cycle
                    // it could be partial
                    if ( !firstCycle && pos < COORDINATES_BLOCK ) {

                        // calculate length of word
                        len = strlen(tmp) + 1;

                        // go to the coordinate field (offset, size)
                        tmp += len;

                        // go to the next word
                        tmp += 2 * sizeof(guint);

                        // update position value
                        pos = pos + len + 2 * sizeof(guint);

                        continue;
                    }

                    for (i = 0; i < maxNumSuggestedWords; i++) {

                        // New word
                        gchar* newWord = g_strdup(tmp);

                        // Add to the word list
                        g_array_append_val(wordList, newWord);

                        // calculate length of word
                        len = strlen(tmp) + 1;

                        // go to the coordinate field (offset, size)
                        tmp += len;

                        // go to the next word
                        tmp += 2 * sizeof(guint);
                    }

                    continueRead = FALSE;
                    break;
                }

                // calculate length of word
                len = strlen(tmp) + 1;

                // go to the coordinate field (offset, size)
                tmp += len;

                // go to the next word
                tmp += 2 * sizeof(guint);

                // update position value
                pos = pos + len + 2 * sizeof(guint);
            }



            if (!continueRead) break;

            // Create some overlap to be sure to not miss words
            gint result = fseek(starDictInfo->idxFile, COORDINATES_BLOCK * (-2), SEEK_CUR);
            if ( result != 0 ) {
                g_critical("--->Cannot seek idx file");
                estardictEngineError = ESTARDICT_ENGINE_ERROR_READING_IDX;
            }

            firstCycle = FALSE;

        } while(continueRead);

    } else {
        g_critical("--->Cannot open idx file");
        estardictEngineError = ESTARDICT_ENGINE_ERROR_OPEN_IDX;
    }

    return wordList;
}


/**
 * @brief Return the definition of the given word for the
 * given dictionary.
 *
 * @param starDictInfo Dictionary
 * @param word Word to search
 * @return Definition for the given word, NULL if not found or in case
 * of error (in this case the estardictEngineError is valued)
 */
char* getDefinition(StarDictInfo* starDictInfo, gchar* word) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    g_assert(starDictInfo != NULL);
    g_assert(word != NULL);
    g_debug("%s %s", starDictInfo->bookname, word);

    char* definition = NULL;
    WordCoordinates* coordinates = NULL;

    coordinates = getWordCoordinates(starDictInfo, word, FALSE);

    if ( coordinates == NULL
        && estardictEngineError != ESTARDICT_ENGINE_ERROR_READING_IDX 
        && estardictEngineError != ESTARDICT_ENGINE_ERROR_OPEN_IDX
        && isPlainLetters(word) ) {
            coordinates = getWordCoordinates(starDictInfo, word, TRUE);
    }

    // Word found
    if ( coordinates != NULL ) {
        g_debug("%s: %ld, %ld", word, coordinates->offset, coordinates->size );

        // Text from dict file
        char* dictText = retrieveDefinition(starDictInfo, coordinates);

        // Free coordinates
        g_free(coordinates);
        coordinates = NULL;

        if (dictText != NULL) {
            definition = parseDefinition(starDictInfo, dictText);
        }

    // Word not found
    } else {
        if ( estardictEngineError != ESTARDICT_ENGINE_ERROR_READING_IDX && estardictEngineError != ESTARDICT_ENGINE_ERROR_OPEN_IDX )
            g_debug("Word not found");
    }

    return definition;
}


/**
 * @brief Main
 * Test function
 */
//int main() {
    //g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    //StarDictInfo* starDictInfo = loadDictionary("/home/root/dictTest/stardict-longman-2.4.2");
    //StarDictInfo* starDictInfo = loadDictionary("/home/zeusone/python_efl/babiloo/doc/efl/dictTest/stardict-longman-2.4.2");
    //StarDictInfo* starDictInfo = loadDictionary("/home/vaudano/tmp/babiloo/babiloo/doc/efl/dictTest/stardict-longman-2.4.2");
    //if ( starDictInfo != NULL ) {
        //char* definition = getDefinition(starDictInfo, "dog");
        //g_debug("definition: %s", definition);
    //}
    //freeData(starDictInfo);
    //return 0;
//}
