/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-gui-definitionPage.c
  * @author Luca Vaudano
  */
#include "estardict-gui-definitionPage.h"
#include "estardict-gui-logic.h"
#include "estardict-gui.h"
#include "estardict-gui-utils.h"
#include "i18n.h"

// Windows states
typedef enum _SUGGESTED_WORDS_BOX_STATE {
    ENABLED,
    DISABLED
} SUGGESTED_WORDS_BOX_STATE;

Evas_Object *suggestedBox, *li, *wordEntry, *scrollerBox, *sc;
SUGGESTED_WORDS_BOX_STATE suggestedWordsBoxState;


/**
 * @brief Action fired when the user double-clicks on the suggested word list
 *
 * @param data List item selected
 * @param obj
 * @param event_info
 */
void suggestedWordSelected(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    Elm_List_Item *it = event_info;
    if (it) {
        const char *word = elm_list_item_label_get(it);
        g_message("Suggested word selected %s", word);

        // Set entry word, no user interaction so it works
        elm_entry_entry_set(wordEntry, word);

        // Show definition
        searchWord(wordEntry, NULL, NULL);
    }
}


/**
 * @brief Create definition page, if needed
 */
void createDefinitionPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Evas_Object *toolbarBox, *ic, *bt, *speakBt, *clearBt, *searchBt, *bx2;
    char buf[PATH_MAX];

    // Main box
    definitionBox = elm_box_add(win);
    evas_object_size_hint_weight_set(definitionBox, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    elm_win_resize_object_add(win, definitionBox);

    // Horizontal box with clear, word field and search button
    toolbarBox = elm_box_add(win);
    elm_box_horizontal_set(toolbarBox, EINA_TRUE);
    evas_object_size_hint_weight_set(toolbarBox, EVAS_HINT_EXPAND, 0);
    evas_object_size_hint_align_set(toolbarBox, EVAS_HINT_FILL, 0);

    // Dictionaries icon
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), DICTIONARIES_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    bt = elm_button_add(win);
    elm_object_part_content_set(bt, "icon", ic);
    evas_object_size_hint_weight_set(bt, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(toolbarBox, bt);
    evas_object_show(bt);
    evas_object_show(ic);
    evas_object_smart_callback_add(bt, "clicked", promoteDictionariesPage, NULL);

    // Settings toolbar icon
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), SETTINGS_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    bt = elm_button_add(win);
    elm_object_part_content_set(bt, "icon", ic);
    evas_object_size_hint_weight_set(bt, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(bt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(toolbarBox, bt);
    evas_object_show(bt);
    evas_object_show(ic);
    evas_object_smart_callback_add(bt, "clicked", promoteSettingsPage, NULL);

    // Speak toolbar icon
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), SPEAKER_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    speakBt = elm_button_add(win);
    elm_object_part_content_set(speakBt, "icon", ic);
    evas_object_size_hint_weight_set(speakBt, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(speakBt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(toolbarBox, speakBt);
    evas_object_show(speakBt);
    evas_object_show(ic);

    elm_box_pack_end(definitionBox, toolbarBox);
    evas_object_show(toolbarBox);

    // Horizontal box with clear, word field and search button
    bx2 = elm_box_add(win);
    elm_box_horizontal_set(bx2, EINA_TRUE);
    evas_object_size_hint_weight_set(bx2, EVAS_HINT_EXPAND, 0);
    evas_object_size_hint_align_set(bx2, EVAS_HINT_FILL, 0);

    // Clear button
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), CLEAR_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    clearBt = elm_button_add(win);
    elm_object_part_content_set(clearBt, "icon", ic);
    evas_object_size_hint_weight_set(clearBt, 0, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(clearBt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(bx2, clearBt);
    evas_object_show(clearBt);
    evas_object_show(ic);

    // Word field
    wordEntry = elm_entry_add(win);
    elm_entry_line_wrap_set(wordEntry, EINA_FALSE);
    //elm_entry_entry_set(wordEntry, "");
    elm_entry_single_line_set(wordEntry, EINA_TRUE);
    elm_object_scale_set(wordEntry, 1.5);
    evas_object_size_hint_weight_set(wordEntry, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(wordEntry, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(bx2, wordEntry);
    evas_object_show(wordEntry);

    // The text within the entry was changed because of user interaction. 
    evas_object_smart_callback_add(wordEntry, "changed,user", showSuggestedWords, wordEntry);
    evas_object_smart_callback_add(clearBt, "clicked", clearWord, wordEntry);
    evas_object_smart_callback_add(speakBt, "clicked", speak, wordEntry);

    // Search button
    ic = elm_icon_add(win);
    snprintf(buf, sizeof(buf), SEARCH_ICON, ROOT_DIR);
    elm_icon_file_set(ic, buf, NULL);
    evas_object_size_hint_aspect_set(ic, EVAS_ASPECT_CONTROL_VERTICAL, 1, 1);
    searchBt = elm_button_add(win);
    elm_object_part_content_set(searchBt, "icon", ic);
    evas_object_size_hint_weight_set(searchBt, 0, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(searchBt, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(bx2, searchBt);
    evas_object_show(searchBt);
    evas_object_show(ic);

    evas_object_smart_callback_add(searchBt, "clicked", searchWord, wordEntry);

    elm_box_pack_end(definitionBox, bx2);
    evas_object_show(bx2);

    // Box with suggested words
    suggestedBox = elm_box_add(win);
    evas_object_size_hint_weight_set(suggestedBox, EVAS_HINT_EXPAND, 0.2);
    evas_object_size_hint_align_set(suggestedBox, EVAS_HINT_FILL, EVAS_HINT_FILL);

    // Suggested word list
    li = elm_list_add(win);
    evas_object_size_hint_weight_set(li, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(li, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_list_horizontal_set(li, EINA_FALSE);
    elm_list_mode_set(li, ELM_LIST_LIMIT);
    elm_box_pack_end(suggestedBox, li);
    evas_object_show(li);

    // Entry scroller
    sc = elm_scroller_add(win);
    evas_object_size_hint_weight_set(sc, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(sc, EVAS_HINT_FILL, EVAS_HINT_FILL);
    
    // Box inside the scroller
    scrollerBox = elm_box_add(win);
    evas_object_size_hint_weight_set(scrollerBox, EVAS_HINT_EXPAND, 0.0);
    evas_object_size_hint_align_set(scrollerBox, EVAS_HINT_FILL, 0.0);
    
    elm_object_content_set(sc, scrollerBox);
    elm_box_pack_end(definitionBox, sc);

    evas_object_show(scrollerBox);
    evas_object_show(sc);

    suggestedWordsBoxState = DISABLED;
}


/**
 * @brief Tips on startup screen
 */
void setTipsOnStartup() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    promoteFirstUsagePage();
}


/**
 * @brief Show the definition
 *
 * @param dictionaryName Dictionary name
 * @param text Definition text
 */
void setDefinition(gchar *dictionaryName, gchar *text) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Evas_Object *definition, *frame;
    
    // Definition frame
    frame = elm_frame_add(win);
    elm_object_text_set(frame," ");
    evas_object_size_hint_weight_set(frame, EVAS_HINT_EXPAND, 0.0);
    evas_object_size_hint_align_set(frame, EVAS_HINT_FILL, 0.0);

    // Definition text area
    definition = elm_entry_add(win);
    elm_entry_line_wrap_set(definition, EINA_TRUE);
    elm_object_scale_set(definition, 1.25);
    elm_entry_entry_set(definition, "");
    elm_entry_single_line_set(definition, EINA_FALSE);
    elm_entry_editable_set(definition, EINA_FALSE);
    evas_object_size_hint_weight_set(definition, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(definition, EVAS_HINT_FILL, EVAS_HINT_FILL);

    // Set dictionary name
    gchar* truncatedName = truncateWord(dictionaryName);
    elm_object_text_set(frame, truncatedName);
    
    // Set definition
    elm_entry_entry_set(definition, text);
    
    elm_object_content_set(frame, definition);
    elm_box_pack_end(scrollerBox, frame);
    evas_object_show(definition);
    evas_object_show(frame);
}


/**
 * @brief Clear the content of the suggested word list
 */
void clearSuggestedWordList() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    elm_list_clear(li);
}


/**
 * @brief Add suggested word
 *
 * @param word Suggested word
 */
void addSuggestedWord(gchar *word) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    elm_list_item_append(li, word, NULL, NULL, suggestedWordSelected, word);
}


/**
 * @brief Show suggested word box
 */
void showSuggestedWordBox() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    elm_list_go(li);
    if ( suggestedWordsBoxState == DISABLED ) {
        suggestedWordsBoxState = ENABLED;

        evas_object_hide(sc);
        elm_box_unpack(definitionBox, sc);

        elm_box_pack_end(definitionBox, suggestedBox);
        evas_object_show(suggestedBox);
    }
}


/**
 * @brief Hide suggested word box
 */
void hideSuggestedWordBox() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    if ( suggestedWordsBoxState == ENABLED ) {
        suggestedWordsBoxState = DISABLED;

        evas_object_hide(suggestedBox);
        elm_box_unpack(definitionBox, suggestedBox);

        elm_box_pack_end(definitionBox, sc);
        evas_object_show(sc);
    }
}


/**
 * @brief Clear definition box
 */
void clearDefinition() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    elm_box_clear(scrollerBox);
}
