/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-config.c
  * @author Luca Vaudano
  */
#include "estardict-config.h"
#include "../include/config.h"
#include <Eet.h>

// Configuration file
Eet_File *ef;

// Configuration properties
EstardictProperties *conf;

// Configuration file path
gchar* configurationFilePath;
int size;

// string that represents the entry in eet file, you might like to have
// different profiles or so in the same file, this is possible with
// different strings
static const char CONFIG_FILE_ENTRY[] = "config";

static Eet_Data_Descriptor *estardictDescriptor;
static Eet_Data_Descriptor *dictionaryDescriptor;


/**
 * @brief Close and save the configuration file
 *
 * @return True if success, false otherwise
 **/
gboolean closeConfigFile() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    g_message("Closed configuration file");
    gboolean result = FALSE;

    result = eet_data_write(ef, estardictDescriptor, CONFIG_FILE_ENTRY, conf, TRUE);
    eet_close(ef);

    g_message("result %d", result);
    return result;
}


/**
 * @brief Print configuration file content
 */
static void printConfigFile() {
    const Eina_List* l;
    DictionaryProperties *dictionary;
    DictionaryProperties *soundDictionary;

    if ( conf != NULL ) {
        g_message("Version %d", conf->version);
        g_message("Player %s", conf->player);
        g_message("SuggestedWordsNumber %d", conf->suggestedWordsNumber);
        g_message("SuggestedWordsVisible %d", conf->suggestedWordsVisible);
        EINA_LIST_FOREACH(conf->dictionaries, l, dictionary)
            g_message("Dictionaries %s", dictionary->dictionaryPath);
        EINA_LIST_FOREACH(conf->soundDictionaries, l, soundDictionary)
            g_message("SoundDictionaries %s", soundDictionary->dictionaryPath);
    }
}


/**
 * @brief Set the default values in the configuration file
 *
 * @return True if success, false otherwise
 */
gboolean initializeDefaultValues() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    DictionaryProperties *dictionary;
    DictionaryProperties *soundDictionary;
    gboolean result = TRUE;

    conf = calloc(1, sizeof(EstardictProperties));

    if (!conf) {
        g_debug("Could not allocate memory for eStardict properties configuration\n");
        result = FALSE;

    } else {

        conf->version = CONFIGURATION_VERSION;
        conf->player = eina_stringshare_add(PLAYER_DEFAULT_VALUE);
        conf->suggestedWordsNumber = SUGGESTED_WORD_NUMBER_DEFAULT_VALUE;
        conf->suggestedWordsVisible = SUGGESTED_WORD_SHOW_DEFAULT_VALUE;

        if( strlen(DICTIONARY_PATH_DEFAULT_VALUE) != 0 ) {
			dictionary = calloc(1, sizeof(DictionaryProperties));
			if ( dictionary ) {
				dictionary->dictionaryPath = eina_stringshare_add(DICTIONARY_PATH_DEFAULT_VALUE);
				conf->dictionaries = eina_list_append(conf->dictionaries, dictionary);
			} else {
				g_debug("Could not allocate memory for dictionary properties configuration\n");
				result = FALSE;
			}
        }

        if ( result && strlen(SOUND_DICTIONARY_PATH_DEFAULT_VALUE) != 0 ) {
            soundDictionary = calloc(1, sizeof(DictionaryProperties));
            if ( soundDictionary ) {
                soundDictionary->dictionaryPath = eina_stringshare_add(SOUND_DICTIONARY_PATH_DEFAULT_VALUE);
                conf->soundDictionaries = eina_list_append(conf->soundDictionaries, soundDictionary);
            } else {
                g_debug("Could not allocate memory for sound dictionary properties configuration\n");
                result = FALSE;
            }
        }

    }

    return result;
}


/**
 * @brief Define the properties descriptor
 */
static void defineDescriptor(void) {
    Eet_Data_Descriptor_Class descriptor;

    // Dictionary descriptor
    EET_EINA_STREAM_DATA_DESCRIPTOR_CLASS_SET(&descriptor, DictionaryProperties);
    dictionaryDescriptor = eet_data_descriptor_stream_new(&descriptor);

    EET_DATA_DESCRIPTOR_ADD_BASIC(dictionaryDescriptor, DictionaryProperties, "dictionaryPath", dictionaryPath, EET_T_STRING);

    // Estardict properties descriptor
    EET_EINA_STREAM_DATA_DESCRIPTOR_CLASS_SET(&descriptor, EstardictProperties);
    estardictDescriptor = eet_data_descriptor_stream_new(&descriptor);

    EET_DATA_DESCRIPTOR_ADD_BASIC(estardictDescriptor, EstardictProperties, "version", version, EET_T_UINT);
    EET_DATA_DESCRIPTOR_ADD_BASIC(estardictDescriptor, EstardictProperties, "player", player, EET_T_STRING);
    EET_DATA_DESCRIPTOR_ADD_BASIC(estardictDescriptor, EstardictProperties, "suggestedWordsNumber", suggestedWordsNumber, EET_T_UINT);
    EET_DATA_DESCRIPTOR_ADD_BASIC(estardictDescriptor, EstardictProperties, "suggestedWordsVisible", suggestedWordsVisible, EET_T_UCHAR);

   // Add the dictionary descriptor as a linked list at 'dictionaries' in the main struct
   EET_DATA_DESCRIPTOR_ADD_LIST(estardictDescriptor, EstardictProperties, "dictionaries", dictionaries, dictionaryDescriptor);

   // Add the dictionary descriptor as a linked list at 'dictionaries' in the main struct
   EET_DATA_DESCRIPTOR_ADD_LIST(estardictDescriptor, EstardictProperties, "soundDictionaries", soundDictionaries, dictionaryDescriptor);
}


/**
 * @brief Load configuration file
 *
 * @return Estartdict configuration properties or NULL
 */
 gboolean loadConfiguration() {
    gboolean result = TRUE;
    int num;

    // Configuration file path
    gchar *homeDir = getenv ("HOME");
    gchar* appName = g_ascii_strdown(PACKAGE, -1);
    gchar* configurationDirectory = g_strconcat(homeDir, "/.", appName, NULL);
    configurationFilePath = g_strconcat(configurationDirectory, "/", CONFIGURATION_FILE, NULL);
    g_message("Configuration path %s", configurationFilePath);

    // Try to read the configuration file
    ef = eet_open(configurationFilePath, EET_FILE_MODE_READ_WRITE);

    if ( !ef ) {
        g_critical("Cannot open the configuration file %s", configurationFilePath);
        result = FALSE;

    } else if ( !eet_list(ef, "*", &num) || num == 0 ) {

        g_message("Configuration file doesn't exist or is empty");
        FILE *configFile = NULL;

        // Create configuration directory
        g_message("Create configuration directory");
        if ( mkdir(configurationDirectory, S_IRWXU) == -1 ) {
            if ( errno != EEXIST ) {
                g_critical("Configuration directory: failed to create %s.", configurationDirectory);
                g_critical("%s", strerror(errno));
                result = FALSE;
            } else {
                g_message("Configuration directory already present");
            }
        }

        /* Create configuration file */
        if ( result && ( (configFile = fopen(configurationFilePath, "w+")) == NULL) ) {
            g_critical("Configuration file: failed to open %s.", configurationFilePath);
            g_critical("%s", strerror(errno));
            result = FALSE;
        } else {
            fclose(configFile);
            g_message("Created configuration file");
        }

        if ( result ) {
            result = initializeDefaultValues();
            g_message("Initialization %d", result);
        }

    } else {
        g_debug("Reading existing configuration file");
        conf = eet_data_read(ef, estardictDescriptor, CONFIG_FILE_ENTRY);
    }

    if ( result ) {
        if ( conf ) {
            printConfigFile();
            // TODO
            //if (my_conf->version < 0x112233) {
            //}
            g_debug("Configuration file checked");
        } else {
            g_critical("Error reading configuration file");
            // TODO
            // ask to the user to delete the configuration file
            // then exit from the program
        }
    } else {
        g_critical("Error initialize configuration file");
    }

    return result;
}


/**
 * @brief Initializa configuration file
 *
 * @return True if success, false otherwise
 */
gboolean initializeConfiguration() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean result = TRUE;

    /* Initialize the configuration */
    if (!eet_init())
        result = FALSE;

    if ( result ) {

        /* Define eet descriptor */
        defineDescriptor();

        /* Load eet configuration */
        result = loadConfiguration();
    }

    return result;
}


/**
 * @brief Get dictionary value from the configuration file
 *
 * @return Dictionary path
 */
Eina_List* getDictionaries() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    Eina_List* dictionaries = NULL;

    if ( conf != NULL ) {
        dictionaries = conf->dictionaries;
    }

    return dictionaries;
}


/**
 * @brief Get sound dictionary value from the configuration file
 *
 * @return Sound dictionary path
 */
const gchar* getSoundDictionary() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    const gchar *result = NULL;
    const Eina_List *l;
    DictionaryProperties *soundDictionary;

	if ( conf != NULL ) {
		
    	// Only one for now!
    	EINA_LIST_FOREACH(conf->soundDictionaries, l, soundDictionary)
        	result = soundDictionary->dictionaryPath;
	}

    g_debug("getSoundDictionary %s", result);
    return result;

}


/**
 * @brief Get player value from the configuration file
 *
 * @return Player file
 */
gchar* getPlayer() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
	gchar* player = NULL;
	
	if ( conf != NULL ) {
    	player = g_strdup(conf->player);
	}

    return player;
}


/**
 * @brief Get suggestedWordShow value from the configuration file
 *
 * @return Boolean value of suggestedWordShow
 */
gboolean getSuggestedWordShow() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean visible = FALSE;
    
    if ( conf != NULL ) {
    	visible = conf->suggestedWordsVisible;
	}

    return visible;
}


/**
 * @brief Set dictionary list into the configuration file
 *
 * @param dictionaryList Dictionary list
 */
gboolean setDictionaries(Eina_List *dictionaryList) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    DictionaryProperties *dictionary;
    gboolean result = TRUE;

    if ( conf != NULL ) {
        const Eina_List *l;
        gchar *dictionaryPath;
        
        // Clear previous data
    	conf->dictionaries = eina_list_free(conf->dictionaries);
        
        EINA_LIST_FOREACH(dictionaryList, l, dictionaryPath) {
			if ( strlen(dictionaryPath) != 0 ) {
				g_message("setDictionary %s", dictionaryPath);
				dictionary = calloc(1, sizeof(DictionaryProperties));
				if ( dictionary ) {
					dictionary->dictionaryPath = eina_stringshare_add(dictionaryPath);
					conf->dictionaries = eina_list_append(conf->dictionaries, dictionary);
				} else {
					g_debug("Could not allocate memory for dictionary properties configuration\n");
					result = FALSE;
					break;
				}
		    }
        }
    }

    return result;
}


/**
 * @brief Set sound dictionary value into the configuration file
 *
 * @param Sound dictionary value
 */
gboolean setSoundDictionary(gchar* soundDictionaryPath) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    DictionaryProperties *soundDictionary;
    gboolean result = TRUE;

    // Set ONLY ONE!
    if ( conf != NULL ) {
		
		// Clear previous data
        conf->soundDictionaries = eina_list_free(conf->soundDictionaries);
    
        g_message("setSoundDictionary %s", soundDictionaryPath);
        soundDictionary = calloc(1, sizeof(DictionaryProperties));
        if ( soundDictionary ) {
            soundDictionary->dictionaryPath = eina_stringshare_add(soundDictionaryPath);
            conf->soundDictionaries = eina_list_append(conf->soundDictionaries, soundDictionary);
        } else {
            g_debug("Could not allocate memory for sound dictionary properties configuration\n");
            result = FALSE;
        }
    }

    return result;
}


/**
 * @brief Set player value into the configuration file
 *
 * @param Player value
 */
gboolean setPlayer(gchar* playerFile) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean result = TRUE;

	if ( conf != NULL ) {
    	conf->player = eina_stringshare_add(playerFile);
	}

    return result;
}


/**
 * @brief Set suggestedWordShow value into the configuration file
 *
 * @param suggestedWordShow value
 */
gboolean setSuggestedWordShow(gboolean suggestedWordShow) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean result = TRUE;

	if ( conf != NULL ) {
    	conf->suggestedWordsVisible = suggestedWordShow;
	}

    return result;
}
