#include "mainwindow.h"
#include "ui_mainwindow.h"

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <X11/Xutil.h>
#endif

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    m_player(new SystemPlayer(this))
{
    ui->setupUi(this);
    this->connectSignals();
    m_isCallActive = false;
    m_state = SystemPlayer::PausedState;
#ifdef Q_WS_MAEMO_5
    setWindowFlags(Qt::WindowStaysOnTopHint | Qt::Dialog);
    setAttribute(Qt::WA_Maemo5AutoOrientation);
    this->setHildonFlags();
    this->connectMCESignals();
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(update()));
#endif
    m_imageUrl = QSettings().value("main/background-url", ":/lockscreen.png").toString();
    QPixmap pixmap(m_imageUrl);
    QPalette p = this->palette();
    p.setBrush(QPalette::Window, pixmap);
    this->setPalette(p);
    this->updateTime();
    this->onStateChanged(m_player->state());
    QTimer *timer = new QTimer(this);
    timer->setInterval(30000);
    connect(timer, SIGNAL(timeout()), this, SLOT(updateTime()));
    timer->start();
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::connectSignals()
{
    //connect(ui->unlockButton, SIGNAL(clicked()), this, SLOT(unlock()));
    connect(ui->horizontalSlider, SIGNAL(valueChanged(int)), this, SLOT(sliderValueChanged(int)));
    connect(m_player, SIGNAL(stateChanged(SystemPlayer::PlaybackState)), this, SLOT(onStateChanged(SystemPlayer::PlaybackState)));
    connect(m_player, SIGNAL(metadataChanged(QMap<QString,QVariant>)), this, SLOT(onMetadataChanged(QMap<QString,QVariant>)));
    connect(ui->playButton, SIGNAL(clicked()), this, SLOT(onPlayButtonClicked()));
    connect(ui->nextButton, SIGNAL(clicked()), m_player, SLOT(next()));
    connect(ui->prevButton, SIGNAL(clicked()), m_player, SLOT(previous()));
    connect(ui->playButton, SIGNAL(pressed()), this, SLOT(playButtonPressed()));
    connect(ui->playButton, SIGNAL(released()), this, SLOT(playButtonPressed()));
    connect(ui->nextButton, SIGNAL(pressed()), this, SLOT(nextButtonPressed()));
    connect(ui->nextButton, SIGNAL(released()), this, SLOT(nextButtonPressed()));
    connect(ui->prevButton, SIGNAL(pressed()), this, SLOT(prevButtonPressed()));
    connect(ui->prevButton, SIGNAL(released()), this, SLOT(prevButtonPressed()));
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
}

void MainWindow::sliderValueChanged(int value)
{
    if (value > ui->horizontalSlider->maximum()-100) {
        this->unlock();
        ui->horizontalSlider->blockSignals(true);
        ui->horizontalSlider->setValue(0);
        ui->horizontalSlider->blockSignals(false);
    } else {
        ui->horizontalSlider->setValue(ui->horizontalSlider->minimum());
    }
}

void MainWindow::playButtonPressed()
{
    if (ui->playButton->isDown()) {
        if (m_state == SystemPlayer::PlayingState)
            ui->playButton->setIcon(QIcon(":/icons/PausePressed.png"));
        else
            ui->playButton->setIcon(QIcon(":/icons/PlayPressed.png"));
    } else {
        if (m_state == SystemPlayer::PlayingState)
            ui->playButton->setIcon(QIcon(":/icons/Pause.png"));
        else
            ui->playButton->setIcon(QIcon(":/icons/Play.png"));
    }
}

void MainWindow::nextButtonPressed()
{
    if (ui->nextButton->isDown())
        ui->nextButton->setIcon(QIcon(":/icons/ForwardPressed.png"));
    else
        ui->nextButton->setIcon(QIcon(":/icons/Forward.png"));
}

void MainWindow::prevButtonPressed()
{
    if (ui->prevButton->isDown())
        ui->prevButton->setIcon(QIcon(":/icons/BackPressed.png"));
    else
        ui->prevButton->setIcon(QIcon(":/icons/Back.png"));
}

void MainWindow::paintEvent(QPaintEvent *)
{
#if 0
    QPainter painter(this);
    if (m_imageUrl.isEmpty())
        m_imageUrl = QSettings().value("main/background-url", ":/lockscreen.png").toString();
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        painter.drawImage(this->rect(), QImage(m_imageUrl));
    } else {
        QMatrix x;
        x.rotate(90);
        painter.drawImage(this->rect(), QImage(m_imageUrl).transformed(x));
    }
    painter.drawImage(this->rect(), QImage(m_imageUrl));
#endif
}

void MainWindow::connectMCESignals()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection::systemBus().connect("",
                                         MCE_SIGNAL_PATH,
                                         MCE_SIGNAL_IF,
                                         MCE_TKLOCK_MODE_SIG, this, SLOT(onTkLockChanged(QString)));
    QDBusConnection::systemBus().connect("",
                                         MCE_SIGNAL_PATH,
                                         MCE_SIGNAL_IF,
                                         MCE_CALL_STATE_SIG, this, SLOT(onCallStateChanged(QString)));
#endif
}

void MainWindow::onCallStateChanged(QString state)
{
#ifdef Q_WS_MAEMO_5
    if (state != MCE_CALL_STATE_NONE)
        m_isCallActive = true;
    else
        m_isCallActive = false;
#else
    Q_UNUSED(state)
#endif
}

void MainWindow::onTkLockChanged(QString locked)
{
    if (m_tklockState == locked)
        return;

    m_tklockState = locked;
#ifdef Q_WS_MAEMO_5
    if (locked == MCE_TK_LOCKED && !m_isCallActive) {
        this->showFullScreen();
    } else if (locked == MCE_TK_UNLOCKED) {
        this->hide();
    }
#endif
}

void MainWindow::setHildonFlags()
{
#ifdef Q_WS_MAEMO_5
    quint32 enable = {10};
    Atom winStackingAtom = XInternAtom(QX11Info::display(), "_HILDON_STACKING_LAYER", false);
    XChangeProperty(QX11Info::display(), winId(), winStackingAtom, XA_CARDINAL, 32, PropModeReplace, (uchar*) &enable, 1);

    enable = 1;
    Atom winDNDAtom = XInternAtom(QX11Info::display(), "_HILDON_DO_NOT_DISTURB", false);
    XChangeProperty(QX11Info::display(), winId(), winDNDAtom, XA_INTEGER, 32, PropModeReplace, (uchar*) &enable, 1);
#endif
}

void MainWindow::unlock()
{
#ifdef Q_WS_MAEMO_5
    QDBusInterface lockScreen("com.nokia.mce", "/com/nokia/mce/request", "com.nokia.mce.request", QDBusConnection::systemBus(), this);
    lockScreen.call("req_tklock_mode_change", "unlocked");
#endif
    this->hide();
}

void MainWindow::updateTime()
{
    QString time = QTime::currentTime().toString("hh:mm ap");
    QString date = QDate::currentDate().toString("dddd d MMMM yyyy");

    ui->timeLabel->setText(time);
    ui->dateLabel->setText(date);
}

void MainWindow::showEvent(QShowEvent *e)
{
    if (m_isCallActive) {
        e->ignore();
    } else {
        e->accept();
        QString imageUrl = QSettings().value("main/background-url", ":/lockscreen.png").toString();
        if (imageUrl == m_imageUrl) {
            return;
        } else {
            m_imageUrl = imageUrl;
            QPixmap pixmap(imageUrl);
            QPalette p = this->palette();
            p.setBrush(QPalette::Window, pixmap);
            this->setPalette(p);
        }
    }
}

void MainWindow::orientationChanged()
{
#ifdef Q_WS_MAEMO_5
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height()) {
        QPixmap pixmap(m_imageUrl);
        QPalette p = this->palette();
        p.setBrush(QPalette::Window, pixmap);
        this->setPalette(p);
    } else {
        QPixmap pixmap(m_imageUrl);
        QPalette p = this->palette();
        QMatrix x;
        x.rotate(90);
        p.setBrush(QPalette::Window, pixmap.transformed(x));
        this->setPalette(p);
    }
    ui->horizontalSlider->setMaximum(screenGeometry.width());
#endif
    this->update();
}

void MainWindow::onStateChanged(SystemPlayer::PlaybackState newState)
{
    if (m_state == newState)
        return;

    m_state = newState;
    switch (newState) {
    case SystemPlayer::PlayingState:
        ui->playButton->setIcon(QIcon(":/icons/Pause.png"));
        break;
    case SystemPlayer::StoppedState:
        ui->playButton->setIcon(QIcon(":/icons/Play.png"));
        break;
    case SystemPlayer::PausedState:
        ui->playButton->setIcon(QIcon(":/icons/Play.png"));
        break;
    case SystemPlayer::UnknownState:
        // wtf
        break;
    }
}

void MainWindow::onPlayButtonClicked()
{
    if (m_player->state() == SystemPlayer::PlayingState)
        m_player->pause();
    else if (m_player->state() == SystemPlayer::PausedState)
        m_player->resume();
    else if (m_player->state() == SystemPlayer::StoppedState)
        m_player->play();
}

void MainWindow::onMetadataChanged(QMap<QString,QVariant> metadata)
{
    QString ascii = QString::fromUtf8("♫");
    ui->metadataLabel->setText(ascii + " " + metadata.value(SYSTEM_PLAYER_SONG_TITLE).toString() + " - "
                               + metadata.value(SYSTEM_PLAYER_SONG_ARTIST).toString() + " " + ascii);
}
