/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Model/Boolean.h>

#include <Lum/Array.h>
#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Popup.h>
#include <Lum/Text.h>
#include <Lum/Toggle.h>

static size_t popupFontSize = 200;

class SetSelector : public Lum::Popup
{
private:
  Lum::Model::ActionRef models[Sudoku::dimension+1]; // Values 0..9, where 0 means empty
  bool                  selected;
  size_t                value;

public:
  SetSelector()
  : selected(false),
    value(Sudoku::dimension+1)
  {
    for (size_t i=0; i<=Sudoku::dimension; i++) {
      models[i]=new Lum::Model::Action();
      Observe(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Array  *array;
    Lum::Panel  *panel;

    panel=new Lum::VPanel();

    array=new Lum::Array();
    array->SetFlex(true,true);
    array->SetHorizontalCount(3);

    for (size_t i=1; i<=Sudoku::dimension; i++) {
      Lum::Text   *text;

      text=new Lum::Text();
      text->SetFlex(true,true);
      text->SetText(Lum::Base::NumberToWString(i),
                    Lum::OS::display->GetFont(popupFontSize));
      text->SetAlignment(Lum::Text::centered);

      array->Add(Lum::Button::Create(text,models[i],true,true));
    }
    panel->Add(array);

    panel->Add(Lum::Button::Create(_(L"GAMEAREA_CLEAR",L"Clear"),models[0],true,false));

    SetMain(panel);

    Popup::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=Sudoku::dimension; ++i) {
      if (model==models[i] && models[i]->IsFinished()) {
        selected=true;
        value=i;
        Exit();
        break;
      }
    }

    Popup::Resync(model,msg);
  }

  bool HasSelection() const
  {
    return selected;
  }

  size_t GetValue() const
  {
    return value;
  }
};

class MarkSelector : public Lum::Popup
{
private:
  Lum::Model::BooleanRef models[Sudoku::dimension];
  size_t                 value;

public:
  MarkSelector(size_t value)
  : value(value)
  {
    for (size_t i=0; i<=8; i++) {
      models[i]=new Lum::Model::Boolean(value & (1 << i));
      Observe(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Array *array;
    Lum::Panel *panel;

    panel=new Lum::VPanel();

    array=new Lum::Array();
    array->SetFlex(true,true);
    array->SetHorizontalCount(3);

    for (size_t i=0; i<=8; i++) {
      Lum::Toggle *toggle;
      Lum::Text   *text;

      text=new Lum::Text();
      text->SetFlex(true,true);
      text->SetText(Lum::Base::NumberToWString(i+1),
                    Lum::OS::display->GetFont(popupFontSize));
      text->SetAlignment(Lum::Text::centered);

      toggle=new Lum::Toggle();
      toggle->SetFlex(true,true);
      toggle->SetModel(models[i]);
      toggle->SetLabel(text);

      array->Add(toggle);
    }
    panel->Add(array);

    SetMain(panel);

    Popup::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=8; ++i) {
      if (model==models[i]) {
        if (models[i]->Get()) {
          value|=(1 << i);
        }
        else {
          value=value & ~(1 << i);
        }

        return;
      }
    }

    Popup::Resync(model,msg);
  }

  size_t GetValue() const
  {
    return value;
  }
};

GameArea::GameArea()
 : minFont(Lum::OS::display->GetFont()),
   minMarkFont(Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,7)),
   font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1)),
   presetColor(0.5,0.5,0.5,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   focusBackgroundColor(0.8,0.8,0.8,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   errorColor(1.0,0,0,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   correctColor(0,0.5,0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   hintColor(1,1,0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   boxSize(0),
   focusedCell(0),
   focusChangedAction(new Lum::Model::Action()),
   hintMode(new Lum::Model::Boolean()),
   finished(new Lum::Model::Boolean())
{
  SetCanFocus(true);
  RequestFocus();

  hintMode->Set(false);
  finished->Set(true);

  Observe(hintMode);
}

int GameArea::GetCellOffset(size_t index) const
{
  return (index+1)*1+(index/3+1)*1+index*boxSize;
}

void GameArea::SetFinished(bool finished)
{
  this->finished->Set(finished);

    hintMode->Set(false);
    if (finished) {
      hintMode->Disable();
    }
    else {
      hintMode->Enable();
    }
}

void GameArea::SetModel(const Sudoku &riddle, const Sudoku &solution)
{
  this->solution=solution;
  this->riddle=riddle;
  this->game=riddle;

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  SetFinished(game==solution);
  focusedCell=0;

  Redraw();
  focusChangedAction->Trigger();
}

void GameArea::SetGame(const Game& game)
{
  riddle.SetAreaAsString(game.riddle);
  solution.SetAreaAsString(game.solution);
  this->game.SetAreaAsString(game.game);

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  if (game.marks.length()==9*Sudoku::size) {
    for (size_t i=0; i<9*Sudoku::size; i++) {
      if (game.marks[i]!=L' ') {
        marks[i/9]|=(1 << (i%9));
      }
    }
  }

  SetFinished(this->game==solution);

  focusedCell=0;
  Redraw();

  focusChangedAction->Trigger();
}

void GameArea::GetGame(Game& game) const
{
  riddle.GetAreaAsString(game.riddle);
  solution.GetAreaAsString(game.solution);
  this->game.GetAreaAsString(game.game);

  game.marks=L"";

  for (size_t i=0; i<Sudoku::size; i++) {
    for (size_t j=0; j<=8; j++) {
      if (marks[i] & (1 << j)) {
        game.marks.append(L"*");
      }
      else {
        game.marks.append(L" ");
      }
    }
  }
}

void GameArea::GetGame(Sudoku& game) const
{
  game=this->game;
}

void GameArea::GetRiddle(Sudoku& game) const
{
  game=this->riddle;
}

void GameArea::GetSolution(Sudoku& game) const
{
  game=this->solution;
}

void GameArea::CalcSize()
{
  size_t markBoxSize=0;
  size_t minBoxSize=0;

  // Size of box containing "normal" content text
  for (size_t i=1; i<=Sudoku::dimension; i++) {
    minBoxSize=std::max(minBoxSize,minFont->StringWidth(Lum::Base::NumberToWString(i),
                                                        Lum::OS::Font::bold));
    minBoxSize=std::max(minBoxSize,font->pixelHeight);
  }

  // Size of box if containing markers (3x3 array)
  for (size_t i=1; i<=Sudoku::dimension; i++) {
    markBoxSize=std::max(markBoxSize,
                         minMarkFont->StringWidth(Lum::Base::NumberToWString(i)));
    markBoxSize=std::max(markBoxSize,minMarkFont->pixelHeight);
  }

  markBoxSize*=3;

  // size of area containing of 9x9 boxes with lines in between
  minWidth=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);
  minHeight=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);

  // now the current with depending on the current font
  boxSize=0;
  for (size_t i=1; i<=9; i++) {
    boxSize=std::max(boxSize,font->StringWidth(Lum::Base::NumberToWString(i),
                                               Lum::OS::Font::bold));
    boxSize=std::max(boxSize,font->pixelHeight);
  }

  size=(3+1)*1+(9+1)*1+9*std::max(markBoxSize,boxSize);

  width=size;
  height=size;

  Object::CalcSize();
}

/**
  * Find the font that can show dimensions number with in a box of size x size..
  */
Lum::OS::Font* GameArea::FindFont(Lum::OS::Font* minFont, size_t size)
{
  Lum::OS::FontRef font;
  bool             fits;
  size_t           fontSize;

  font=minFont;
  fontSize=font->pixelHeight+1;

  do {
    Lum::OS::FontRef tmpFont;
    size_t           boxSize=0;

    tmpFont=Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,
                                             fontSize);

    if (tmpFont.Valid()) {
      for (size_t i=1; i<=Sudoku::dimension; i++) {
        boxSize=std::max(boxSize,tmpFont->StringWidth(Lum::Base::NumberToWString(i),
                                                      Lum::OS::Font::bold));
        boxSize=std::max(boxSize,tmpFont->pixelHeight);
      }

      fits=boxSize<=size && fontSize<=size;
    }
    else {
      fits=fontSize<=size;
    }

    if (fits) {
      font=tmpFont;
      fontSize++;
    }

  } while (fits);

  return font.Get();
}

void GameArea::Layout()
{
  boxSize=std::min((width-(3+1)*1-(9+1)*1)/9,
                   (height-(3+1)*1-(9+1)*1)/9);

  size=(3+1)*1+(9+1)*1+9*boxSize;

  font=FindFont(minFont,boxSize);
  markFont=FindFont(minMarkFont,boxSize/3);

  Object::Layout();
}

void GameArea::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  Lum::OS::DrawInfo *draw=GetDrawInfo();
  size_t            offset;

  //
  // Draw game area
  //
  draw->PushForeground(Lum::OS::display->blackColor);
  offset=0;
  for (size_t i=0; i<10; i++) {
    // Vertical
    draw->DrawLine(this->x+offset,this->y,
                   this->x+offset,this->y+size-1);

    // Horizontal
    draw->DrawLine(this->x,this->y+offset,
                   this->x+this->size-1,this->y+offset);

    if (i%3==0) {
      offset++;

      // Vertical
      draw->DrawLine(this->x+offset,this->y,
                     this->x+offset,this->y+this->size-1);

      // Horizontal
      draw->DrawLine(this->x,this->y+offset,
                     this->x+this->size-1,this->y+offset);
    }
    offset=offset+boxSize+1;
  }
  draw->PopForeground();

  //
  // Draw values
  //
  draw->PushFont(font,Lum::OS::Font::bold);
  for (size_t i=0; i<Sudoku::dimension*Sudoku::dimension; i++) {

    // Draw background
    if ((HasFocus() || true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/)
        && !finished->Get() && i==focusedCell) {
      draw->PushForeground(focusBackgroundColor);
    }
    else {
      draw->PushForeground(Lum::OS::display->whiteColor);
    }
    draw->FillRectangle(this->x+GetCellOffset(i%Sudoku::dimension),
                        this->y+GetCellOffset(i/Sudoku::dimension),
                        boxSize,boxSize);
    draw->PopForeground();

    if (riddle.Get(i)!=Sudoku::empty) { // Preset fields
      std::wstring        text(Lum::Base::NumberToWString(riddle.Get(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      draw->PushForeground(presetColor);
      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                       text);
      draw->PopForeground();
    }
    else if (finished->Get() && solution.Get(i)!=Sudoku::empty) {
      std::wstring        text(Lum::Base::NumberToWString(solution.Get(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      if (solution.Get(i)!=game.Get(i)) {
        draw->PushForeground(errorColor);
      }
      else {
        draw->PushForeground(correctColor);
      }

      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                       text);
      draw->PopForeground();
    }
    else if (game.Get(i)!=Sudoku::empty) { // manually set fields
      std::wstring         text(Lum::Base::NumberToWString(game.Get(i)));
      Lum::OS::FontExtent  extent;
      Sudoku::Bitset       allowed;

      game.GetPotential(i,allowed);

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      if (!allowed.test(game.Get(i))) {
        draw->PushForeground(errorColor);
      }
      else {
        draw->PushForeground(Lum::OS::display->blackColor);
      }

      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                       text);
      draw->PopForeground();
    }
    else if (!finished->Get() &&
             hintMode->Get() &&
             i==focusedCell) {
      std::wstring        text(Lum::Base::NumberToWString(solution.Get(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      draw->PushForeground(hintColor);
      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-extent.height)/2+extent.ascent,
                       text);
      draw->PopForeground();
    }
    else {
      for (size_t j=0; j<Sudoku::dimension; j++) {
        if (marks[i] & (1 << j)) {
          std::wstring         text(Lum::Base::NumberToWString(j+1));
          Lum::OS::FontExtent  extent;

          markFont->StringExtent(text,extent);

          draw->PushFont(markFont);
          draw->PushForeground(Lum::OS::display->blackColor);
          draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+
                           (boxSize/3)*(j%3)+
                           (boxSize/3-extent.width+extent.left+extent.right)/2-extent.left,
                           this->y+GetCellOffset(i/Sudoku::dimension)+
                           (boxSize/3)*(j/3)+
                           (boxSize/3-extent.height)/2+extent.ascent,
                           text);
          draw->PopForeground();
          draw->PopFont();
        }
      }
    }
  }
  draw->PopFont();
}

bool GameArea::HandlesFocus() const
{
  return Object::HandlesFocus() && !finished->Get();
}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (finished->Get()) {
    return false;
  }

  if ((event.type==Lum::OS::MouseEvent::down && PointIsIn(event) &&
       event.button==Lum::OS::MouseEvent::button1)) {
    size_t tmp=0;

    while (tmp<Sudoku::dimension*Sudoku::dimension) {
      if (event.x>=this->x+GetCellOffset(tmp%Sudoku::dimension) &&
          event.x<this->x+GetCellOffset(tmp%Sudoku::dimension)+(int)boxSize &&
          event.y>=this->y+GetCellOffset(tmp/Sudoku::dimension) &&
          event.y<this->y+GetCellOffset(tmp/Sudoku::dimension)+(int)boxSize) {
        break;
      }

      tmp++;
    }

    if (tmp<Sudoku::dimension*Sudoku::dimension) {
      if (tmp!=focusedCell) {
        focusedCell=tmp;
        Redraw();
        focusChangedAction->Trigger();
      }

      if (riddle.Get(focusedCell)!=Sudoku::empty) {
        return true;
      }

      if (true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) {
        // No popup dialog, if we need to be finger friendly!
        // We will make use of an external keypad instead in this case.
        return true;
      }

      if (event.x-(this->x+GetCellOffset(tmp%Sudoku::dimension))<=(int)boxSize/2 &&
          event.y-(this->y+GetCellOffset(tmp/Sudoku::dimension))<=(int)boxSize/2) {
        MarkSelector *selector;

        selector=new MarkSelector(marks[focusedCell]);
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x+5,GetWindow()->GetY()+event.y+5);
        selector->GetWindow()->SetOpacity(0.8);

        if (selector->Open()) {
          selector->EventLoop();
          selector->Close();

          marks[focusedCell]=selector->GetValue();
          if (marks[focusedCell]!=0) {
            game.Set(focusedCell,Sudoku::empty);
          }

          Redraw();
        }

        delete selector;
      }
      else {
        SetSelector *selector;

        selector=new SetSelector();
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x+5,GetWindow()->GetY()+event.y+5);
        selector->GetWindow()->SetOpacity(0.8);
        selector->Open();
        selector->EventLoop();
        selector->Close();

        if (selector->HasSelection()) {
          game.Set(focusedCell,selector->GetValue());
          marks[focusedCell]=0;
          SetFinished(game==solution);
          Redraw();
        }

        delete selector;
      }

    }


    return true;
  }

  return false;
}

bool GameArea::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{
  if (finished->Get()) {
    return false;
  }

  size_t value;

  if (event.type==Lum::OS::KeyEvent::down) {
    switch (event.key) {
    case Lum::OS::keyLeft:
      if (focusedCell%Sudoku::dimension>0) {
        focusedCell--;
        Redraw();
        focusChangedAction->Trigger();
      }
      return true;
    case Lum::OS::keyRight:
      if (focusedCell%Sudoku::dimension<Sudoku::dimension-1) {
        focusedCell++;
        Redraw();
        focusChangedAction->Trigger();
      }
      return true;
    case Lum::OS::keyUp:
      if (focusedCell/Sudoku::dimension>0) {
        focusedCell-=Sudoku::dimension;
        Redraw();
        focusChangedAction->Trigger();
      }
      return true;
    case Lum::OS::keyDown:
      if (focusedCell/Sudoku::dimension<Sudoku::dimension-1) {
        focusedCell+=Sudoku::dimension;
        Redraw();
        focusChangedAction->Trigger();
      }
      return true;
    case Lum::OS::keySpace:
    case Lum::OS::keyBackspace:
    case Lum::OS::keyDelete:
      if (riddle.Get(focusedCell)==Sudoku::empty) {
        game.Set(focusedCell,Sudoku::empty);
        Redraw();
        focusChangedAction->Trigger();
      }
      return true;
    default:
      if (Lum::Base::WStringToNumber(event.text,value)) {
        if (value>=0 && value<=Sudoku::dimension) {
          if (riddle.Get(focusedCell)==Sudoku::empty) {
            game.Set(focusedCell,value);
            SetFinished(game==solution);
            Redraw();
            // Focus did not changed, but we simulate it to get the value change
            // notified
            focusChangedAction->Trigger();
          }
        }

        return true;
      }
      break;
    }
  }

  return false;
}

Lum::Model::Action* GameArea::GetFocusChangedAction() const
{
  return focusChangedAction.Get();
}

Lum::Model::Boolean* GameArea::GetHintMode() const
{
  return hintMode.Get();
}

Lum::Model::Boolean* GameArea::GetFinishedModel() const
{
  return finished.Get();
}

bool GameArea::IsFocusedCellEditable() const
{
  return !finished->Get() && riddle.Get(focusedCell)==Sudoku::empty;
}

bool GameArea::IsCellInMarkModus() const
{
  return game.Get(focusedCell)==Sudoku::empty;
}

size_t GameArea::GetFocusedCellValue() const
{
  assert(game.Get(focusedCell)!=Sudoku::empty);

  return game.Get(focusedCell);
}

size_t GameArea::GetFocusedCellMarks() const
{
  assert(game.Get(focusedCell)==Sudoku::empty);

  return marks[focusedCell];
}

void GameArea::SetFocusedCellMarks(size_t value)
{
  assert(!GetFinishedModel()->Get());
  assert(IsFocusedCellEditable());

  game.Set(focusedCell,Sudoku::empty);
  marks[focusedCell]=value;

  Redraw();

  focusChangedAction->Trigger();
}

void GameArea::SetFocusedCellValue(size_t value)
{
  assert(!GetFinishedModel()->Get());
  assert(IsFocusedCellEditable());

  game.Set(focusedCell,value);
  marks[focusedCell]=0;

  Redraw();

  focusChangedAction->Trigger();
  SetFinished(game==solution);
}

void GameArea::MarkEmpty()
{
  assert(!finished->Get());

  for (size_t cell=0; cell<Sudoku::dimension*Sudoku::dimension; cell++) {
    if (riddle.Get(cell)==Sudoku::empty &&
        game.Get(cell)==Sudoku::empty &&
        marks[cell]==0) {
      marks[cell]=0;
      for (size_t mark=0; mark<Sudoku::dimension; mark++) {
        marks[cell]|=(1 << mark);
      }
    }
  }

  Redraw();
}

void GameArea::Solve()
{
  SetFinished(true);
  Redraw();
}

void GameArea::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hintMode) {
    Redraw();
  }

  Object::Resync(model,msg);
}

