/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <vector>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Help.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Progress.h>

#include <Lum/Images/Image.h>
#include <Lum/Images/Loader.h>

#include <Lum/Array.h>
#include <Lum/AspectRatio.h>
#include <Lum/Button.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Image.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Text.h>
#include <Lum/Toggle.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include "config.h"
#include "Configuration.h"
#include "Dancing.h"
#include "GameArea.h"
#include "GenerateDialog.h"
#include "Statistics.h"

static Lum::Def::AppInfo info;

// TODO
static Sudoku riddle,solution;

class GUIAbortTester : public Sudoku::AbortTester
{
private:
  const Lum::Dlg::ProgressTask& task;

public:
  GUIAbortTester(const Lum::Dlg::ProgressTask& task)
   : task(task)
  {
    // no code
  }

  operator bool() const
  {
    return task.IsAborted();
  };
};

class GenerateTask : public Lum::Dlg::ProgressTask
{
public:
  Sudoku::Weight weight;
  size_t         minimumPoints;

public:
  void Run()
  {
    Sudoku area;

    SetProgress(0);
    SetAction(_(L"PROGRESS_SOLUTION",L"Calculating a new solution..."));
    area.FillRandom();
    riddle=area;
    solution=area;

    if (IsAborted()) {
      return;
    }

    SetProgress(10);
    SetAction(_(L"PROGRESS_RIDDLE",L"Calculating puzzle..."));

    Sudoku::GenerateRiddle(riddle,minimumPoints,GUIAbortTester(*this));

    if (IsAborted()) {
      return;
    }

    SetProgress(90);
    SetAction(_(L"PROGRESS_WEIGHT",L"Calculating weight..."));

    area=riddle;
    area.CalculateWeight(solution,weight);

    if (IsAborted()) {
      return;
    }

    SetProgress(100);
  }
};

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef         clearAction;
  Lum::Model::ActionRef         generateAction;
  Lum::Model::ActionRef         solveAction;
  Lum::Model::ActionRef         statAction;
  Lum::Model::ActionRef         helpAction;
  Lum::Model::ActionRef         aboutAction;
  GameArea                      *gameArea;
  GenerateTask                  generateTask;
  Lum::Model::BooleanRef        keypad[Sudoku::dimension];
  Lum::Model::BooleanRef        mark;
  bool                          lock;
  Lum::Images::StatefulImageRef hintImage;
  Lum::Images::StatefulImageRef statisticsImage;
  Lum::Images::StatefulImageRef solveImage;
  Lum::Images::StatefulImageRef generateImage;

public:
  MainDialog()
  : clearAction(new Lum::Model::Action),
    generateAction(new Lum::Model::Action),
    solveAction(new Lum::Model::Action),
    statAction(new Lum::Model::Action),
    helpAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    gameArea(new GameArea()),
    mark(new Lum::Model::Boolean(false)),
    lock(false),
    hintImage(new Lum::Images::StatefulImage()),
    statisticsImage(new Lum::Images::StatefulImage()),
    solveImage(new Lum::Images::StatefulImage()),
    generateImage(new Lum::Images::StatefulImage())
  {
    for (size_t i=0; i<Sudoku::dimension; i++) {
      keypad[i]=new Lum::Model::Boolean(false);
      Observe(keypad[i]);
    }
    Observe(mark);

    generateTask.SetParent(this);
    generateTask.SetCaption(_(L"PROGRESS_TITLE",L"Generating puzzle..."));
    generateTask.minimumPoints=4000;

    Observe(clearAction);
    Observe(generateAction);
    Observe(solveAction);
    Observe(statAction);
    Observe(helpAction);
    Observe(aboutAction);
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
    Observe(generateTask.GetAbortedAction());
    Observe(generateTask.GetFinishedAction());
    gameArea->GetFinishedModel()->Off();
    Observe(gameArea->GetFocusChangedAction());
    Observe(gameArea->GetFinishedModel());

    Lum::Images::loader->LoadApplication(L"Hint.png",hintImage);
    Lum::Images::loader->LoadApplication(L"Statistics.png",statisticsImage);
    Lum::Images::loader->LoadApplication(L"Solve.png",solveImage);
    Lum::Images::loader->LoadApplication(L"Generate.png",generateImage);
  }

  void PreInit()
  {
    Lum::Array *array;
    Lum::Panel *horiz,*vert;

    horiz=Lum::HPanel::Create(true,true);

    gameArea->SetFlex(true,true);
    gameArea->SetModel(riddle,solution);

    horiz->Add(gameArea);

    horiz->AddSpace();

    vert=Lum::VPanel::Create(false,true);

    if (true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) {
      array=new Lum::Array();
      array->SetFlex(true,true);
      array->SetHorizontalCount(3);
      array->SetSpace(false,false);

      for (size_t i=0; i<Sudoku::dimension; i++) {
        array->Add(Lum::Toggle::Create(Lum::Base::NumberToWString(i+1),keypad[i],true,true));
      }

      vert->Add(array);

      array=new Lum::Array();
      array->SetFlex(true,false);
      array->SetHorizontalCount(2);
      array->SetSpace(false,false);

      array->Add(Lum::Button::Create(_(L"MAIN_CLEAR",L"Clear"),clearAction,true,true));
      array->Add(Lum::Toggle::Create(_(L"MAIN_MARK",L"Mark"),mark,true,true));

      vert->Add(array);

      vert->AddSpace();
    }
    else {
      vert->AddSpace(true);
    }

    array=new Lum::Array();
    array->SetFlex(true,false);
    array->SetHorizontalCount(4);
    array->SetSpace(false,false);

    if (hintImage->IsValid()) {
      Lum::Toggle* toggle=new Lum::Toggle();

      toggle->SetFlex(true,false);
      toggle->SetModel(gameArea->GetHintMode());
      toggle->SetLabel(Lum::Image::Create(hintImage));

      array->Add(toggle);
    }
    else {
      array->Add(Lum::Toggle::Create(_(L"GAME_Hint",L"_Hint"),gameArea->GetHintMode(),true,false));
    }
    if (statisticsImage->IsValid()) {
      array->Add(Lum::Button::Create(Lum::Image::Create(statisticsImage),statAction,true,false));
    }
    else {
      array->Add(Lum::Button::Create(_(L"GAME_STAT",L"S_tatistics"),statAction,true,false));
    }

    if (solveImage->IsValid()) {
      array->Add(Lum::Button::Create(Lum::Image::Create(solveImage),solveAction,true,false));
    }
    else {
      array->Add(Lum::Button::Create(_(L"SOLVE",L"_Solve"),solveAction,true,false));
    }

    if (generateImage->IsValid()) {
      array->Add(Lum::Button::Create(Lum::Image::Create(generateImage),generateAction,true,false));
    }
    else {
      array->Add(Lum::Button::Create(_(L"GENERATE",L"_New"),generateAction,true,false));
    }

    vert->Add(array);

    horiz->Add(vert);

    SetMain(horiz);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->Group(_(L"MENU_GAME",L"_Game"))
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_NEW",L"_Generate new game"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"g"),
                                  generateAction)
                 .SetOpensDialog())
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_SOLVE",L"_Solve game"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"s"),
                                  solveAction))
        ->Separator()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_STAT",L"S_tatistics"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"t"),
                                  statAction)
                 .SetOpensDialog())
      ->End()
      ->GroupHelp()
        ->ActionHelp(helpAction)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void GenerateRiddle()
  {
    riddle.Clear();
    gameArea->GetFinishedModel()->Off();
    gameArea->SetModel(riddle,solution);

    generateTask.Start();
  }

  void UpdateKeypad()
  {
    if (lock) {
      return;
    }

    lock=true;
    if (gameArea==NULL ||
        gameArea->GetFinishedModel()->Get() ||
        !gameArea->IsFocusedCellEditable()) {
      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid()) {
          keypad[i]->Set(false);
          keypad[i]->Disable();
        }
      }

      if (mark.Valid()) {
        mark->Disable();
      }

      if (clearAction.Valid()) {
        clearAction->Disable();
      }

      lock=false;
      return;
    }

    if (gameArea->IsCellInMarkModus()) {
      size_t value=gameArea->GetFocusedCellMarks();
      size_t marks=0;

      for (size_t i=0; i<Sudoku::dimension; i++) {
        keypad[i]->Enable();
        keypad[i]->Set(value & (1 << i));

        if (value & (1 << i)) {
          marks++;
        }
      }

      if (marks==1) {
        mark->Enable();
        mark->Set(gameArea->IsCellInMarkModus());
      }
      else {
        mark->Set(marks>1);
        mark->Disable();
      }

      if (marks>=1) {
        clearAction->Enable();
      }
      else {
        clearAction->Disable();
      }
    }
    else {
      size_t value=gameArea->GetFocusedCellValue();

      for (size_t i=0; i<Sudoku::dimension; i++) {
        keypad[i]->Enable();
        keypad[i]->Set(i+1==value);
      }

      mark->Set(false);
      mark->Enable();

      if (value!=Sudoku::empty) {
        clearAction->Enable();
      }
      else {
        clearAction->Disable();
      }
    }

    lock=false;
  }

  void UpdateCell()
  {
    if (lock) {
      return;
    }

    if (gameArea->GetFinishedModel()->Get() ||
        !gameArea->IsFocusedCellEditable()) {
      // We ignore possible changes if there is no valid game
      return;
    }

    lock=true;

    size_t marks=0;
    for (size_t i=0; i<Sudoku::dimension; i++) {
      if (keypad[i].Valid() && keypad[i]->Get()) {
        marks++;
      }
    }

    if (marks==1) {
      mark->Enable();
    }
    else {
      mark->Set(marks>1);
      mark->Disable();
    }

    if (mark->Get()) {
      size_t value=0;

      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid() && keypad[i]->Get()) {
          value|=(1 << i);
        }
      }

      gameArea->SetFocusedCellMarks(value);
    }
    else {
      bool hasValue=false;
      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid() && keypad[i]->Get()) {
          gameArea->SetFocusedCellValue(i+1);
          hasValue=true;
        }
      }

      if (!hasValue) {
        gameArea->SetFocusedCellValue(Sudoku::empty);
      }
    }

    lock=false;
  }

  void ClearCell()
  {
    gameArea->SetFocusedCellValue(Sudoku::empty);
  }

  void SetRiddle()
  {
    gameArea->SetModel(riddle,solution);
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      if (LoadConfig()) {
        if (game!=NULL) {
          Sudoku         riddle,solution;
          Sudoku::Weight weight;

          riddle.SetAreaAsString(game->riddle);
          solution.SetAreaAsString(game->solution);

          riddle.CalculateWeight(solution,weight);

          gameArea->SetGame(*game);

          delete game;
          game=NULL;
        }
      }

      if (gameArea->GetFinishedModel()->Get()) {
        solveAction->Disable();
        statAction->Disable();
      }

      gameArea->GetFinishedModel()->On(false);
    }
    else if (model==GetClosedAction() &&
             GetClosedAction()->IsFinished()) {
      if (!gameArea->GetFinishedModel()->Get()) {
        game=new Game();
        gameArea->GetGame(*game);
      }
      SaveConfig();
    }
    else if (model==clearAction &&
             clearAction->IsFinished()) {
      ClearCell();
    }
    else if (model==generateAction &&
             generateAction->IsFinished()) {
      if (ShowGenerateDialog(this,generateTask.minimumPoints)) {
        GenerateRiddle();
      }
    }
    else if (model==generateTask.GetFinishedAction() &&
             generateTask.GetFinishedAction()->IsFinished()) {
      solveAction->Enable();
      statAction->Enable();
      SetRiddle();
      gameArea->GetFinishedModel()->On(false);
    }
    else if (model==generateTask.GetAbortedAction() &&
             generateTask.GetAbortedAction()->IsFinished()) {
      gameArea->GetFinishedModel()->On(false);
    }
    else if (model==solveAction &&
             solveAction->IsFinished() &&
             solveAction->IsEnabled()) {
      if (Lum::Dlg::Msg::Ask(this,
                             _(L"SOLVE_TITLE",L"Are you sure?"),
                             _(L"SOLVE_BODY",
                               L"Are you sure that you want to solve the game? \n"
                               L"\n"
                               L"Solving the game cannot be undone!"),
                             _(L"SOLVE_BUTTONS",L"_Solve it!*|_Cancel^"))==0) {
        gameArea->GetFinishedModel()->Off();
        gameArea->Solve();
        gameArea->GetFinishedModel()->On(false);
        solveAction->Disable();
        statAction->Disable();
      }
    }
    else if (model==statAction &&
             statAction->IsFinished()) {
      Sudoku game;
      Sudoku riddle;
      Sudoku solution;

      gameArea->GetGame(game);
      gameArea->GetRiddle(riddle);
      gameArea->GetSolution(solution);

      StatisticsDialog::Show(GetWindow(),game,riddle,solution);
    }
    else if (model==helpAction &&
             helpAction->IsFinished()) {
      Lum::Dlg::Help::Show(this,
                           _(L"HELP_BODY",
                             L"The interface supports setting numbers and marking\n"
                             L"fields as potentially holding a subset of numbers.\n"
                             L"\n"
                             L"Marks will be shown as small numbers, set values as black,\n"
                             L"predefined number will be drawn grey, and conflicts\n"
                             L"red.\n"
                             L"\n"
                             L"To set/mark a field use the keypad. Explcitely use the mark\n"
                             L"button to tag an entry as marked.\n"
                             L"\n"
                             L"Have fun :-)"));
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==gameArea->GetFinishedModel() &&
             gameArea->GetFinishedModel()->Get()) {
      Lum::Dlg::Msg::ShowOk(this,
                            _(L"CONGRATS_TITLE",L"Yeeeaarrrr!!\n"),
                            _(L"CONGRATS_BODY",
                              L"It looks like you have successfully finished\n"
                              L"the given puzzle!\n"
                              L"Congratulations!"));
      solveAction->Disable();
      statAction->Disable();
      UpdateKeypad();
    }
    else if (model==gameArea->GetFocusChangedAction() &&
             gameArea->GetFocusChangedAction()->IsFinished()) {
      UpdateKeypad();
    }
    else if (model==mark) {
      UpdateCell();
      UpdateKeypad();
    }
    else {
      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (model==keypad[i]) {
          UpdateCell();
          UpdateKeypad();
        }
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A simple Sudoku implementation..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2006, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    Lum::Base::Path path;

    path.SetNativeDir(Lum::Base::Path::GetApplicationDataDir());
    path.AppendDir(L"icons");

    Lum::Images::loader->AddApplicationSearchPath(path);

    path.SetRoot(L"");
    path.SetDir(L"data");
    path.AppendDir(L"icons");
    path.AppendDir(L"16x16");
    Lum::Images::loader->AddApplicationSearchPath(path);


    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"EightyOne")
