#ifndef SUDOKU_H
#define SUDOKU_H

/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <functional>
#include <string>
#include <vector>

class Game
{
public:
  std::wstring riddle;
  std::wstring solution;
  std::wstring game;
  std::wstring marks;
};

class Sudoku
{
public:
  static const size_t quadrantDimension=3;
  static const size_t dimension=quadrantDimension*quadrantDimension;
  static const size_t size=dimension*dimension;

  static const size_t empty=0;

public:
  class AbortTester
  {
  public:
    virtual operator bool() const = 0;
  };

  class Bitset
  {
  private:
    size_t value;
    size_t bits;

  public:
    inline Bitset()
    : value(0),bits(0)
    {
    }

    inline void set(size_t index)
    {
      if (!(this->value & (1 << index))) {
        this->value|=(1 << index);
        ++bits;
      }
    }

    inline void clear(size_t index)
    {
      if (this->value & (1 << index)) {
        this->value&=(~(1 << index));
        --bits;
      }
    }

    inline bool get(size_t index) const
    {
      return (value & (1 << index))!=0;
    }

    inline bool test(size_t index) const
    {
      return (value & (1 << index))!=0;
    }

    inline size_t count() const
    {
      return bits;
    }

    inline size_t getValue() const
    {
      return value;
    }
  };

private:
  class CalcStep
  {
  public:
    size_t              x;
    size_t              y;
    std::vector<size_t> potential;
  };

public:
  struct Weight
  {
  public:
    size_t filledFields;
    size_t directFills;
    size_t crossHatch;
    size_t singleCandidate;
    size_t singleSquare;
    size_t guesses;
    double rating;

  public:
    Weight();
  };

private:
  std::vector<size_t> area;

private:
  bool CalculateWeightDirectFills(const Sudoku& solution, Weight &weight);
  bool CalculateWeightCrossHatch(const Sudoku& solution, Weight &weight);
  bool CalculateWeightSingleSquare(const Sudoku& solution, Weight &weight);
  bool CalculateWeightSingleCandidate(const Sudoku& solution, Weight &weight);

public:
  Sudoku();

  Sudoku(const Sudoku& sudoku);

  Sudoku& operator=(const Sudoku& sudoku);
  bool operator==(const Sudoku& sudoku);

  inline void Set(size_t x, size_t y, size_t value)
  {
    //assert(x>=0 && x<dimension && y>=0 && y<dimension);

    area[x+y*dimension]=value;
  }

  inline void Set(size_t pos, size_t value)
  {
    //assert(x>=0 && x<dimension && y>=0 && y<dimension);

    area[pos]=value;
  }

  inline size_t Get(size_t x,size_t y) const
  {
    //assert(x>=0 && x<dimension && y>=0 && y<dimension);

    return area[x+y*dimension];
  }

  inline size_t Get(size_t pos) const
  {
    //assert(x>=0 && x<dimension && y>=0 && y<dimension);

    return area[pos];
  }

  size_t GetFieldsFree() const;
  size_t GetFieldsSet() const;

  bool IsEmpty() const;
  bool IsValid() const;

  void GetPotential(size_t pos, Bitset& possible, bool incQuad=true) const;
  void GetPotential(size_t x, size_t y, Bitset& possible, bool incQuad=true) const;
  void GetPotential(size_t x, size_t y, std::vector<size_t>& list, bool incQuad=true) const;

  void SetAreaAsString(const std::wstring& value);
  void GetAreaAsString(std::wstring& value) const;

  void Draw();
  void Clear();

  void FillRandom();
  static void GenerateRiddle(Sudoku &riddle, size_t minimumPointLimit,
                             const AbortTester& abortTester);
  bool HasOneSolution();
  void CalculateWeight(const Sudoku& solution, Weight &weight);
};

#endif
