/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Statistics.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Model/DataStream.h>

#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/PercentBar.h>
#include <Lum/Tab.h>
#include <Lum/Text.h>

static Lum::Object* GetPercentBar(size_t count)
{
  Lum::PercentBar                 *bar;
  Lum::Model::DoubleDataStreamRef stream;

  stream=new Lum::Model::DoubleDataStream();
  stream->SetChannels(1);
  stream->Set(0,(1.0*count)/9);

  bar=new Lum::PercentBar();
  bar->SetFlex(true,true);
  bar->DisplayVertical();
  bar->SetHeight(Lum::Base::Size::stdCharHeight,5);
  bar->SetModel(stream);

  return bar;
}

static Lum::Object* GetText(const std::wstring& label)
{
  Lum::Text *text;

  text=new Lum::Text();
  text->SetFlex(true,false);
  text->SetAlignment(Lum::Text::centered);
  text->SetText(label);

  return text;
}

StatisticsDialog::StatisticsDialog(const Sudoku& game, const Sudoku& riddle, const Sudoku& solution)
 : game(game),
   riddle(riddle),
   solution(solution)
{
  SetExitAction(GetClosedAction());
}

StatisticsDialog::~StatisticsDialog()
{
  // No code
}

Lum::Object* StatisticsDialog::GetContent()
{
  Lum::Grid       *grid;
  Lum::Label      *label;
  Lum::Tab        *tab;
  Sudoku:: Weight weight;

  riddle.CalculateWeight(solution,weight);

  tab=Lum::Tab::Create(true,true);

  grid=new Lum::Grid();
  grid->SetFlex(true,true);
  grid->SetSpace(true,true);
  grid->SetSize(9,2);
  grid->SetEqualDimensions(true,false);

  for (size_t i=1; i<=9; i++) {
    size_t count=0;

    for (size_t j=0; j<Sudoku::size; j++) {
      Sudoku::Bitset allowed;

      if (game.Get(j)==i) {
        game.GetPotential(j,allowed);

        if (allowed.test(i)) {
          count++;
        }
      }
    }

    grid->SetObject(i-1,0,GetPercentBar(count));
  }

  grid->SetObject(0,1,GetText(L"1"));
  grid->SetObject(1,1,GetText(L"2"));
  grid->SetObject(2,1,GetText(L"3"));
  grid->SetObject(3,1,GetText(L"4"));
  grid->SetObject(4,1,GetText(L"5"));
  grid->SetObject(5,1,GetText(L"6"));
  grid->SetObject(6,1,GetText(L"7"));
  grid->SetObject(7,1,GetText(L"8"));
  grid->SetObject(8,1,GetText(L"9"));
  tab->Add(_(L"POINTS_DISTRIBUTION",L"Distribution"),grid);

  label=Lum::Label::Create(true,true);

  label->AddLabel(_(L"POINTS_FILLED_FIELDS",L"Filled fields:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.filledFields),
                                    Lum::Text::right,
                                    true,false));
  label->AddLabel(_(L"POINTS_DIRECT_FILLS",L"Direct fills:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.directFills),
                                    Lum::Text::right,
                                    true,false));
  label->AddLabel(_(L"POINTS_CROSSHATCH",L"Cross hatch:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.crossHatch),
                                    Lum::Text::right,
                                    true,false));
  label->AddLabel(_(L"POINTS_SINGLE_CAND",L"Single candidate:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.singleCandidate),
                                    Lum::Text::right,
                                    true,false));
  label->AddLabel(_(L"POINTS_SINGLE_SQUARE",L"Single square:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.singleSquare),
                                    Lum::Text::right,
                                    true,false));
  label->AddLabel(_(L"POINTS_GUESS",L"Guesses:"),
                  Lum::Text::Create(Lum::Base::NumberToWString(weight.guesses),
                                    Lum::Text::right,
                                    true,false));

  tab->Add(_(L"POINTS_RATING",L"Rating"),label);

  return tab;
}

void StatisticsDialog::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
}

void StatisticsDialog::Show(Lum::OS::Window* parent,
                            const Sudoku& game,
                            const Sudoku& riddle,
                            const Sudoku& solution)
{
  StatisticsDialog *dialog;

  dialog=new StatisticsDialog(game,riddle,solution);
  dialog->SetParent(parent);
  dialog->SetTitle(_(L"POINTS_TITLE",L"Statistics"));

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  delete dialog;
}

