/* This file is part of effing
 *
 * Copyright (C) 2010 Richard Merren <richard@merren.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "fbactions.h"
#include "logindialog.h"
#include "constants.h"
#include "facebookgraph/facebookgraph.h"
#include "configdialog.h"
#include <QUrl>
#include <QList>
#include <QMessageBox>
#include <QVariantList>
#include <QVariantMap>
#include <QVariant>
#include <QDebug>
#include <QtScript/QScriptEngine>

FbActions::FbActions(QObject *parent) :
    QObject(parent)
{
    connect(&access_manager,SIGNAL(finished(QNetworkReply*)),this,SLOT(handle_http_response(QNetworkReply*)));
}

void FbActions::request_status()
{
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        emit_simple_message(tr("Not logged in to Facebook"));
    } else {
        emit_simple_message(tr("Requesting status from Facebook..."));
        QUrl url;
        url.setScheme("https");
        url.setHost(AUTH_HOST);
        url.setPath("/me/posts");
        url.addQueryItem("access_token",token);
//        url.addQueryItem("limit","10");
//        url.addQueryItem("fields","id,message,updated_time,comments");
        qDebug()<<"about to request url: "<<url.toString();
        QNetworkRequest req(url);
        QNetworkReply* reply = access_manager.get(req);
        status_replies.append(reply);
    }
}

QString FbActions::get_auth_token()
{
    QString auth_token = settings_manager.get_facebook_auth();
    if (auth_token.isNull() || auth_token.isEmpty()) {
        LoginDialog login_dialog;
        int result = login_dialog.exec();
        if (result==QDialog::Accepted) {
            auth_token = login_dialog.get_auth_token();
            settings_manager.set_facebook_auth(auth_token);
        } else {
            auth_token = QString::null;
        }
    }
    return auth_token;
}

void FbActions::handle_http_response(QNetworkReply *reply)
{
    if (reply->error()==QNetworkReply::NoError) {
        QByteArray bytes = reply->readAll();
        qDebug()<<"http response: "<<bytes;
        if (status_replies.contains(reply))
        {
            handle_status_response(bytes);
            status_replies.removeAll(reply);
        }
        else if (set_status_replies.contains(reply))
        {
            handle_set_status_response(bytes);
            set_status_replies.removeAll(reply);
        }
        else
        {
            emit_simple_message(tr("Unknown response from Facebook."));
        }
    } else {
        emit_simple_message(QString(tr("Network error: %1")).arg(reply->errorString()));
    }
    reply->deleteLater();
}

void FbActions::handle_status_response(QByteArray bytes)
{
    FacebookGraph facebookGraph(0,bytes);
    if (facebookGraph.getParseResult() == FacebookGraph::MESSAGES)
        emit status_available(QVariant(facebookGraph.getFormattedMessage()));
    else
        emit_simple_message(tr("Unknown response from Facebook."));
}

void FbActions::handle_set_status_response(QByteArray bytes)
{
    FacebookGraph facebookGraph(0,bytes);
    if (facebookGraph.getParseResult() == FacebookGraph::STATUS_SUCCESSFUL)
    {
        emit status_successfully_set();
        request_status();
    }
    else
        emit_simple_message(tr("Unknown response from Facebook."));
}

void FbActions::unauthorize()
{
    settings_manager.clear_facebook_auth();
    emit_simple_message(tr("Facebook Authorization Removed."));
}

void FbActions::set_status(QString status)
{
    if (status.isEmpty()) return;
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        emit_simple_message(tr("Not logged in to Facebook."));
        return;
    }
    emit_simple_message(tr("Sending status to Facebook..."));
    QUrl url;
    url.setScheme("https");
    url.setHost(AUTH_HOST);
    url.setPath("/me/feed");
    qDebug()<<"about to set status with url: "<<url.toString();
    QNetworkRequest req(url);
    QString encodedStatus(QUrl::toPercentEncoding(status,"",""));
    QString post_data = QString("access_token=%1&message=%2").arg(token,encodedStatus);
    qDebug()<<"using post data: "<<post_data;
    QNetworkReply* reply = access_manager.post(req,post_data.toAscii());
    set_status_replies.append(reply);
}

void FbActions::emit_simple_message(QString text)
{
    emit status_available(QVariant(text));
}

QDateTime FbActions::parseTime(QScriptValue jsonNode)
{
    QDateTime time;
    if (jsonNode.isDate())
    {
        time = jsonNode.toDateTime();
    }
    else
    {
        //The toDateTime function does not seem to properly parse this date time
        QString timestamp_str = jsonNode.toString();
        time = QDateTime::fromString(timestamp_str,Qt::ISODate);
        //QT doesn't seem to like the time zone data, so we set it explicitly to UTC and then convert to local.
        time.setTimeSpec(Qt::UTC);
        time = time.toLocalTime();
    }
    return time;
}

void FbActions::do_configuration()
{
    ConfigDialog config_dialog;
    if (settings_manager.is_prefer_g_rated_tag_line())
        config_dialog.set_tagline_G(false);
    else
        config_dialog.set_tagline_non_G(true);

    if (settings_manager.is_prefer_facebook_touch())
        config_dialog.set_target_touch(true);
    else
        config_dialog.set_target_full(true);

    QObject::connect(&config_dialog,SIGNAL(do_unauthorize()),this,SLOT(unauthorize()));

    int result = config_dialog.exec();

    if (result==QDialog::Accepted)
    {
        settings_manager.set_prefer_g_rated_tag_line(config_dialog.is_tagline_G());
        settings_manager.set_prefer_facebook_touch(config_dialog.is_target_touch());
    }
}


void FbActions::do_not_implemented()
{
    QMessageBox messageBox;
    messageBox.setText(tr("This feature has not yet been implemented"));
    messageBox.exec();
}
