#!/usr/bin/python

"""
@todo Add logging support to make debugging random user issues a lot easier
"""

from __future__ import with_statement


import sys
import gc
import os
import threading
import warnings
import ConfigParser
import socket

import gtk
import gtk.glade

try:
	import hildon
except ImportError:
	hildon = None

import gtk_toolbox


socket.setdefaulttimeout(10)


class DoneIt(object):

	__pretty_app_name__ = "DoneIt"
	__app_name__ = "doneit"
	__version__ = "0.3.0"
	__app_magic__ = 0xdeadbeef

	_glade_files = [
		'/usr/lib/doneit/doneit.glade',
		os.path.join(os.path.dirname(__file__), "doneit.glade"),
		os.path.join(os.path.dirname(__file__), "../lib/doneit.glade"),
	]

	_user_data = os.path.expanduser("~/.%s/" % __app_name__)
	_user_settings = "%s/settings.ini" % _user_data

	def __init__(self):
		self._todoUIs = {}
		self._todoUI = None
		self._osso = None
		self._deviceIsOnline = True
		self._connection = None
		self._fallbackUIName = ""
		self._defaultUIName = ""

		for path in self._glade_files:
			if os.path.isfile(path):
				self._widgetTree = gtk.glade.XML(path)
				break
		else:
			self.display_error_message("Cannot find doneit.glade")
			gtk.main_quit()
		try:
			os.makedirs(self._user_data)
		except OSError, e:
			if e.errno != 17:
				raise

		self._clipboard = gtk.clipboard_get()
		self.__window = self._widgetTree.get_widget("mainWindow")
		self.__errorDisplay = gtk_toolbox.ErrorDisplay(self._widgetTree)

		self._app = None
		self._isFullScreen = False
		if hildon is not None:
			self._app = hildon.Program()
			self.__window = hildon.Window()
			self._widgetTree.get_widget("mainLayout").reparent(self.__window)
			self._app.add_window(self.__window)
			self._widgetTree.get_widget("usernameentry").set_property('hildon-input-mode', 7)
			self._widgetTree.get_widget("passwordentry").set_property('hildon-input-mode', 7|(1 << 29))

			gtkMenu = self._widgetTree.get_widget("mainMenubar")
			menu = gtk.Menu()
			for child in gtkMenu.get_children():
				child.reparent(menu)
			self.__window.set_menu(menu)
			gtkMenu.destroy()

			self.__window.connect("key-press-event", self._on_key_press)
			self.__window.connect("window-state-event", self._on_window_state_change)
		else:
			pass # warnings.warn("No Hildon", UserWarning, 2)

		callbackMapping = {
			"on_doneit_quit": self._on_close,
			"on_paste": self._on_paste,
			"on_about": self._on_about_activate,
		}
		self._widgetTree.signal_autoconnect(callbackMapping)

		if self.__window:
			if hildon is None:
				self.__window.set_title("%s" % self.__pretty_app_name__)
			self.__window.connect("destroy", self._on_close)
			self.__window.show_all()

		backgroundSetup = threading.Thread(target=self._idle_setup)
		backgroundSetup.setDaemon(True)
		backgroundSetup.start()

	def _idle_setup(self):
		# Barebones UI handlers
		import null_view
		with gtk_toolbox.gtk_lock():
			nullView = null_view.GtkNull(self._widgetTree)
			self._todoUIs[nullView.name()] = nullView
			self._todoUI = nullView
			self._todoUI.enable()
			self._fallbackUIName = nullView.name()

		# Setup maemo specifics
		try:
			import osso
		except ImportError:
			osso = None
		self._osso = None
		if osso is not None:
			self._osso = osso.Context(DoneIt.__app_name__, DoneIt.__version__, False)
			device = osso.DeviceState(self._osso)
			device.set_device_state_callback(self._on_device_state_change, 0)
		else:
			pass # warnings.warn("No OSSO", UserWarning, 2)

		try:
			import conic
		except ImportError:
			conic = None
		self._connection = None
		if conic is not None:
			self._connection = conic.Connection()
			self._connection.connect("connection-event", self._on_connection_change, self.__app_magic__)
			self._connection.request_connection(conic.CONNECT_FLAG_NONE)
		else:
			pass # warnings.warn("No Internet Connectivity API ", UserWarning)

		# Setup costly backends
		import rtm_view
		with gtk_toolbox.gtk_lock():
			rtmView = rtm_view.GtkRtMilk(self._widgetTree, self.__errorDisplay)
		self._todoUIs[rtmView.name()] = rtmView
		self._defaultUIName = rtmView.name()

		config = ConfigParser.SafeConfigParser()
		config.read(self._user_settings)
		with gtk_toolbox.gtk_lock():
			self.load_settings(config)

	def display_error_message(self, msg):
		"""
		@note UI Thread
		"""
		error_dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE, msg)

		def close(dialog, response, editor):
			editor.about_dialog = None
			dialog.destroy()
		error_dialog.connect("response", close, self)
		error_dialog.run()

	def load_settings(self, config):
		"""
		@note UI Thread
		"""
		for todoUI in self._todoUIs.itervalues():
			try:
				todoUI.load_settings(config)
			except ConfigParser.NoSectionError, e:
				warnings.warn(
					"Settings file %s is missing section %s" % (
						self._user_settings,
						e.section,
					),
					stacklevel=2
				)

		try:
			activeUIName = config.get(self.__pretty_app_name__, "active")
		except ConfigParser.NoSectionError, e:
			activeUIName = ""
			warnings.warn(
				"Settings file %s is missing section %s" % (
					self._user_settings,
					e.section,
				),
				stacklevel=2
			)

		try:
			self._switch_ui(activeUIName)
		except KeyError, e:
			self._switch_ui(self._defaultUIName)

	def save_settings(self, config):
		"""
		@note Thread Agnostic
		"""
		config.add_section(self.__pretty_app_name__)
		config.set(self.__pretty_app_name__, "active", self._todoUI.name())

		for todoUI in self._todoUIs.itervalues():
			todoUI.save_settings(config)

	def _switch_ui(self, uiName):
		"""
		@note UI Thread
		"""
		newActiveUI = self._todoUIs[uiName]
		try:
			newActiveUI.login()
		except RuntimeError:
			return # User cancelled the operation

		self._todoUI.disable()
		self._todoUI = newActiveUI
		self._todoUI.enable()

		if uiName != self._fallbackUIName:
			self._defaultUIName = uiName

	def _save_settings(self):
		"""
		@note Thread Agnostic
		"""
		config = ConfigParser.SafeConfigParser()
		self.save_settings(config)
		with open(self._user_settings, "wb") as configFile:
			config.write(configFile)

	def _on_device_state_change(self, shutdown, save_unsaved_data, memory_low, system_inactivity, message, userData):
		"""
		For system_inactivity, we have no background tasks to pause

		@note Hildon specific
		"""
		if memory_low:
			gc.collect()

		if save_unsaved_data or shutdown:
			self._save_settings()

	def _on_connection_change(self, connection, event, magicIdentifier):
		"""
		@note Hildon specific
		"""
		import conic

		status = event.get_status()
		error = event.get_error()
		iap_id = event.get_iap_id()
		bearer = event.get_bearer_type()

		if status == conic.STATUS_CONNECTED:
			self._deviceIsOnline = True
			self._switch_ui(self._defaultUIName)
		elif status == conic.STATUS_DISCONNECTED:
			self._deviceIsOnline = False
			self._switch_ui(self._fallbackUIName)

	def _on_window_state_change(self, widget, event, *args):
		"""
		@note Hildon specific
		"""
		if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
			self._isFullScreen = True
		else:
			self._isFullScreen = False

	def _on_close(self, *args, **kwds):
		try:
			if self._osso is not None:
				self._osso.close()

			self._save_settings()
		finally:
			gtk.main_quit()

	def _on_paste(self, *args):
		pass

	def _on_key_press(self, widget, event, *args):
		"""
		@note Hildon specific
		"""
		if event.keyval == gtk.keysyms.F6:
			if self._isFullScreen:
				self.__window.unfullscreen()
			else:
				self.__window.fullscreen()

	def _on_logout(self, *args):
		self._todoUI.logout()
		self._switch_ui(self._fallbackUIName)

	def _on_about_activate(self, *args):
		dlg = gtk.AboutDialog()
		dlg.set_name(self.__pretty_app_name__)
		dlg.set_version(self.__version__)
		dlg.set_copyright("Copyright 2008 - LGPL")
		dlg.set_comments("")
		dlg.set_website("")
		dlg.set_authors([""])
		dlg.run()
		dlg.destroy()


def run_doctest():
	import doctest

	failureCount, testCount = doctest.testmod()
	if not failureCount:
		print "Tests Successful"
		sys.exit(0)
	else:
		sys.exit(1)


def run_doneit():
	gtk.gdk.threads_init()

	if hildon is not None:
		gtk.set_application_name(DoneIt.__pretty_app_name__)
	handle = DoneIt()
	gtk.main()


class DummyOptions(object):

	def __init__(self):
		self.test = False


if __name__ == "__main__":
	if len(sys.argv) > 1:
		try:
			import optparse
		except ImportError:
			optparse = None

		if optparse is not None:
			parser = optparse.OptionParser()
			parser.add_option("-t", "--test", action="store_true", dest="test", help="Run tests")
			(commandOptions, commandArgs) = parser.parse_args()
	else:
		commandOptions = DummyOptions()
		commandArgs = []

	if commandOptions.test:
		run_doctest()
	else:
		run_doneit()
