/*
 *  datetoday home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  László Tari <n900nseries@gmail.com>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libintl.h>
#include <locale.h>
#include <calendar-backend/CalInterface.h>
#include <libhildondesktop/libhildondesktop.h>
#include <clockd/libtime.h>
#include "datetoday-home-widget.h"
#include <math.h>
	
HD_DEFINE_PLUGIN_MODULE(DateTodayPlugin, date_today_plugin, HD_TYPE_HOME_PLUGIN_ITEM)


#define DATE_TODAY_SETTINGS_FILE "/.datetoday_home_widget"
#define HOME_DIR g_get_home_dir()
#define HUGE_SHEET_ICON "/usr/share/pixmaps/datetoday/cal_huge.png"
#define LONG_SHEET_ICON "/usr/share/pixmaps/datetoday/cal_long.png"
#define SHORT_SHEET_ICON "/usr/share/pixmaps/datetoday/cal.png"
#define SHEET_HEIGHT 214
#define SHORT_SHEET_WIDTH 207
#define LONG_SHEET_WIDTH 405
#define HUGE_SHEET_WIDTH 658

const gchar* date_formats[] = 
{
  "wdgt_va_day_numeric",
  "wdgt_va_week_number",
  "wdgt_va_month_name_short",
  "wdgt_va_date_short",
  "wdgt_va_week",
  "wdgt_va_month",
  "wdgt_va_date_day_name_short",
  "wdgt_va_date",
  "wdgt_va_fullmonth_year",
  "wdgt_va_fulldate_day_name_short",
  "wdgt_va_date_medium",
  "wdgt_va_date_long",
};

static void
_load_and_render_background_pixbuf(DateTodayPlugin* desktop_plugin)
{
  GdkPixbuf* pixbuf;
  if(desktop_plugin->background_pixbuf)
    g_object_unref(desktop_plugin->background_pixbuf);
  
  if(desktop_plugin->sheet_size == 0)
  {
    pixbuf = 
	gdk_pixbuf_new_from_file(SHORT_SHEET_ICON, NULL);
  }
  else if(desktop_plugin->sheet_size == 1)
  {
    pixbuf = 
      gdk_pixbuf_new_from_file(LONG_SHEET_ICON, NULL);
  }
  else
  {
    pixbuf = 
      gdk_pixbuf_new_from_file(HUGE_SHEET_ICON, NULL);
  }
  cairo_t* cr;
  int img_width = gdk_pixbuf_get_width(pixbuf);
  int img_height = gdk_pixbuf_get_height(pixbuf);
  cairo_surface_t* surface = cairo_image_surface_create_for_data(gdk_pixbuf_get_pixels(pixbuf),
								 CAIRO_FORMAT_RGB24,
								 img_width, img_height,
								 gdk_pixbuf_get_rowstride(pixbuf));
  cr = cairo_create(surface);
  cairo_set_source_rgba(cr, 0.05, 0.05, 0.05, 1.0);
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  PangoLayout* layout;
  PangoFontDescription* desc;
  
  layout = pango_cairo_create_layout(cr);
  pango_layout_set_text(layout, desktop_plugin->today_string, -1);
  desc = pango_font_description_from_string("Sans Bold 55");
  pango_layout_set_font_description(layout ,desc);
  pango_font_description_free(desc);
  int text_width = 0;
  int text_height = 0;
  pango_layout_get_pixel_size(layout, &text_width, &text_height);
  if((img_width - 48)<text_width)
  {
    double scale_factor = ((double)(img_width - 48)/text_width);
    cairo_scale(cr, scale_factor, scale_factor);
    cairo_move_to(cr, 
		  (img_width/scale_factor - text_width)/2.0,
		  (img_height/scale_factor - text_height)/2.0);
  }
  else
  {
    cairo_move_to(cr, (img_width - text_width)/2.0, (img_height - text_height)/2.0);
  }
  pango_cairo_show_layout(cr, layout);
  g_object_unref(layout); 
  cairo_surface_destroy(surface);
  cairo_destroy(cr);
  desktop_plugin->background_pixbuf = pixbuf;
}

gboolean
date_today_update_day(gpointer data)
{
  DateTodayPlugin* desktop_plugin = DATE_TODAY_PLUGIN(data);
  time_t time = time_get_time();
  struct tm* date_time = localtime(&time);
  int date_day = date_time->tm_mday;
  if(date_day != desktop_plugin->current_day)
  {
    desktop_plugin->current_day = date_day;
    gchar date_buf[256];
    strftime(date_buf, 255, dgettext("hildon-libs", date_formats[desktop_plugin->date_format]), date_time);
    if(desktop_plugin->today_string)
      g_free(desktop_plugin->today_string);
    desktop_plugin->today_string = g_strdup(date_buf);
    _load_and_render_background_pixbuf(desktop_plugin);
    gtk_widget_queue_draw(desktop_plugin->content);
    return FALSE;
  }
}

static GtkWidget*
build_ui()
{
  GtkVBox *contents = GTK_VBOX(gtk_vbox_new(0, FALSE));
  gtk_widget_show_all(GTK_WIDGET(contents));
  gtk_widget_add_events(GTK_WIDGET(contents), GDK_BUTTON_PRESS_MASK);
  return GTK_WIDGET(contents);
}

static void
date_today_on_current_desktop(GtkWidget* widget,
			      gpointer data)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
  {
    date_today_update_day(desktop_plugin);
  }
}

void
save_settings(DateTodayPlugin* desktop_plugin)
{
  GKeyFile *keyFile;
  gchar *fileData;
  FILE *iniFile;
  gsize size;
  gchar *filename;

  keyFile = g_key_file_new();
  filename = g_strconcat (HOME_DIR, DATE_TODAY_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);

  g_key_file_set_integer(keyFile, desktop_plugin->iD, "launchView", desktop_plugin->launch_view);
  g_key_file_set_integer(keyFile, desktop_plugin->iD, "size", desktop_plugin->size);
  g_key_file_set_integer(keyFile, desktop_plugin->iD, "date_format", desktop_plugin->date_format);

  fileData = g_key_file_to_data (keyFile, &size, NULL);

  g_file_set_contents(filename, fileData, size, NULL);
  g_key_file_free (keyFile);
  g_free (fileData);
  g_free (filename);
}

static GtkListStore*
_create_format_list(DateTodayPlugin* desktop_plugin)
{
  GtkListStore* list_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
  GtkTreeIter iter;
  gchar date_buf[256];
  time_t time = time_get_time();
  struct tm* date_time = localtime(&time);
  int i = 0;
  for(i = 0; i<12;++i)
  {
    strftime(date_buf, 255, dgettext("hildon-libs", date_formats[i]), date_time);
    gtk_list_store_append(list_store, &iter);
    gtk_list_store_set(list_store, &iter,
		       0, date_buf,
		       1, i,
		       -1);
  }
  return list_store;
}

static void
date_today_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  GtkWidget *dialog = gtk_dialog_new_with_buttons("Settings",
						  NULL,
						  0,
						  "Save", 
						  GTK_RESPONSE_ACCEPT, 
						  NULL);
  GtkWidget* content = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
  GtkSizeGroup *group = GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));
  GtkWidget *selector = hildon_touch_selector_new_text();
  GtkWidget *sizeBox = gtk_hbox_new (FALSE, 0);
  GtkWidget* sizeScale = gtk_hscale_new_with_range(50, 200, 10);
  GtkWidget *sizeLabel = gtk_label_new ("");
  GtkWidget *launch_view_button;
  GtkWidget* date_format_picker_button;
  GtkWidget* date_format = hildon_touch_selector_new_text();
  GtkListStore* format_list_store = _create_format_list (desktop_plugin);

  hildon_touch_selector_set_model(HILDON_TOUCH_SELECTOR(date_format), 0, GTK_TREE_MODEL(format_list_store));
  date_format_picker_button = hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT,
						       HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(date_format_picker_button),
				    HILDON_TOUCH_SELECTOR(date_format));

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(date_format), 0, desktop_plugin->date_format);

  gtk_label_set_markup (GTK_LABEL(sizeLabel), "Size");
  gtk_range_set_value(GTK_RANGE(sizeScale), desktop_plugin->size);
  
  gtk_box_pack_start (GTK_BOX (sizeBox), sizeScale, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (sizeBox), sizeLabel, FALSE, FALSE, 0);

  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "month");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "week");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "agenda");

  launch_view_button = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, 
						 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(selector), 0, desktop_plugin->launch_view -1);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (launch_view_button),
                                     HILDON_TOUCH_SELECTOR (selector));
  hildon_button_set_title(HILDON_BUTTON(launch_view_button), "Calendar View");

  gtk_container_add(GTK_CONTAINER(content), launch_view_button);
  gtk_container_add(GTK_CONTAINER(content), date_format_picker_button);
  gtk_container_add(GTK_CONTAINER(content), sizeBox);

  gtk_widget_show_all(dialog);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) 
  {
    desktop_plugin->launch_view = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0) + 1;
    desktop_plugin->size = gtk_range_get_value(GTK_RANGE(sizeScale));
    desktop_plugin->date_format = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(date_format), 0);
    if(desktop_plugin->date_format < 4)
    {
      desktop_plugin->sheet_size = 0;
    }
    else if(desktop_plugin->date_format < 10)
    {
      desktop_plugin->sheet_size = 1;
    }
    else
    {
      desktop_plugin->sheet_size = 2;
    }

    int width = SHORT_SHEET_WIDTH;
    int height = desktop_plugin->size;
    if(desktop_plugin->sheet_size == 1)
      width = LONG_SHEET_WIDTH;
    else if(desktop_plugin->sheet_size == 2)
      width = HUGE_SHEET_WIDTH;
    width *= (double)height/SHEET_HEIGHT;
    desktop_plugin->current_day = 0;
    date_today_update_day(desktop_plugin);

    gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), width, height);
    gtk_window_resize(GTK_WINDOW(desktop_plugin),
		      width,
		      height);
    gtk_widget_queue_draw(desktop_plugin->content);
    save_settings(desktop_plugin);
  }
  g_object_unref(format_list_store);
  gtk_widget_destroy (dialog);
}

static void 
read_settings(DateTodayPlugin *desktop_plugin)
{
  GKeyFile *keyFile;
  gchar* fileName;
  gboolean fileExists;
  keyFile = g_key_file_new();
  fileName = g_strconcat(HOME_DIR, DATE_TODAY_SETTINGS_FILE, NULL);
  fileExists = g_key_file_load_from_file (keyFile, fileName, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(fileExists)
  {
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "launchView", NULL))
      desktop_plugin->launch_view = g_key_file_get_integer(keyFile, desktop_plugin->iD, "launchView", NULL);
    else if(g_key_file_has_key(keyFile, "config", "launchView", NULL))
      desktop_plugin->launch_view = g_key_file_get_integer(keyFile, "config", "launchView", NULL);
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "size", NULL))
      desktop_plugin->size = g_key_file_get_integer(keyFile, desktop_plugin->iD, "size", NULL);
    else if(g_key_file_has_key(keyFile, "config", "size", NULL))
      desktop_plugin->size = g_key_file_get_integer(keyFile, "config", "size", NULL);
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "date_format", NULL))
      desktop_plugin->date_format = g_key_file_get_integer(keyFile, desktop_plugin->iD, "date_format", NULL);
    else if(g_key_file_has_key(keyFile, "config", "date_format", NULL))
      desktop_plugin->date_format = g_key_file_get_integer(keyFile, "config", "date_format", NULL);
  }
  if(desktop_plugin->date_format < 4)
  {
    desktop_plugin->sheet_size = 0;
  }
  else if(desktop_plugin->date_format < 10)
  {
    desktop_plugin->sheet_size = 1;
  }
  else
  {
    desktop_plugin->sheet_size = 2;
  }
  g_key_file_free(keyFile);
  g_free(fileName);
}

static void
date_today_plugin_init(DateTodayPlugin *desktop_plugin)
{
  GtkWidget *contents = build_ui();
  desktop_plugin->content = contents;
  desktop_plugin->sheet_size = 0;
  desktop_plugin->background_pixbuf = NULL;
  desktop_plugin->size = 100;
  desktop_plugin->launch_view = 1;
  desktop_plugin->today_string = NULL;
  desktop_plugin->current_day = 0;
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(date_today_on_current_desktop), NULL);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(date_today_show_settings_dialog), NULL);
  //  register_dbus_signal_on_time_change(desktop_plugin);
}

static void
datetoday_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  DateTodayPlugin* desktop_plugin = DATE_TODAY_PLUGIN(widget);
  desktop_plugin->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  read_settings(desktop_plugin);
  date_today_update_day(desktop_plugin);
  int width = SHORT_SHEET_WIDTH;
  int height = desktop_plugin->size;
  if(desktop_plugin->sheet_size == 1)
    width = LONG_SHEET_WIDTH;
  else if(desktop_plugin->sheet_size == 2)
    width = HUGE_SHEET_WIDTH;

  width *= (double)height/SHEET_HEIGHT;
  gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), width, height);
  GTK_WIDGET_CLASS(date_today_plugin_parent_class)->realize(widget);
}

static gboolean
datetoday_expose(GtkWidget* widget, GdkEventExpose *event)
{
  DateTodayPlugin* desktop_plugin = DATE_TODAY_PLUGIN(widget);
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);

  int height = desktop_plugin->size;
  cairo_scale(cr, (double)height/SHEET_HEIGHT, (double)height/SHEET_HEIGHT);
  if(desktop_plugin->background_pixbuf)
  {
    gdk_cairo_set_source_pixbuf(cr, desktop_plugin->background_pixbuf, 0, 0);
    cairo_paint(cr);
  }
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(date_today_plugin_parent_class)->expose_event(widget, event);
}

static gint
button_press(GtkWidget* widget, GdkEventButton *event)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  DBusGConnection *dbus_conn = hd_home_plugin_item_get_dbus_g_connection ( &desktop_plugin->hitem, 
									   DBUS_BUS_SESSION, 
									   NULL );


  DBusGProxy *cal_proxy = dbus_g_proxy_new_for_name(dbus_conn,
						    CALENDAR_SERVICE,
						    CALENDAR_PATH,
						    CALENDAR_INTERFACE);
  int type = desktop_plugin->launch_view; 
  time_t tt = time(NULL); 
  int msgID = tt; 
  const char* text = "DUMMY";  
  if(cal_proxy)
  {
    dbus_g_proxy_call_no_reply(cal_proxy, 
			       CALENDAR_LAUNCH_METHOD, 
			       G_TYPE_UINT, type, 
			       G_TYPE_INT, msgID, 
			       G_TYPE_STRING, text,
			       DBUS_TYPE_INVALID); 
  }
  g_object_unref(cal_proxy);
  dbus_g_connection_unref(dbus_conn);  
  return FALSE;
}

static void
date_today_plugin_finalize(GObject *object)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(object);
  if(desktop_plugin->today_string)
  {
    g_free(desktop_plugin->today_string);
  }
  g_free(desktop_plugin->iD);
  G_OBJECT_CLASS (date_today_plugin_parent_class)->finalize (object);
}

static void
date_today_plugin_class_init(DateTodayPluginClass *klass) 
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = datetoday_realize;
  widget_class->expose_event = datetoday_expose;
//Define button_release_event instead of button_press_event to avoid calendar application execution during scrolling between desktops
  //widget_class->button_press_event = button_press;
  widget_class->button_release_event = button_press;
//------------------------------------------------------------------------------------------------László Tari <n900nseries@gmail.com>
  G_OBJECT_CLASS(klass)->finalize = date_today_plugin_finalize;
}

static void
date_today_plugin_class_finalize(DateTodayPluginClass *class) 
{
}
