# -*- coding: utf-8 -*-

"""
Module implementing CutetubeWindows.
"""
from PyQt4.QtGui import *
from PyQt4.QtCore import *
from PyQt4.QtMaemo5 import QMaemo5InformationBox
from Ui_cutetubewindows import Ui_CutetubeWindows

from videoinfowindow import VideoInfoWindow
import customwidgets
import youtubeservice
import cutetubedialogs
from cutetubeconfig import GeneralSettings, YouTubeAPISettings
from cutetubelog import CutetubeLog
from videoplaybackwindow import VideoPlaybackWindow

class CutetubeWindows(QMainWindow, Ui_CutetubeWindows):
    """
    Abstract class for video-list windows.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QMainWindow.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5StackedWindow)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, GeneralSettings.autoOrientation)
        self.lineFilter = QLineEdit(self.toolBar)
        self.toolBar.addWidget(self.lineFilter)
        closeFilter = QAction(self.toolBar)
        closeIcon = QIcon(":/images/ui-images/closeicon.png")
        closeIcon.actualSize(QSize(72, 66))
        closeFilter.setIcon(closeIcon)
        self.toolBar.addAction(closeFilter)
        self.toolBar.setVisible(False)
        self.listWidget.keyPressEvent = self.filterList
        self.listWidget.setContextMenuPolicy(Qt.ActionsContextMenu)
        queueLength = len(youtubeservice.YouTubeQueueFeed.playlist)
        if queueLength > 0:
            self.actionGoToQueue.setVisible(True)
            self.actionGoToQueue.setText("Queue (%d videos)" % queueLength)
        taskQueueLength = len(youtubeservice.YouTubeVideoDownloader.taskQueue)
        if taskQueueLength > 0:
            self.actionDownloads.setVisible(True)
            self.actionDownloads.setText("Downloads (%d videos)" % taskQueueLength)
        self.connect(closeFilter, SIGNAL("triggered()"), self.closeFilter)
        self.connect(self.actionAddToQueue, SIGNAL("triggered()"), self.addToQueue)
        self.connect(self.actionAddToPlaylist, SIGNAL("triggered()"), self.addToPlaylist)
        self.connect(self.actionAddToFavourites, SIGNAL("triggered()"), self.addToFavourites)
        self.connect(self.actionDeleteVideo, SIGNAL("triggered()"), self.confirmVideoDelete)
        self.connect(self.actionEditVideo, SIGNAL("triggered()"), self.editVideo)
        self.connect(self.moreButton, SIGNAL("pressed()"), self.changeButtonStyleSheet)
        self.connect(self.moreButton, SIGNAL("released()"), self.changeButtonStyleSheet)
        self.orientationChanged()
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged) 
        
    def changeButtonStyleSheet(self):
        """
        Setting the QToolButton style sheets in the conventional 
        way caused trouble with the Maemo specific widgets, so 
        I'm using a signal instead.
        """
        button = self.sender()
        if button.isDown():
            self.moreButton.setStyleSheet("font-size: 18px; border-radius: 4px; background-color: rgb(120, 120, 120, 150)")
        else:
            self.moreButton.setStyleSheet("font-size: 18px; border-radius: 4px; background-color: rgb(0, 0, 0, 100)")
            
    def orientationChanged(self):
        """
        Sets the position of the 'more' button according to 
        the screen orientation.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.moreButton.setGeometry(QRect(402, 670, 85, 80))
        else:
            self.moreButton.setGeometry(QRect(722, 352, 85, 80))

    def filterList(self, event):
        """
        Shows the toolbar containing a lineEdit 
        and filters the listWidgetItems according 
        to the string entered by the user.
        """
        if event.key() not in (Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down, Qt.Key_Backspace, Qt.Key_Return, Qt.Key_Control):
            self.toolBar.setVisible(True)
            self.lineFilter.insert(event.text())
        if event.key() == Qt.Key_Backspace:
            self.lineFilter.backspace()
        for row in range(self.listWidget.count()):
            item = self.listWidget.item(row)
            itemWidget = self.listWidget.itemWidget(item)
            title = itemWidget.titleLabel.text()
            author = itemWidget.authorLabel.text()
            if not title.contains(self.lineFilter.text(), Qt.CaseInsensitive) and not author.contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setRowHidden(row, True)
            else:
                self.listWidget.setRowHidden(row, False)
            if self.lineFilter.text() == "":
                self.toolBar.setVisible(False) 

    def closeFilter(self):
        self.lineFilter.setText("")
        for row in range(self.listWidget.count()):
            self.listWidget.setRowHidden(row, False)
        self.toolBar.setVisible(False)

    def toggleBusy(self, isBusy):
        """
        Shows the progress indicator if argument is True, 
        and hides it otherwise.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)

    def addToQueue(self):
        """
        Adds the video to the user's playback queue.
        """
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        videoTitle = video.media.title.text
        try:
            youtubeservice.YouTubeQueueFeed.playlist.append(video)
            QMaemo5InformationBox.information(self, "\'%s\' added to your playback queue" % unicode(videoTitle, "utf-8"))
            queueLength = len(youtubeservice.YouTubeQueueFeed.playlist)
            self.queueChanged(queueLength)
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to your playback queue")
        self.listWidget.setCurrentRow(-1)

    def addToPlaylist(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        addToPlaylistDialog = cutetubedialogs.AddToPlaylistDialog(self, [video])
        self.listWidget.setCurrentRow(-1)

    def addToFavourites(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        videoId = video.media.video_id.text
        try:
            addedToFavourites = youtubeservice.YouTubeDataService.addVideoToFavourites(videoId)
            if addedToFavourites == True:
                QMaemo5InformationBox.information(self, "\'%s\' has been added to your favourites" % unicode(video.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, addedToFavourites)
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to your favourites")
        self.listWidget.setCurrentRow(-1)

    def confirmVideoDelete(self):
        self.row = self.listWidget.currentRow()
        self.video = self.yt.videoList[self.row]
        confirmDeleteDialog = cutetubedialogs.ConfirmDeleteDialog(self, self.video.media.title.text)
        self.connect(confirmDeleteDialog, SIGNAL("deleteConfirmed()"), self.deleteVideo)
        self.listWidget.setCurrentRow(-1)

    def deleteVideo(self):
        try:
            self.yt.deleteVideo(self.video.id.text)
            self.reloadData()
            QMaemo5InformationBox.information(self, "\'%s\' has been deleted from your channel" % unicode(video.media.title.text, "utf-8"))
        except:
            QMaemo5InformationBox.information(self, "Unable to delete video from your channel")
        self.listWidget.setCurrentRow(-1)

    def editVideo(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        editVideoDialog = cutetubedialogs.EditVideoDialog(self, video)
        self.listWidget.setCurrentRow(-1)
        self.connect(editVideoDialog, SIGNAL("accepted()"), self.reloadData)

    def loadData(self):
        try:
            self.yt.start()
        except:
            try:
                QMaemo5InformationBox.information(self, "Unable to connect to YouTube, please try again")
            except:
                pass

        
    def getMoreResults(self, scrollBarPos):
        """
        Shows the 'more' button to enable the user to 
        get more results.
        """
        if self.yt.isFinished() and self.yt.moreResults and scrollBarPos > (self.yt.results - 10) * 94:
            self.moreButton.setVisible(True)
        else:
            self.moreButton.setVisible(False)
            
    @pyqtSignature("")
    def on_moreButton_clicked(self):
        """
        Appends more results to the list.
        """
        self.loadData()
        self.moreButton.setVisible(False)

    def reloadData(self):
        """
        Refreshes the listWidget following a change 
        to the data.
        """
        self.listWidget.clear()
        self.closeFilter()
        self.yt.resetResultCount()
        self.loadData()

    def addVideo(self, videoData):
        """
        Appends a video item to the listWidget.
        """
        size = QSize()
        size.setHeight(94)
        item = QListWidgetItem(self.listWidget)
        item.setSizeHint(size)
        itemWidget = customwidgets.VideoItemWidget(self.listWidget, videoData)
        if self.actionPencil.isChecked():
            itemWidget.showCheckBox()
            item.setFlags(Qt.ItemIsEnabled)
        self.listWidget.addItem(item)
        self.listWidget.setItemWidget(item, itemWidget)
        title = itemWidget.titleLabel.text()
        author = itemWidget.authorLabel.text()
        if not title.contains(self.lineFilter.text(), Qt.CaseInsensitive) and not author.contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setItemHidden(item, True)
                
    def addVideoThumbnail(self, videoData):
        """
        Adds the video thumbnail to the appropriate video item in the listWidget.
        """
        row, thumbnail = videoData
        item = self.listWidget.item(row)
        thumbLabel = self.listWidget.itemWidget(item).thumbLabel
        try:
            thumbLabel.setPixmap(thumbnail) 
        except:
            thumbLabel.setPixmap(QPixmap(":/images/ui-images/thumbnail.png"))

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Toggles the item's checkstate if the pencil action 
        is enabled, otherwise a VideoInfoWindow is created.
        """
        if self.actionPencil.isChecked():
            checkBox = self.listWidget.itemWidget(item).checkBox
            checkBox.toggle()
            if checkBox.isChecked():
                self.pencilList.append(self.listWidget.row(item))
                self.actionPencilQueue.setVisible(True)
                self.actionPencilDownloads.setVisible(True)
                if youtubeservice.YouTubeDataService.currentUser != "":
                    self.actionPencilPlaylist.setVisible(True)
                    self.actionPencilFavourites.setVisible(True)
            else:
                self.pencilList.remove(self.listWidget.row(item))
                if len(self.pencilList) == 0:
                    self.actionPencilQueue.setVisible(False)
                    self.actionPencilDownloads.setVisible(False)
                    self.actionPencilPlaylist.setVisible(False)
                    self.actionPencilFavourites.setVisible(False)           
        else:
            row = self.listWidget.row(item)
            video = self.yt.videoList[row]
            videoInfoWindow = VideoInfoWindow(self, video)
            self.connect(videoInfoWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
            self.connect(videoInfoWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
            self.listWidget.setCurrentRow(-1)
            self.actionPencil.setChecked(False)
            self.on_actionPencil_toggled(False)
            self.closeFilter()
            self.pencilList = []
            
            
        
    @pyqtSignature("")
    def on_actionSubscribe_triggered(self):
        """
        Adds a subscription to the user's account.
        """
        try:
            subscribed = youtubeservice.YouTubeDataService.addSubscription(self.username)
            if subscribed == True:
                QMaemo5InformationBox.information(self, "You have subscribed to \'%s\'!" % unicode(self.username, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, subscribed)
        except:
            QMaemo5InformationBox.information(self, "Unable to add subscription")
        
    @pyqtSignature("")
    def on_actionSearch_triggered(self):
        """
        Raises a SearchDialog.
        """
        searchDialog = cutetubedialogs.SearchDialog(self)

    @pyqtSignature("")
    def on_actionNewSubscriptionVideos_triggered(self):
        """
        Displays a feed of new videos from the user's 
        subscriptions.
        """
        newSubscriptionVideosWindow = NewSubscriptionVideosWindow(self)

    @pyqtSignature("")
    def on_actionAddComment_triggered(self):
        """
        Raises a dialog asking the user to enter and 
        submit a comment for the current video.
        """
        addCommentDialog = cutetubedialogs.AddCommentDialog(self, self.video)
        self.connect(addCommentDialog, SIGNAL("accepted()"), self.reloadData)

    @pyqtSignature("")
    def on_actionGoToQueue_triggered(self):
        """
        Creates a PlaybackQueueWindow.
        """
        playbackQueueWindow = PlaybackQueueWindow(self)
        self.connect(playbackQueueWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(playbackQueueWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.actionPencil.setChecked(False)
        self.on_actionPencil_toggled(False)
        
    @pyqtSignature("")
    def on_actionDownloads_triggered(self):
        """
        Creates a DownloadsWindow
        """
        downloadsWindow = DownloadsWindow(self)
        self.connect(downloadsWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(downloadsWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.actionPencil.setChecked(False)
        self.on_actionPencil_toggled(False)

    @pyqtSignature("bool")
    def on_actionPencil_toggled(self, isChecked):
        """
        Enables multiple selection of listWidgetItems.
        """
        if isChecked:
            for row in range(self.listWidget.count()):
                item = self.listWidget.item(row)
                item.setFlags(Qt.ItemIsEnabled)
                widget = self.listWidget.itemWidget(item)
                widget.showCheckBox()
        else:
            self.actionPencilQueue.setVisible(False)
            self.actionPencilDownloads.setVisible(False)
            self.actionPencilPlaylist.setVisible(False)
            self.actionPencilFavourites.setVisible(False)
            for row in range(self.listWidget.count()):
                item = self.listWidget.item(row)
                item.setFlags(Qt.ItemIsEnabled | Qt.ItemIsSelectable)
                widget = self.listWidget.itemWidget(item)
                widget.hideCheckBox()
            self.listWidget.setCurrentRow(-1)

    @pyqtSignature("")
    def on_actionPencilQueue_triggered(self):
        """
        Adds all checked videos to the user's playback queue.
        """
        try:
            for row in self.pencilList:  
                video = self.yt.videoList[row]              
                youtubeservice.YouTubeQueueFeed.playlist.append(video)
            QMaemo5InformationBox.information(self, "Your chosen videos have been added to your playback queue")
            queueLength = len(youtubeservice.YouTubeQueueFeed.playlist)
            self.queueChanged(queueLength)
        except:
            QMaemo5InformationBox.information(self, "Unable to add videos to your playback queue")
        self.actionPencil.toggle()
            
    @pyqtSignature("")
    def on_actionPencilDownloads_triggered(self):
        """
        Adds all checked videos to the user's download queue.
        """
        try:
            for row in self.pencilList:  
                video = self.yt.videoList[row]              
                youtubeservice.YouTubeVideoDownloader.addTask(video)
            QMaemo5InformationBox.information(self, "Your chosen videos have been added to your download queue")
            taskQueueLength = len(youtubeservice.YouTubeVideoDownloader.taskQueue)
            self.downloadsChanged(taskQueueLength)
        except:
            QMaemo5InformationBox.information(self, "Unable to add videos to your download queue")
        self.actionPencil.toggle()

    @pyqtSignature("")
    def on_actionPencilPlaylist_triggered(self):
        """
        Adds all checked videos to the user's chosen playlist.
        """
        videoList = []
        for row in self.pencilList:          
            videoList.append(self.yt.videoList[row])
        addToPlaylistDialog = cutetubedialogs.AddToPlaylistDialog(self, videoList)
        self.actionPencil.toggle()

    @pyqtSignature("")
    def on_actionPencilFavourites_triggered(self):
        """
        Adds all checked videos to the user's favourites.
        """
        try:
            self.toggleBusy(True)
            for row in self.pencilList:
                videoId = self.yt.videoList[row].media.video_id.text
                youtubeservice.YouTubeDataService.addVideoToFavourites(videoId)
            QMaemo5InformationBox.information(self, "Your chosen videos have been added to your favourites")
        except:
            QMaemo5InformationBox.information(self, "Unable to add videos to your favourites")
        self.toggleBusy(False)
        self.actionPencil.toggle()

    def queueChanged(self, queueLength):
        """
        Makes actionGoToQueue visible if there are 
        videos in the user's playback queue.
        """
        if queueLength > 0:
            self.actionGoToQueue.setVisible(True)
            self.actionGoToQueue.setText("Queue (%d videos)" % queueLength)
        else:
            self.actionGoToQueue.setVisible(False)
        self.emit(SIGNAL("queueChanged(int)"), queueLength)
        
    def downloadsChanged(self,  taskQueueLength):
        """
        Makes actionDownloads visible if there are videos 
        in the user's download queue.
        """
        if taskQueueLength > 0:
            self.actionDownloads.setVisible(True)
            self.actionDownloads.setText("Downloads (%d videos)" % taskQueueLength)
        else:
            self.actionDownloads.setVisible(False)
        self.emit(SIGNAL("downloadsChanged(int)"), taskQueueLength)

class MostRecentWindow(CutetubeWindows):
    """
    Displays videos from the 'Most Recent' feed.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Most Recent")
        self.actionAddToQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.pencilList = []
        self.show()
        location = YouTubeAPISettings.locationFilter
        category = YouTubeAPISettings.categoryFilter
        self.videoFeed = "http://gdata.youtube.com/feeds/api/standardfeeds/%smost_recent%s" % (location, category)
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)   
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()
        

class MostViewedWindow(CutetubeWindows):
    """
    Displays videos from the 'Most Viewed' feed.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Today's Most Viewed")
        self.actionAddToQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.pencilList = []
        self.show()
        location = YouTubeAPISettings.locationFilter
        category = YouTubeAPISettings.categoryFilter
        self.videoFeed = "http://gdata.youtube.com/feeds/api/standardfeeds/%smost_viewed%s?time=today" % (location, category)
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)        
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()
        
class SearchResultsWindow(CutetubeWindows):
    """
    Displays videos matching the search string entered
    by the user.
    """
    def __init__(self, parent, searchString, searchOrder, username = None):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Search Results")
        self.actionAddToQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.searchString = searchString
        self.pencilList = []
        self.info = False
        self.show() 
        searchTerms = self.searchString.replace(" ", "+")
        if username == None:
            self.videoFeed = "http://gdata.youtube.com/feeds/api/videos?q=%s&safeSearch=none&orderby=%s" % (searchTerms, searchOrder)
        else:
            self.videoFeed = "http://gdata.youtube.com/feeds/api/videos?author=%s&q=%s&safeSearch=none&orderby=%s" % (username, searchTerms, searchOrder)
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusyandCheckResults)       
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults) 
        self.loadData()

    def addVideo(self, videoData):
        """
        Overrides the method in CutetubeWindows to inform 
        the user of search results.
        """
        size = QSize()
        size.setHeight(94)
        item = QListWidgetItem()
        item.setSizeHint(size)
        itemWidget = customwidgets.VideoItemWidget(self.listWidget, videoData)
        if self.actionPencil.isChecked():
            itemWidget.showCheckBox()
            item.setFlags(Qt.ItemIsEnabled)
        self.listWidget.addItem(item)
        self.listWidget.setItemWidget(item, itemWidget)
        title = itemWidget.titleLabel.text()
        author = itemWidget.authorLabel.text()
        if not self.info:
            QMaemo5InformationBox.information(self,  "Showing results for \'%s\'" % self.searchString)
            self.info = True
                
    def toggleBusyandCheckResults(self, isBusy):
        """
        Hides the progress indicator and informs the user if there are no results.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)
        if not (self.info or isBusy):
            if self.yt.results == 0:
                QMaemo5InformationBox.information(self,  "No results found for \'%s\'" % self.searchString)
            self.info == True

class UploadsWindow(CutetubeWindows):
    """
    Displays the user's video uploads.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("My Uploads")
        #self.actionNewUpload.setVisible(True)
        self.actionSearch.setVisible(True)
        self.actionSearch.setText("Search my uploads")
        self.actionAddToQueue.setVisible(True)
        self.actionAddToFavourites.setVisible(True)
        self.actionAddToPlaylist.setVisible(True)
        self.actionEditVideo.setVisible(True)
        self.actionDeleteVideo.setVisible(True)
        self.actionPencil.setVisible(True)
        self.pencilList = []
        self.show()
        self.videoFeed = "http://gdata.youtube.com/feeds/api/users/default/uploads"
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)  
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)      
        self.loadData()
        
    @pyqtSignature("")
    def on_actionSearch_triggered(self):
        """
        Raises a SearchDialog.
        """
        currentUser = unicode(youtubeservice.YouTubeDataService.currentUser) # currentUser is a QString - YouTube API requires unicode.
        searchDialog = cutetubedialogs.SearchDialog(self,  currentUser)

class UserWindow(CutetubeWindows):
    """
    Displays the specified user's videos.
    """
    def __init__(self, parent, username, subscribed = False):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("%s\'s Videos" % username)
        self.username = username
        if not subscribed:
            self.actionSubscribe.setVisible(True)
            self.actionSubscribe.setText("Subscribe to \'%s\'" % unicode(self.username, "utf-8"))
        self.actionSearch.setVisible(True)
        self.actionSearch.setText("Search user's videos")
        self.actionAddToQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.pencilList = []
        self.show()
        self.videoFeed = "http://gdata.youtube.com/feeds/api/users/%s/uploads" % self.username
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()
        
    @pyqtSignature("")
    def on_actionSearch_triggered(self):
        """
        Raises a SearchDialog.
        """
        searchDialog = cutetubedialogs.SearchDialog(self,  self.username)

class RelatedVideosWindow(CutetubeWindows):
    """
    Displays a related videos feed.
    """
    def __init__(self, parent, video):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Related Videos")
        self.actionAddToQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.video = video
        self.pencilList = []
        self.show()
        for link in video.link:
            if link.rel.split("#")[-1] == "video.related":
                self.videoFeed = link.href
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)        
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()

class FavouritesWindow(CutetubeWindows):
    """
    Displays the user's favourite videos.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("My Favourites")
        self.actionAddToQueue.setVisible(True)
        self.actionAddToPlaylist.setVisible(True)
        self.actionRemoveFromFavourites.setVisible(True)
        self.actionPencil.setVisible(True)
        self.actionPencilFavourites.setVisible(False)
        self.connect(self.actionRemoveFromFavourites, SIGNAL("triggered()"), self.removeFromFavourites)        
        self.pencilList = []
        self.show()
        self.videoFeed = "http://gdata.youtube.com/feeds/api/users/default/favorites"
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy) 
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()

    def removeFromFavourites(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        videoId = video.media.video_id.text
        try:
            removedFromFavourites = youtubeservice.YouTubeDataService.removeVideoFromFavourites(videoId)
            if removedFromFavourites == True:
                self.reloadData()
                QMaemo5InformationBox.information(self, "\'%s\' has been removed from your favourites" % unicode(video.media.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, removedFromFavourites)
        except:
            QMaemo5InformationBox.information(self, "Unable to remove video from your favourites")
        self.listWidget.setCurrentRow(-1)

class PlaylistsWindow(CutetubeWindows):
    """
    Displays a list of the user's playlists.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("My Playlists")
        self.actionNewPlaylist.setVisible(True)
        self.actionEditPlaylist.setVisible(True)
        self.actionDeletePlaylist.setVisible(True)
        self.connect(self.actionEditPlaylist, SIGNAL("triggered()"), self.editPlaylist)
        self.connect(self.actionDeletePlaylist, SIGNAL("triggered()"), self.deletePlaylist)        
        self.show()
        self.yt = youtubeservice.YouTubePlaylistFeed(self)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.loadPlaylists)
        self.loadData()
        self.toggleBusy(True)

    def filterList(self, event):
        """
        Shows the toolbar containing a lineEdit 
        and filters the listWidgetItems according 
        to the string entered by the user.
        """
        if event.key() not in (Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down, Qt.Key_Backspace, Qt.Key_Return, Qt.Key_Control):
            self.toolBar.setVisible(True)
            self.lineFilter.insert(event.text())
        if event.key() == Qt.Key_Backspace:
            self.lineFilter.backspace()
        for row in range(self.listWidget.count()):
            if not self.listWidget.item(row).text().contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setRowHidden(row, True)
            else:
                self.listWidget.setRowHidden(row, False)
        if self.lineFilter.text() == "":
            self.toolBar.setVisible(False)

    def loadPlaylists(self, playlists):
        for playlist in playlists:
            item = QListWidgetItem(playlist.title.text)
            item.setTextAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
            self.listWidget.addItem(item)
        self.toggleBusy(False)

    def editPlaylist(self):
        row = self.listWidget.currentRow()
        playlist = self.yt.playlistFeed.entry[row]
        editPlaylistDialog = cutetubedialogs.EditPlaylistDialog(self, playlist)
        self.listWidget.setCurrentRow(-1)
        self.connect(editPlaylistDialog, SIGNAL("accepted()"), self.reloadData)

    def deletePlaylist(self):
        row = self.listWidget.currentRow()
        playlist = self.yt.playlistFeed.entry[row]
        playlistId = playlist.playlist_id.text
        try:
            playlistDeleted = youtubeservice.YouTubeDataService.deletePlaylist(playlistId)
            if playlistDeleted == True:
                self.listWidget.setRowHidden(row, True)
                QMaemo5InformationBox.information(self, "Your playlist: \'%s\' has been deleted" % unicode(playlist.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, playlistDeleted)
        except:
            QMaemo5InformationBox.information(self, "Unable to delete your playlist")
        self.listWidget.setCurrentRow(-1)
        
    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Raises a PlaylistVideosWindow.
        """
        row = self.listWidget.row(item)
        playlistName = self.yt.playlistFeed.entry[row].title.text
        playlistId = self.yt.playlistFeed.entry[row].playlist_id.text
        playlistVideosWindow = PlaylistVideosWindow(self, playlistName, playlistId)
        self.connect(playlistVideosWindow,  SIGNAL("queueChanged(int)"),  self.queueChanged)
        self.connect(playlistVideosWindow,  SIGNAL("downloadsChanged(int)"),  self.downloadsChanged)
        self.listWidget.setCurrentRow(-1) 
        self.closeFilter()

    @pyqtSignature("")
    def on_actionNewPlaylist_triggered(self):
        """
        Raises a NewPlaylistDialog.
        """
        newPlaylistDialog = cutetubedialogs.NewPlaylistDialog(self)
        self.connect(newPlaylistDialog, SIGNAL("accepted()"), self.reloadData)
        
    @pyqtSignature("")
    def on_actionGoToQueue_triggered(self):
        """
        Creates a PlaybackQueueWindow.
        """
        playbackQueueWindow = PlaybackQueueWindow(self)
        self.connect(playbackQueueWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(playbackQueueWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        
    @pyqtSignature("")
    def on_actionDownloads_triggered(self):
        """
        Creates a DownloadsWindow
        """
        downloadsWindow = DownloadsWindow(self)
        self.connect(downloadsWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(downloadsWindow, SIGNAL("queueChanged(int)"), self.queueChanged)

class PlaylistVideosWindow(CutetubeWindows):
    """
    Displays the videos from the chosen playlist.
    """
    def __init__(self, parent, playlistName, playlistId):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.playlistName = playlistName
        self.playlistId = playlistId
        self.setWindowTitle(self.playlistName)
        self.actionPlayPlaylist.setVisible(True)
        self.actionSearch.setVisible(True)
        self.actionSearch.setText("Find more of \'%s\'" % self.playlistName)
        self.actionAddToQueue.setVisible(True)
        self.actionAddToFavourites.setVisible(True)
        self.actionAddToPlaylist.setVisible(True)
        self.actionRemoveFromPlaylist.setVisible(True)
        self.actionEditVideo.setVisible(True)
        self.actionPencil.setVisible(True)
        self.connect(self.actionRemoveFromPlaylist, SIGNAL("triggered()"), self.removeFromPlaylist)
        self.pencilList = []
        self.show()
        self.videoFeed = "http://gdata.youtube.com/feeds/api/playlists/%s" % playlistId
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()

    def editVideo(self):
        row = self.listWidget.currentRow()
        playlistVideo = self.yt.videoList[row]
        position = row + 1
        count = len(self.yt.videoList)
        editPlaylistVideoDialog = cutetubedialogs.EditPlaylistVideoDialog(self, self.playlistId, playlistVideo, position, count)
        self.listWidget.setCurrentRow(-1)
        self.connect(editPlaylistVideoDialog, SIGNAL("accepted()"), self.reloadData)

    def addToPlaylist(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        addToPlaylistDialog = cutetubedialogs.AddToPlaylistDialog(self, video)
        self.listWidget.setCurrentRow(-1)

    def removeFromPlaylist(self):
        row = self.listWidget.currentRow()
        video = self.yt.videoList[row]
        videoId = self.yt.videoList[row].id.text.split(":")[-1]
        try:
            removedFromPlaylist = youtubeservice.YouTubeDataService.removeVideoFromPlaylist(self.playlistId, videoId)
            if removedFromPlaylist == True:
                self.reloadData()
                QMaemo5InformationBox.information(self, "\'%s\' has been removed from your playlist" % unicode(video.media.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, removedFromPlaylist)
        except:
            QMaemo5InformationBox.information(self, "Unable to remove video from your playlist")
        self.listWidget.setCurrentRow(-1)

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Toggles the item's checkstate if the pencil action 
        is enabled, otherwise a VideoInfoWindow is created.
        """
        if self.actionPencil.isChecked():
            checkBox = self.listWidget.itemWidget(item).checkBox
            checkBox.toggle()
            if checkBox.isChecked():
                self.pencilList.append(self.listWidget.row(item))
                self.actionPencilQueue.setVisible(True)
                self.actionPencilDownloads.setVisible(True)
                self.actionPencilPlaylist.setVisible(True)
                self.actionPencilFavourites.setVisible(True)
            else:
                self.pencilList.remove(self.listWidget.row(item))
                if len(self.pencilList) == 0:
                    self.actionPencilQueue.setVisible(False)
                    self.actionPencilDownloads.setVisible(False)
                    self.actionPencilPlaylist.setVisible(False)
                    self.actionPencilFavourites.setVisible(False)           
        else:
            row = self.listWidget.currentRow()
            playlistVideo = self.yt.videoList[row]
            position = row + 1
            count = len(self.yt.videoList)
            videoInfoWindow = VideoInfoWindow(self, playlistVideo, self.playlistId, position, count)
            self.connect(videoInfoWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
            self.connect(videoInfoWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
            self.listWidget.setCurrentRow(-1)
            self.actionPencil.setChecked(False)
            self.on_actionPencil_toggled(False)
            self.pencilList = []
            self.closeFilter()

    @pyqtSignature("")
    def on_actionSearch_triggered(self):
        """
        Performs a search for videos matching the 
        playlist title.
        """
        searchOrder = "published"
        findMoreWindow = SearchResultsWindow(self, self.playlistName,  searchOrder)
        
    @pyqtSignature("")
    def on_actionPlayPlaylist_triggered(self):
        """
        Plays the entire playlist passed to the 
        media player.
        """
        try:
            videoPlaybackWindow = VideoPlaybackWindow(self, playlist = self.yt.videoList)
        except:
            QMaemo5InformationBox.information(self, "Unable to play playlist")

class PlaybackQueueWindow(CutetubeWindows):
    """
    Displays the user's playback queue.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("My Playback Queue")
        self.actionPlayQueue.setVisible(True)
        self.actionClearQueue.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionSaveQueue.setVisible(True)
            self.actionAddToFavourites.setVisible(True)
            self.actionAddToPlaylist.setVisible(True)
        self.actionRemoveFromQueue.setVisible(True)
        self.actionGoToQueue.setVisible(False)
        self.connect(self.actionRemoveFromQueue, SIGNAL("triggered()"), self.removeFromQueue)
        self.show()
        self.yt = youtubeservice.YouTubeQueueFeed(self)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)
        self.loadData()
        
    def addVideo(self, videoData):
        """
        Overrides the method in CutetubeWindows.
        """
        size = QSize()
        size.setHeight(94)
        item = QListWidgetItem()
        item.setSizeHint(size)
        itemWidget = customwidgets.VideoItemWidget(self.listWidget, videoData)
        self.listWidget.addItem(item)
        self.listWidget.setItemWidget(item, itemWidget)
        title = itemWidget.titleLabel.text()
        author = itemWidget.authorLabel.text()
        if not title.contains(self.lineFilter.text(), Qt.CaseInsensitive) and not author.contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setItemHidden(item, True)
        
    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Raises a VideoInfoWindow.
        """
        row = self.listWidget.row(item)
        video = youtubeservice.YouTubeQueueFeed.playlist[row]
        videoInfoWindow = VideoInfoWindow(self, video)
        self.connect(videoInfoWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.listWidget.setCurrentRow(-1)
        self.closeFilter()            

    def queueChanged(self, queueLength):
        """
        Overrides the method in ChannelWindow.
        """
        pass

    def addToPlaylist(self):
        row = self.listWidget.currentRow()
        video = youtubeservice.YouTubeQueueFeed.playlist[row]
        addToPlaylistDialog = cutetubedialogs.AddToPlaylistDialog(self, [video])
        self.listWidget.setCurrentRow(-1)

    def addToFavourites(self):
        row = self.listWidget.currentRow()
        video = youtubeservice.YouTubeQueueFeed.playlist[row]
        videoId = video.media.video_id.text
        try:
            addedToFavourites = youtubeservice.YouTubeDataService.addVideoToFavourites(videoId)
            if addedToFavourites == True:
                QMaemo5InformationBox.information(self, "\'%s\' has been added to your favourites" % unicode(video.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, addedToFavourites)
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to your favourites")
        self.listWidget.setCurrentRow(-1)

    def removeFromQueue(self):
        """
        Removes the video from the user's playback queue.
        """
        row = self.listWidget.currentRow()
        video = youtubeservice.YouTubeQueueFeed.playlist[row]
        videoTitle = video.media.title.text
        try:
            youtubeservice.YouTubeQueueFeed.playlist.remove(video)
            self.listWidget.setRowHidden(row, True)
            QMaemo5InformationBox.information(self, "\'%s\' has been removed from your playback queue" % unicode(videoTitle, "utf-8"))
            queueLength = len(youtubeservice.YouTubeQueueFeed.playlist)
            if queueLength == 0:
                self.actionClearQueue.setVisible(False)
                self.actionSaveQueue.setVisible(False)
                self.actionPlayQueue.setVisible(False)
            self.emit(SIGNAL("queueChanged(int)"), queueLength)
        except:
            QMaemo5InformationBox.information(self, "Unable to remove video from your playback queue")
        self.listWidget.setCurrentRow(-1)
        
    @pyqtSignature("")
    def on_actionPlayQueue_triggered(self):
        """
        Plays the full playback queue.
        """
        try:
            playlist = youtubeservice.YouTubeQueueFeed.playlist
            videoPlaybackWindow = VideoPlaybackWindow(self, playlist = playlist)
        except:
            QMaemo5InformationBox.information(self, "Unable to play your playback queue")

    @pyqtSignature("")
    def on_actionClearQueue_triggered(self):
        """
        Clears the user's playback queue and 
        the listWidget.
        """
        try:
            youtubeservice.YouTubeQueueFeed.playlist = []
            self.listWidget.clear()
            QMaemo5InformationBox.information(self, "Your playback queue has been cleared")
            self.actionClearQueue.setVisible(False)
            self.actionSaveQueue.setVisible(False)
            self.actionPlayQueue.setVisible(False)
            queueLength = len(youtubeservice.YouTubeQueueFeed.playlist)
            self.emit(SIGNAL("queueChanged(int)"), queueLength)
        except:
            QMaemo5InformationBox.information(self, "Unable to clear your playback queue")

    @pyqtSignature("")
    def on_actionSaveQueue_triggered(self):
        """
        Saves the current playback queue as a 
        YouTube playlist.
        """
        videoQueue = youtubeservice.YouTubeQueueFeed.playlist
        saveQueueAsPlaylistDialog = cutetubedialogs.NewPlaylistDialog(self, videoQueue)


class SubscriptionsWindow(CutetubeWindows):
    """
    Displays a list of the user's subscriptions.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("My Subscriptions")
        self.actionUnsubscribe.setVisible(True)
        self.actionNewSubscriptionVideos.setVisible(True)
        self.connect(self.actionUnsubscribe, SIGNAL("triggered()"), self.removeSubscription)
        self.show()
        self.yt = youtubeservice.YouTubeSubscriptionsFeed(self)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.loadSubscriptions)
        self.loadData()    
        self.toggleBusy(True)

    def removeSubscription(self):
        row = self.listWidget.currentRow()
        subscription = self.yt.subscriptionsFeed.entry[row]
        try:
            unsubscribed = youtubeservice.YouTubeDataService.removeSubscription(subscription.id.text)
            if unsubscribed == True:
                self.listWidget.setRowHidden(row, True)
                QMaemo5InformationBox.information(self, "You have unsubscribed to \'%s\'" % unicode(subscription.username.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self, unsubscribed) 
        except:
            QMaemo5InformationBox.information(self, "Unable to remove your subscription")    
        self.listWidget.setCurrentRow(-1)

    def filterList(self, event):
        """
        Shows the toolbar containing a lineEdit 
        and filters the listWidgetItems according 
        to the string entered by the user.
        """
        if event.key() not in (Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down, Qt.Key_Backspace, Qt.Key_Return, Qt.Key_Control):
            self.toolBar.setVisible(True)
            self.lineFilter.insert(event.text())
        if event.key() == Qt.Key_Backspace:
            self.lineFilter.backspace()
        for row in range(self.listWidget.count()):
            if not self.listWidget.item(row).text().contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setRowHidden(row, True)
            else:
                self.listWidget.setRowHidden(row, False)
        if self.lineFilter.text() == "":
            self.toolBar.setVisible(False) 

    def loadSubscriptions(self, subscriptionsList):
        for subscription in subscriptionsList:
            item = QListWidgetItem(subscription.username.text)
            item.setTextAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
            self.listWidget.addItem(item)
        self.toggleBusy(False)

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Raises a UserWindow.
        """
        row = self.listWidget.row(item)
        subscriptionName = self.yt.subscriptionsFeed.entry[row].username.text
        subscriptionVideosWindow = UserWindow(self, subscriptionName, True)
        self.connect(subscriptionVideosWindow,  SIGNAL("queueChanged(int)"),  self.queueChanged)
        self.connect(subscriptionVideosWindow,  SIGNAL("downloadsChanged(int)"),  self.downloadsChanged)
        self.listWidget.setCurrentRow(-1) 
        self.closeFilter()
        
    @pyqtSignature("")
    def on_actionGoToQueue_triggered(self):
        """
        Creates a PlaybackQueueWindow.
        """
        playbackQueueWindow = PlaybackQueueWindow(self)
        self.connect(playbackQueueWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(playbackQueueWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        
    @pyqtSignature("")
    def on_actionDownloads_triggered(self):
        """
        Creates a DownloadsWindow
        """
        downloadsWindow = DownloadsWindow(self)
        self.connect(downloadsWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(downloadsWindow, SIGNAL("queueChanged(int)"), self.queueChanged)


class NewSubscriptionVideosWindow(CutetubeWindows):
    """
    Displays a feed of new videos from the user's 
    subscriptions.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Latest Subscription Videos")
        self.actionAddToFavourites.setVisible(True)
        self.actionAddToPlaylist.setVisible(True)
        self.actionPencil.setVisible(True)
        self.pencilList = []
        self.show()
        try:
            QMaemo5InformationBox.information(self, "Showing latest videos from your subscriptions")
        except:
            pass
        self.videoFeed = "http://gdata.youtube.com/feeds/api/users/default/newsubscriptionvideos"
        self.yt = youtubeservice.YouTubeVideoFeed(self, self.videoFeed)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy) 
        self.connect(self.listWidget.verticalScrollBar(), SIGNAL("valueChanged(int)"), self.getMoreResults)
        self.loadData()

class DownloadsWindow(CutetubeWindows):
    """
    Displays the user's download queue.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Downloads")
        self.actionRemoveFromQueue.setVisible(True)
        self.actionRemoveFromQueue.setText("Cancel download")
        self.actionClearQueue.setVisible(True)
        self.actionClearQueue.setText("Clear downloads")
        self.actionPauseDownload.setVisible(True)
        self.actionResumeDownload.setVisible(True)
        self.actionDownloads.setVisible(False)
        self.actionGoToQueue.setVisible(False)
        self.actionResumeDownloads.setVisible(True)
        self.actionPauseDownloads.setVisible(True)
        self.filterGroup = QActionGroup(self)
        self.filterGroup.setExclusive(True)
        self.actionQueued = QAction("Current", self.filterGroup)
        self.actionQueued.setCheckable(True)
        self.actionQueued.setChecked(False)
        self.actionAll = QAction("All", self.filterGroup)
        self.actionAll.setCheckable(True)
        self.actionAll.setChecked(True)
        self.actionCompleted = QAction("Completed", self.filterGroup)
        self.actionCompleted.setCheckable(True)
        self.actionCompleted.setChecked(False)
        self.menuMenu.addActions(self.filterGroup.actions())
        self.connect(self.filterGroup, SIGNAL("selected(QAction*)"), self.filterDownloads)
        self.connect(self.actionRemoveFromQueue,  SIGNAL("triggered()"),  self.cancelDownload)
        self.connect(self.actionPauseDownload,  SIGNAL("triggered()"),  self.pauseDownload)
        self.connect(self.actionResumeDownload,  SIGNAL("triggered()"),  self.resumeDownload)
        self.show()
        self.yt = youtubeservice.YouTubeDownloadQueueFeed(self)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.addVideo)
        self.connect(self.yt, SIGNAL("thumbnailLoaded(PyQt_PyObject)"), self.addVideoThumbnail)
        self.connect(self.yt, SIGNAL("feedCompleted(bool)"), self.toggleBusy)
        self.loadData()
        for task in youtubeservice.YouTubeVideoDownloader.taskQueue:
            self.connect(task, SIGNAL("downloadUpdated()"), self.updateDownloadStatus)
            self.connect(task, SIGNAL("statusChanged()"), self.updateTaskStatus)
            
    def filterDownloads(self, filter):
        """
        Filters the downloads list according to the user's chosen filter.
        """
        if filter is self.actionAll:
            for row in range(self.listWidget.count()):
                self.listWidget.setRowHidden(row, False)
        elif filter is self.actionQueued:
            for row in range(self.listWidget.count()):
                task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
                if task.getStatus() == "completed":
                    self.listWidget.setRowHidden(row, True)
                else:
                    self.listWidget.setRowHidden(row, False)
        else:
            for row in range(self.listWidget.count()):
                task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
                if task.getStatus() == "completed":
                    self.listWidget.setRowHidden(row, False)
                else:
                    self.listWidget.setRowHidden(row, True)
        self.filterMenu(filter)
        self.listWidget.setCurrentRow(-1)
        
    def filterMenu(self, filter):
        """
        Filters the available menu options.
        """
        if filter is self.actionAll:
            for action in (self.actionPauseDownload, self.actionResumeDownload, self.actionRemoveFromQueue, self.actionResumeDownloads, self.actionPauseDownloads, self.actionClearQueue):
                action.setVisible(True)
        elif filter is self.actionQueued:
            for action in (self.actionPauseDownload, self.actionResumeDownload, self.actionRemoveFromQueue, self.actionResumeDownloads, self.actionPauseDownloads):
                action.setVisible(True)
            self.actionClearQueue.setVisible(False)
        else:
            for action in (self.actionPauseDownload, self.actionResumeDownload, self.actionRemoveFromQueue, self.actionResumeDownloads, self.actionPauseDownloads, self.actionClearQueue):
                action.setVisible(False)
        

    def addVideo(self, videoData):
        size = QSize()
        size.setHeight(94)
        item = QListWidgetItem()
        item.setSizeHint(size)
        self.itemWidget = customwidgets.VideoDownloadItemWidget(self.listWidget, videoData)
        self.listWidget.addItem(item)
        self.listWidget.setItemWidget(item, self.itemWidget)
        row = self.listWidget.row(item)
        task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
        self.updateTaskStatus(task)
        
    def filterList(self, event):
        """
        Shows the toolbar containing a lineEdit 
        and filters the listWidgetItems according 
        to the string entered by the user.
        """
        if event.key() not in (Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down, Qt.Key_Backspace, Qt.Key_Return, Qt.Key_Control):
            self.toolBar.setVisible(True)
            self.lineFilter.insert(event.text())
        if event.key() == Qt.Key_Backspace:
            self.lineFilter.backspace()
        for row in range(self.listWidget.count()):
            item = self.listWidget.item(row)
            itemWidget = self.listWidget.itemWidget(item)
            title = itemWidget.titleLabel.text()
            if not title.contains(self.lineFilter.text(), Qt.CaseInsensitive):
                self.listWidget.setRowHidden(row, True)
            else:
                self.listWidget.setRowHidden(row, False)
            if self.lineFilter.text() == "":
                self.toolBar.setVisible(False) 
                
        
    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Toggles the item's checkstate if the pencil action 
        is enabled, otherwise a VideoInfoWindow is created.
        """
        row = self.listWidget.row(item)
        video = youtubeservice.YouTubeVideoDownloader.taskQueue[row].video
        videoInfoWindow = VideoInfoWindow(self, video)
        self.connect(videoInfoWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.listWidget.setCurrentRow(-1)
        self.closeFilter()
            
    @pyqtSignature("")
    def on_actionClearQueue_triggered(self):
        """
        Clears the user's download queue and 
        the listWidget.
        """
        if youtubeservice.YouTubeVideoDownloader.currentStatus == "active":
            QMaemo5InformationBox.information(self,  "Downloads are still in progress")
        else:
            try:
                youtubeservice.YouTubeVideoDownloader.clearTasks()
                self.listWidget.clear()
                QMaemo5InformationBox.information(self, "Your downloads have been cleared")
                self.actionClearQueue.setVisible(False)
                self.actionPlayQueue.setVisible(False)
                self.emit(SIGNAL("downloadsChanged(int)"), 0)
            except:
                QMaemo5InformationBox.information(self, "Unable to clear your downloads")
            
    def cancelDownload(self):
        """
        Cancels the download of the selected video and 
        removes the associated item from the listWidget.
        """
        row = self.listWidget.currentRow()
        youtubeservice.YouTubeVideoDownloader.cancelTask(row)
        task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
        if task.getStatus() != "completed":
            self.updateTaskStatus(task)
        self.listWidget.setCurrentRow(-1)
        
    def pauseDownload(self):
        """
        Pauses the download of the selected video if it's 
        current status is 'active'.
        """
        row = self.listWidget.currentRow()
        youtubeservice.YouTubeVideoDownloader.pauseTask(row)
        task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
        if task.getStatus() != "completed":
            self.updateTaskStatus(task)
        self.listWidget.setCurrentRow(-1)
        
    def resumeDownload(self):
        """
        Sets the status of the video to 'queued'.
        """
        row = self.listWidget.currentRow()
        youtubeservice.YouTubeVideoDownloader.resumeTask(row)
        task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
        if task.getStatus() != "completed":
            self.updateTaskStatus(task)
        self.listWidget.setCurrentRow(-1)
        
    @pyqtSignature("")
    def on_actionResumeDownloads_triggered(self):
        """
        Sets the status of all videos to 'queued'.
        """
        for row in range(self.listWidget.count()):
            youtubeservice.YouTubeVideoDownloader.resumeTask(row)
            task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
            if task.getStatus() != "completed":
                self.updateTaskStatus(task)
            
    @pyqtSignature("")
    def on_actionPauseDownloads_triggered(self):
        """
        Sets the status of all videos to 'paused'.
        """
        for row in range(self.listWidget.count()):
            youtubeservice.YouTubeVideoDownloader.pauseTask(row)
            task = youtubeservice.YouTubeVideoDownloader.taskQueue[row]
            if task.getStatus() != "completed":
                self.updateTaskStatus(task)

    def updateDownloadStatus(self):
        """
        Updates the progress bar of the appropriate video item.
        """
        currentTask = youtubeservice.YouTubeVideoDownloader.currentTask
        row = youtubeservice.YouTubeVideoDownloader.taskQueue.index(currentTask)
        item = self.listWidget.item(row)
        widget = self.listWidget.itemWidget(item)
        try:
            if widget.speedLabel.text() != "completed":
                widget.progressBar.setValue(int(currentTask.progress*100))
                widget.speedLabel.setText("d/l speed: %.2f KB/s" % currentTask.speed)
                if currentTask.progress  == 1.0:
                    QMaemo5InformationBox.information(self,  "Download of \'%s\' is completed" % unicode(currentTask.getTitle(), "utf-8"))
                    widget.speedLabel.setText("completed")
                    self.filterDownloads(self.filterGroup.checkedAction())
        except AttributeError:
            pass
            
    def updateTaskStatus(self,  task = youtubeservice.YouTubeVideoDownloader.currentTask):
        try:
            row = youtubeservice.YouTubeVideoDownloader.taskQueue.index(task)
            item = self.listWidget.item(row)
            widget = self.listWidget.itemWidget(item)
            for status in ("queued",  "paused",  "completed",  "cancelled",  "failed"):
                if task.getStatus() == status:
                    widget.speedLabel.setText(status)
                widget.progressBar.setValue(int(task.progress*100))
        except ValueError:
            pass

class CommentsWindow(CutetubeWindows):
    """
    Displays the comments feed cutetubewindows.pyfor the video.
    """
    def __init__(self, parent, video):
        """
        Constructor
        """
        CutetubeWindows.__init__(self, parent)
        self.setWindowTitle("Comments")
        if youtubeservice.YouTubeDataService.currentUser != "":
            self.actionAddComment.setVisible(True)
        self.actionGoToQueue.setVisible(False)
        self.show()
        self.video = video
        videoId = self.video.media.video_id.text
        self.yt = youtubeservice.YouTubeCommentsFeed(self, videoId)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.loadComments)
        self.loadData()
        self.toggleBusy(True)

    def filterList(self, event):
        """
        Overrides the filterList method of 
        CutetubeWindows.
        """
        pass

    def loadComments(self, comments):
        """
        Loads the comments into the QListWidget.
        """
        if len(comments) < 1:
            item = QListWidgetItem("No comments")
            self.listWidget.addItem(item)  
        else:
            try:
                commentsNumber = unicode(len(comments))
                videoTitle = self.video.title.text
                for comment in comments:
                    item = QListWidgetItem()
                    item.setFlags(Qt.ItemIsEnabled)
                    self.listWidget.addItem(item)
                    commentWidget = customwidgets.CommentWidget(self.listWidget, comment.author[0].name.text, comment.content.text)
                    size = QSize()
                    size.setHeight(commentWidget.height())
                    item.setSizeHint(size)
                    self.listWidget.setItemWidget(item, commentWidget)
            except AttributeError:
                pass
        self.toggleBusy(False)

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Overrides the method in ChannelWindow.
        """
        pass

    def queueChanged(self, queueLength):
        """
        Overrides the method in ChannelWindow.
        """
        pass
        
    def downloadsChanged(self,  taskQueueLength):
        """
        Overrides the method in ChannelWindow.
        """
        pass
