# -*- coding: utf-8 -*-

"""
Module implementing VideoInfoWindow.
"""

from PyQt4.QtGui import *
from PyQt4.QtCore import *
import time
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass
from Ui_videoinfowindow import Ui_VideoInfoWindow

import cutetubedialogs
import cutetubewindows
import youtubeservice
import customwidgets

class VideoInfoWindow(QMainWindow, Ui_VideoInfoWindow):
    """
    This window displays all video information and provides options 
    for the user to play/download the video and add comments etc.
    """
    def __init__(self, parent, video, playlistId = None, position = None, count = None):
        """
        Constructor
        """
        QMainWindow.__init__(self, parent)
        try:
            self.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
            self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        except:
            pass
        self.setupUi(self)
        self.descriptionTextBox.viewport().setAutoFillBackground(False)
        self.titleLabel.viewport().setAutoFillBackground(False)
        self.video = video
        if playlistId != None:
            self.playlistId = playlistId
            self.playlistPosition = position
            self.playlistLength = count
            self.actionEditVideo.setVisible(True)
            self.actionEditVideo.setText("Edit Custom Metadata")
        if self.video.author[0].name.text == youtubeservice.YouTubeDataService.currentUser:
            self.actionRateVideo.setVisible(False)
            self.actionUserVideos.setVisible(False)
            self.actionEditVideo.setVisible(True)
        self.show()
        self.loadData()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.ratingLabel.move(260, 10)
            self.viewsLabel.move(260, 55)
            self.viewsNumberLabel.move(315, 55)
            self.commentsLabel.move(260, 85)
            self.commentNumberLabel.move(315, 90)
            self.titleLabel.setGeometry(QRect(10, 205, 460, 70))
            self.descriptionTextBox.setGeometry(QRect(10, 275, 460, 295))
            self.downloadButton.setGeometry(QRect(10, 580, 460, 70))
            self.playButton.setGeometry(QRect(10, 660, 460, 70))
        else:
            self.titleLabel.setGeometry(QRect(260, 20, 530, 70))
            self.viewsLabel.setGeometry(QRect(10, 260, 50, 30))
            self.ratingLabel.setGeometry(QRect(10, 210, 180, 45))
            self.commentsLabel.setGeometry(QRect(10, 290, 50, 40))
            self.viewsNumberLabel.setGeometry(QRect(70, 260, 180, 25))
            self.commentNumberLabel.setGeometry(QRect(70, 295, 180, 25))
            self.descriptionTextBox.setGeometry(QRect(260, 90, 530, 215))
            self.downloadButton.setGeometry(QRect(10, 340, 340, 70))
            self.playButton.setGeometry(QRect(450, 340, 340, 70))

    def loadData(self):
        """
        Loads the video data into the appropriate widgets.
        """
        self.loadTitleAndDescription()
        self.loadViewsCommentsRating()
        self.loadThumbnail()

    def loadTitleAndDescription(self):
        """
        Loads the title and description into the QTextEdit widgets.
        """
        # Load the title and description. Some videos do not have a description, in which case None is returned.
        self.actionUserVideos.setText("%s\'s Videos" % unicode(self.video.author[0].name.text, "utf-8"))
        self.titleLabel.setText(unicode(self.video.title.text, "utf-8"))       
        try:
            if "playlists" in self.video.id.text:
                self.descriptionTextBox.setText(unicode(self.video.content.text, "utf-8"))
            else:
                self.descriptionTextBox.setText(unicode(self.video.media.description.text, "utf-8"))
        except:
            self.descriptionTextBox.setText("No description")

    def loadViewsCommentsRating(self):
        """
        Loads the view count, comment count and rating 
        into the appropriate widgets.
        """        
        # Load view count and comment count. Some videos do not have a view count, in which case None is returned.
        try:
            self.viewsNumberLabel.setText(self.video.statistics.view_count)
        except:
            self.viewsNumberLabel.setText("--")
        try: 
            self.commentNumberLabel.setText(self.video.comments.feed_link[0].count_hint)
            if self.video.comments.feed_link[0].count_hint == "0":
                self.actionViewComments.setVisible(False)
        except:
            self.commentNumberLabel.setText("0")
            self.actionViewComments.setVisible(False)
        # Load video's average rating. Some videos may not have a rating, in which case it is set to '3 stars'.
        try:
            self.ratingLabel.setPixmap(QPixmap(":/images/ui-images/rating%s.png" % unicode(round(float(self.video.rating.average)))))
        except:
            self.ratingLabel.setPixmap(QPixmap(":/images/ui-images/rating3.0.png"))
                
    def loadThumbnail(self):
        """
        Loads the duration and thumbnail into the durationLabel and thumbLabel.
        """
        # Load video duration, converting seconds to MM:SS format.
        self.durationLabel.setText(unicode(time.strftime('%M:%S', time.gmtime(float(self.video.media.duration.seconds)))))
        
        thumbUrl = self.video.media.thumbnail[3].url
        thumbnail = youtubeservice.YouTubeDataService.getVideoThumbnail(thumbUrl)
        self.thumbLabel.setPixmap(thumbnail) 
    
    def playVideo(self, event = None):
        """
        Plays the video.
        """
        #self.toggleBusy(True)
        videoTitle = self.video.title.text
        try:
            self.ytPlayer = youtubeservice.YouTubeVideoPlayer(self, self.video)
            self.ytPlayer.start()
            #self.connect(self.ytPlayer, SIGNAL("playbackFinshed(bool)"), self.toggleBusy)
            QMaemo5InformationBox.information(self, "Playing \'%s\'" % unicode(videoTitle, "utf-8"))
        except:
            QMaemo5InformationBox.information(self, "Unable to play video")

    def downloadVideo(self):
        videoTitle = self.video.media.title.text
        try:
            youtubeservice.YouTubeVideoDownloader.addTask(self.video)
            QMaemo5InformationBox.information(self, "\'%s\' has been added to your download queue" % unicode(videoTitle, "utf-8"))
            self.emit(SIGNAL("downloadsChanged(int)"), len(youtubeservice.YouTubeVideoDownloader.taskQueue))
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to your download queue")
    
    def toggleBusy(self, isBusy):
        """
        Shows the progress indicator if argument is True, 
        and hides it otherwise.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)
    
    @pyqtSignature("")
    def on_playButton_clicked(self):
        self.playVideo()
    
    @pyqtSignature("")
    def on_downloadButton_clicked(self):
        """
        Adds the current video to the download queue.
        """
        self.downloadVideo()

    @pyqtSignature("")
    def on_actionAddToQueue_triggered(self):
        """
        Adds the video to the user's playback queue.
        """
        videoTitle = self.video.media.title.text
        try:
            youtubeservice.YouTubeVideoPlayer.playlist.append(self.video)
            QMaemo5InformationBox.information(self, "\'%s\' added to playback queue" % unicode(videoTitle, "utf-8"))
            self.emit(SIGNAL("queueChanged(int)"), len(youtubeservice.YouTubeVideoPlayer.playlist))
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to playback queue")
       
    
    @pyqtSignature("")
    def on_actionRateVideo_triggered(self):
        """
        Raises a dialog asking the user to select a 
        rating from 1-5 stars.
        """
        rateVideoDialog = cutetubedialogs.RateVideoDialog(self, self.video.id.text)

    @pyqtSignature("")
    def on_actionAddComment_triggered(self):
        """
        Raises a dialog asking the user to enter and 
        submit a comment for the current video.
        """
        addCommentDialog = cutetubedialogs.AddCommentDialog(self, self.video)
        self.connect(addCommentDialog, SIGNAL("accepted()"), self.on_actionViewComments_triggered)

    @pyqtSignature("")
    def on_actionViewComments_triggered(self):
        """
        Raises a dialog asking the user to enter and 
        submit a comment for the current video.
        """
        commentsWindow = cutetubewindows.CommentsWindow(self, self.video)
    
    @pyqtSignature("")
    def on_actionUserVideos_triggered(self):
        """
        Creates a ChannelWindow displaying all videos 
        from the uploader of the current video.
        """
        userName = self.video.author[0].name.text
        userWindow = cutetubewindows.UserWindow(self, userName)
        self.connect(userWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(userWindow, SIGNAL("queueChanged(int)"), self.queueChanged)

    @pyqtSignature("")
    def on_actionRelatedVideos_triggered(self):
        """
        Creates a ChannelWindow displaying a 
        related videos feed.
        """
        relatedVideosWindow = cutetubewindows.RelatedVideosWindow(self, self.video)
        self.connect(relatedVideosWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(relatedVideosWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
    
    @pyqtSignature("")
    def on_actionVideoResponse_triggered(self):
        """
        Asks the user to choose an existing video from 
        their account, or upload a new video chosen via 
        a file dialog.
        """
        fileDialog = QFileDialog(self, ("Choose Video Response"))
        fileDialog.setFileMode(QFileDialog.ExistingFiles)
        fileDialog.setReadOnly(True)
        if fileDialog.exec_():
            pass
    
    @pyqtSignature("")
    def on_actionAddToFavourites_triggered(self):
        """
        Adds the video to the user's favourites and 
        displays an information box to confirm.
        """
        try:
            youtubeservice.YouTubeDataService.addVideoToFavourites(self.video)
            QMaemo5InformationBox.information(self, "\'%s\' has been added to your favourites" % unicode(self.video.media.title.text, "utf-8"))
        except:
            QMaemo5InformationBox.information(self,  "Unable to add video to your favourites")

    @pyqtSignature("")
    def on_actionAddToPlaylist_triggered(self):
        """
        Raises a dialog asking the user to select a 
        playlist to which to add the current video.
        """
        playlistDialog = cutetubedialogs.AddToPlaylistDialog(self, [self.video])

    @pyqtSignature("")
    def on_actionEditVideo_triggered(self):
        """
        Raises an EditVideoDialog.
        """
        if "playlists" in self.video.id.text:
            editVideoDialog = cutetubedialogs.EditPlaylistVideoDialog(self, self.playlistId, self.video, self.playlistPosition, self.playlistLength)
        else:
            editVideoDialog = cutetubedialogs.EditVideoDialog(self, self.video)
        self.connect(editVideoDialog, SIGNAL("accepted()"), self.loadData)

    @pyqtSignature("")
    def on_ratingButton_clicked(self):
        """
        Raises a dialog asking the user to select a 
        rating from 1-5 stars.
        """
        rateVideoDialog = cutetubedialogs.RateVideoDialog(self, self.video.id.text)
        
    def queueChanged(self, queueLength):
        """
        Connects the the queueChanged signal to the parent window.
        """
        self.emit(SIGNAL("queueChanged(int)"), queueLength)
        
    def downloadsChanged(self,  taskQueueLength):
        """
        Connects the downloadsChanged signal to the parent window.
        """
        self.emit(SIGNAL("downloadsChanged(int)"), taskQueueLength)
        

