/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MEDIAPLAYER_H
#define MEDIAPLAYER_H

#include <QMediaPlayer>
#include <QUrl>
#include "../../base/station.h"
#include "../../base/enums.h"

class StreamExtractor;

class MediaPlayer : public QObject
{
    Q_OBJECT
    Q_PROPERTY(Station* currentStation
               READ currentStation
               WRITE setCurrentStation
               NOTIFY currentStationChanged)
    Q_PROPERTY(PlaybackMode::Mode playbackMode
               READ playbackMode
               NOTIFY playbackModeChanged)
    Q_PROPERTY(MediaState::State state
               READ state
               NOTIFY stateChanged)
    Q_PROPERTY(bool playing
               READ playing
               NOTIFY stateChanged)
    Q_PROPERTY(bool paused
               READ paused
               NOTIFY stateChanged)
    Q_PROPERTY(bool stopped
               READ stopped
               NOTIFY stateChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY stateChanged)
    Q_PROPERTY(bool buffering
               READ buffering
               NOTIFY stateChanged)
    Q_PROPERTY(bool seekable
               READ seekable
               NOTIFY seekableChanged)
    Q_PROPERTY(QString stateString
               READ stateString
               NOTIFY stateChanged)
    Q_PROPERTY(QString errorString
               READ errorString
               NOTIFY error)
    Q_PROPERTY(qint64 position
               READ position
               WRITE setPosition
               NOTIFY positionChanged)
    Q_PROPERTY(qint64 duration
               READ duration
               NOTIFY durationChanged)
    Q_PROPERTY(int bufferProgress
               READ bufferProgress
               NOTIFY bufferProgressChanged)
    Q_PROPERTY(int volume
               READ volume
               WRITE setVolume
               NOTIFY volumeChanged)
    Q_PROPERTY(QString title
               READ title
               NOTIFY metaDataChanged)
    Q_PROPERTY(QString artist
               READ artist
               NOTIFY metaDataChanged)
    Q_PROPERTY(QString album
               READ album
               NOTIFY metaDataChanged)
    Q_PROPERTY(QString genre
               READ genre
               NOTIFY metaDataChanged)
    Q_PROPERTY(QString description
               READ description
               NOTIFY metaDataChanged)
    Q_PROPERTY(qint64 bitRate
               READ bitRate
               NOTIFY metaDataChanged)
    Q_PROPERTY(bool sleepTimerEnabled
               READ sleepTimerEnabled
               WRITE setSleepTimerEnabled
               NOTIFY sleepTimerEnabledChanged)
    Q_PROPERTY(qint64 sleepTimerRemaining
               READ sleepTimerRemaining
               NOTIFY sleepTimerRemainingChanged)
    Q_PROPERTY(int tickInterval
               READ tickInterval
               WRITE setTickInterval)

public:
    explicit MediaPlayer(QObject *parent = 0);
    ~MediaPlayer();

    static MediaPlayer* instance();

    Station* currentStation() const;
    void setCurrentStation(Station *station);

    PlaybackMode::Mode playbackMode() const;

    bool playing() const;

    bool paused() const;

    bool stopped() const;

    bool loading() const;

    bool buffering() const;

    bool seekable() const;

    MediaState::State state() const;
    QString stateString() const;

    QString errorString() const;

    qint64 position() const;

    qint64 duration() const;

    int bufferProgress() const;

    int volume() const;

    QString title() const;
    QString artist() const;
    QString album() const;
    QString genre() const;
    QString description() const;
    qint64 bitRate() const;

    bool sleepTimerEnabled() const;

    qint64 sleepTimerRemaining() const;

    int tickInterval() const;
    
public slots:
    void play();
    void play(const QUrl &url);
    void play(Station *station);
    void pause();
    void stop();
    void clear();

    void setPosition(qint64 position);
    void setVolume(int volume);
    void setSleepTimerEnabled(bool enabled);
    void setTickInterval(int interval);

private:
    void setPlaybackMode(PlaybackMode::Mode playbackMode);

    void setState(MediaState::State state);

    void setErrorString(const QString &errorString);

    void setSleepTimerRemaining(qint64 remaining);
    int sleepTimerId() const;
    void setSleepTimerId(int timerId);

    void timerEvent(QTimerEvent *event);

private slots:
    void updateSleepTimer();
    void onStateChanged(QMediaPlayer::State state);
    void onMediaStatusChanged(QMediaPlayer::MediaStatus status);
    void onMediaError(QMediaPlayer::Error mediaError);
    void onExtractorError(const QString &errorString);
    void onStationPlayed();
    void onMediaPlayerChanged(const QString &mediaPlayer);

signals:
    void currentStationChanged(Station *station);
    void metaDataChanged();
    void seekableChanged(bool seekable);
    void bufferProgressChanged(int progress);
    void positionChanged(qint64 position);
    void durationChanged(qint64 duration);
    void volumeChanged(int volume);
    void playbackModeChanged(PlaybackMode::Mode playbackMode);
    void stateChanged(MediaState::State state);
    void sleepTimerEnabledChanged(bool enabled);
    void sleepTimerRemainingChanged(qint64 remaining);
    void error(const QString &errorString);

private:
    static MediaPlayer *self;

    QMediaPlayer *m_mediaPlayer;

    StreamExtractor *m_extractor;

    Station *m_station;

    MediaState::State m_state;
    PlaybackMode::Mode m_playbackMode;
    QString m_errorString;

    bool m_sleepTimerEnabled;

    qint64 m_sleepTimerRemaining;
    int m_sleepTimerId;
};

#endif // MEDIAPLAYER_H
