/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "tuneinstationlistwindow.h"
#include "tuneinstationinfowindow.h"
#include "tuneinstationlistdelegate.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "tuneinaddstationdialog.h"
#include "launcher.h"
#include "../base/utils.h"
#include "../base/settings.h"
#include "../base/streamextractor.h"
#include "../tunein/tuneinstationfilterlistmodel.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

TuneInStationListWindow::TuneInStationListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new TuneInStationFilterListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_infoAction(m_contextMenu->addAction(tr("View info"), this, SLOT(showStationInfo()))),
    m_addAction(m_contextMenu->addAction(tr("Add to database"), this, SLOT(showAddStationDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No stations found")), this))
{
    this->setWindowTitle(tr("Stations"));
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TuneInStationListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void TuneInStationListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void TuneInStationListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void TuneInStationListWindow::searchStations(const QString &query) {
    this->setWindowTitle(tr("Search ('%1')").arg(query));
    m_model->searchStations(query);
}

void TuneInStationListWindow::showStationsByGenre(const QString &genre, const QString &id) {
    this->setWindowTitle(genre);
    m_model->showStationsByGenre(id);
}

void TuneInStationListWindow::showStationsByCountry(const QString &country, const QString &id) {
    this->setWindowTitle(country);
    m_model->showStationsByCountry(id);
}

void TuneInStationListWindow::showStations(const QString &title, const QString &url) {
    this->setWindowTitle(title);
    m_model->showStations(url);
}

void TuneInStationListWindow::clear() {
    m_model->clear();
}

void TuneInStationListWindow::reload() {
    m_model->reload();
}

void TuneInStationListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void TuneInStationListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void TuneInStationListWindow::onCountChanged(int count) {
    m_view->setHidden((!m_model->loading()) && (count == 0));
}

void TuneInStationListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_infoAction);
}

void TuneInStationListWindow::showStationInfo() {
    if (m_view->currentIndex().isValid()) {
        TuneInStationInfoWindow *window = new TuneInStationInfoWindow(m_view->currentIndex().data(TuneInStationListModel::IdRole).toString(), this);
        window->show();
    }
}

void TuneInStationListWindow::showAddStationDialog() {
    if (m_view->currentIndex().isValid()) {
        TuneInAddStationDialog *dialog = new TuneInAddStationDialog(m_view->currentIndex().data(TuneInStationListModel::IdRole).toString(), this);
        dialog->open();
    }
}

void TuneInStationListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        if (Settings::instance()->mediaPlayer() == "cuteradio") {
            MediaPlayer::instance()->play(m_model->get(index));
        }
        else {
            QUrl url = index.data(TuneInStationListModel::SourceRole).toUrl();

            if (Utils::urlIsPlaylist(url)) {
                StreamExtractor::instance()->getStreamUrl(url);
            }
            else {
                Launcher::launch(url);
            }
        }
    }
}
