/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QMenuBar>
#include <QVBoxLayout>
#include <QLabel>
#include <QAction>
#include <QMaemo5InformationBox>
#include "tuneinstationinfowindow.h"
#include "separatorlabel.h"
#include "logolabel.h"
#include "listview.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "tuneinaddstationdialog.h"
#include "launcher.h"
#include "../base/utils.h"
#include "../base/station.h"
#include "../base/settings.h"
#include "../base/streamextractor.h"
#include "../tunein/tunein.h"
#include "../tunein/tuneinstation.h"
#include "../thumbnailcache/thumbnailcache.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

TuneInStationInfoWindow::TuneInStationInfoWindow(const QString &id, QWidget *parent) :
    StackedWindow(parent),
    m_station(0),
    m_playAction(this->menuBar()->addAction(tr("Play"), this, SLOT(playStation()))),
    m_addAction(this->menuBar()->addAction(tr("Add to database"), this, SLOT(showAddStationDialog()))),
    m_descriptionLabel(new QLabel(this)),
    m_genreLabel(new QLabel(this)),
    m_countryLabel(new QLabel(this)),
    m_languageLabel(new QLabel(this)),
    m_sourceLabel(new QLabel(this)),
    m_lastPlayedLabel(new QLabel(this)),
    m_logoLabel(new LogoLabel(this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(tr("Station info"));
    this->setCentralWidget(m_scrollArea);

    m_descriptionLabel->setWordWrap(true);
    m_genreLabel->setWordWrap(true);
    m_countryLabel->setWordWrap(true);
    m_languageLabel->setWordWrap(true);
    m_sourceLabel->setWordWrap(true);
    m_lastPlayedLabel->setWordWrap(true);

    m_logoLabel->setFixedSize(124, 124);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->addWidget(m_logoLabel);
    vbox->addWidget(m_descriptionLabel);
    vbox->addWidget(new SeparatorLabel(tr("Properties"), this));
    vbox->addWidget(m_genreLabel);
    vbox->addWidget(m_countryLabel);
    vbox->addWidget(m_languageLabel);
    vbox->addWidget(m_sourceLabel);
    vbox->addWidget(m_lastPlayedLabel);
    vbox->addSpacerItem(new QSpacerItem(10, 10, QSizePolicy::Expanding, QSizePolicy::Expanding));

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_logoLabel, SIGNAL(clicked()), this, SLOT(playStation()));
    this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    this->connect(MediaPlayer::instance(), SIGNAL(currentStationChanged(Station*)), this, SLOT(onCurrentStationChanged(Station*)));

    this->toggleBusy(true);

    TuneInStation *station = TuneIn::getStation(id);
    this->connect(station, SIGNAL(finished(TuneInStation*)), this, SLOT(onStationFinished(TuneInStation*)));
    this->onCurrentStationChanged(MediaPlayer::instance()->currentStation());
}

void TuneInStationInfoWindow::playStation() {
    if (m_station) {
        if (Settings::instance()->mediaPlayer() == "cuteradio") {
            MediaPlayer::instance()->play(m_station);
        }
        else if (Utils::urlIsPlaylist(m_station->source())) {
            StreamExtractor::instance()->getStreamUrl(m_station->source());
        }
        else {
            Launcher::launch(m_station->source());
        }
    }
}

void TuneInStationInfoWindow::showAddStationDialog() {
    if (m_station) {
        TuneInAddStationDialog *dialog = new TuneInAddStationDialog(m_station->id(), this);
        dialog->open();
    }
}

void TuneInStationInfoWindow::onLastPlayedChanged(qint64 lastPlayed) {
    m_lastPlayedLabel->setText(QString("%1: %2").arg(tr("Last played")).arg(lastPlayed > 0 ? Utils::dateFromMSecs(lastPlayed) : m_station->lastPlayedString()));
}

void TuneInStationInfoWindow::onCurrentStationChanged(Station *station) {
    if (m_station) {
        m_playAction->setEnabled((!station) || (station->id() != m_station->id()));
        m_logoLabel->setEnabled(m_playAction->isEnabled());
    }
}

void TuneInStationInfoWindow::onStationFinished(TuneInStation *station) {
    this->toggleBusy(false);

    m_station = station->result();

    if (m_station) {
        this->setWindowTitle(m_station->title());
        this->onLastPlayedChanged(m_station->lastPlayed());
        this->setLogo();
        this->connect(m_station, SIGNAL(lastPlayedChanged(qint64)), this, SLOT(onLastPlayedChanged(qint64)), Qt::UniqueConnection);

        m_station->setParent(this);
        m_descriptionLabel->setText(m_station->description().isEmpty() ? tr("No description") : m_station->description());
        m_genreLabel->setText(QString("%1: %2").arg(tr("Genre")).arg(m_station->genre().isEmpty() ? tr("Unknown") : m_station->genre()));
        m_countryLabel->setText(QString("%1: %2").arg(tr("Country")).arg(m_station->country().isEmpty() ? tr("Unknown") : m_station->country()));
        m_languageLabel->setText(QString("%1: %2").arg(tr("Language")).arg(m_station->language().isEmpty() ? tr("Unknown") : m_station->language()));
        m_sourceLabel->setText(QString("%1: %2").arg(tr("Source")).arg(m_station->source().toString()));
    }
    else {
        switch (station->error()) {
        case QNetworkReply::NoError:
            QMaemo5InformationBox::information(0, tr("No station information found"));
            break;
        default:
            QMaemo5InformationBox::information(0, station->errorString());
            break;
        }
    }

    station->deleteLater();
}

void TuneInStationInfoWindow::setLogo() {
    if (m_station) {
        QImage logo = ThumbnailCache::instance()->thumbnail(m_station->logo(), QSize(124, 124));

        if (logo.isNull()) {
            this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()), Qt::UniqueConnection);
        }
        else {
            this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()));
        }

        if (logo.isNull()) {
            logo = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_stream.png");
        }

        m_logoLabel->setImage(logo);
    }
}
