/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QMenuBar>
#include <QVBoxLayout>
#include <QLabel>
#include <QAction>
#include <QMessageBox>
#include "podcastepisodeinfowindow.h"
#include "separatorlabel.h"
#include "logolabel.h"
#include "listview.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "addstationdialog.h"
#include "launcher.h"
#include "../base/utils.h"
#include "../base/database.h"
#include "../base/station.h"
#include "../base/settings.h"
#include "../base/streamextractor.h"
#include "../thumbnailcache/thumbnailcache.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

PodcastEpisodeInfoWindow::PodcastEpisodeInfoWindow(Station *episode, QWidget *parent) :
    StackedWindow(parent),
    m_episode(episode),
    m_playAction(this->menuBar()->addAction(tr("Play"), this, SLOT(playEpisode()))),
    m_descriptionLabel(new QLabel(m_episode->description().isEmpty() ? tr("No description") : m_episode->description(), this)),
    m_genreLabel(new QLabel(QString("%1: %2").arg(tr("Genre")).arg(m_episode->genre().isEmpty() ? tr("Unknown") : m_episode->genre()), this)),
    m_countryLabel(new QLabel(QString("%1: %2").arg(tr("Country")).arg(m_episode->country().isEmpty() ? tr("Unknown") : m_episode->country()), this)),
    m_languageLabel(new QLabel(QString("%1: %2").arg(tr("Language")).arg(m_episode->language().isEmpty() ? tr("Unknown") : m_episode->language()), this)),
    m_sourceLabel(new QLabel(QString("%1: %2").arg(tr("Source")).arg(m_episode->source().toString()), this)),
    m_lastPlayedLabel(new QLabel(QString("%1: %2").arg(tr("Last played")).arg(m_episode->lastPlayedString()), this)),
    m_logoLabel(new LogoLabel(this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(m_episode->title());
    this->setCentralWidget(m_scrollArea);

    m_descriptionLabel->setWordWrap(true);
    m_genreLabel->setWordWrap(true);
    m_countryLabel->setWordWrap(true);
    m_languageLabel->setWordWrap(true);
    m_sourceLabel->setWordWrap(true);
    m_lastPlayedLabel->setWordWrap(true);

    m_logoLabel->setFixedSize(124, 124);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->addWidget(m_logoLabel);
    vbox->addWidget(m_descriptionLabel);
    vbox->addWidget(new SeparatorLabel(tr("Properties"), this));
    vbox->addWidget(m_genreLabel);
    vbox->addWidget(m_countryLabel);
    vbox->addWidget(m_languageLabel);
    vbox->addWidget(m_sourceLabel);
    vbox->addWidget(m_lastPlayedLabel);
    vbox->addSpacerItem(new QSpacerItem(10, 10, QSizePolicy::Expanding, QSizePolicy::Expanding));

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_logoLabel, SIGNAL(clicked()), this, SLOT(playEpisode()));
    this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    this->connect(m_episode, SIGNAL(lastPlayedChanged(qint64)), this, SLOT(onLastPlayedChanged(qint64)));
    this->connect(MediaPlayer::instance(), SIGNAL(currentStationChanged(Station*)), this, SLOT(onCurrentStationChanged(Station*)));

    this->onCurrentStationChanged(MediaPlayer::instance()->currentStation());
    this->setLogo();
}

void PodcastEpisodeInfoWindow::playEpisode() {
    if (Settings::instance()->mediaPlayer() == "cuteradio") {
        MediaPlayer::instance()->play(m_episode);
    }
    else if (Utils::urlIsPlaylist(m_episode->source())) {
        StreamExtractor::instance()->getStreamUrl(m_episode->source());
    }
    else {
        Launcher::launch(m_episode->source());
    }
}

void PodcastEpisodeInfoWindow::setLogo() {
    if (m_episode) {
        QImage logo = ThumbnailCache::instance()->thumbnail(m_episode->logo(), QSize(124, 124));

        if (logo.isNull()) {
            this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()), Qt::UniqueConnection);
        }
        else {
            this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()));
        }

        if (logo.isNull()) {
            logo = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_stream.png");
        }

        m_logoLabel->setImage(logo);
    }
}

void PodcastEpisodeInfoWindow::onLastPlayedChanged(qint64 lastPlayed) {
    m_lastPlayedLabel->setText(QString("%1: %2").arg(tr("Last played")).arg(lastPlayed > 0 ? Utils::dateFromMSecs(lastPlayed) : m_episode->lastPlayedString()));
}

void PodcastEpisodeInfoWindow::onCurrentStationChanged(Station *station) {
    m_playAction->setEnabled((!station) || (station->id() != m_episode->id()));
    m_logoLabel->setEnabled(m_playAction->isEnabled());
}
