#include "logodownloader.h"
#include "networkaccessmanager.h"
#include <QImage>
#include <QDir>

static const QRegExp ILLEGAL_CHARS("[\"@&~=:?#!|<>*^]");
static const int MAX_RETRIES = 8;

LogoDownloader::LogoDownloader(QObject *parent) :
    QObject(parent),
    m_reply(0),
    m_retries(0)
{
}

LogoDownloader::LogoDownloader(const QUrl &url, const QString &localPath, QObject *parent) :
    QObject(parent),
    m_reply(0),
    m_url(url),
    m_path(localPath)
{
}

LogoDownloader::~LogoDownloader() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

QUrl LogoDownloader::url() const {
    return m_url;
}

void LogoDownloader::setUrl(const QUrl &url) {
    m_url = url;
}

QString LogoDownloader::localPath() const {
    return m_path;
}

void LogoDownloader::setLocalPath(const QString &path) {
    m_path = path;
    m_path.replace(ILLEGAL_CHARS, "_");
}

QNetworkReply::NetworkError LogoDownloader::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString LogoDownloader::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

int LogoDownloader::retries() const {
    return m_retries;
}

void LogoDownloader::setRetries(int retries) {
    m_retries = retries;
}

void LogoDownloader::start() {
    if (m_reply) {
        m_reply->deleteLater();
        m_reply = 0;
    }

    this->setRetries(0);

    m_reply = NetworkAccessManager::instance()->get(QNetworkRequest(this->url()));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void LogoDownloader::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void LogoDownloader::retry(const QUrl &url) {
    if (m_reply) {
        m_reply->deleteLater();
        m_reply = 0;
    }

    m_reply = NetworkAccessManager::instance()->get(QNetworkRequest(url));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void LogoDownloader::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if ((!redirect.isEmpty()) && (this->retries() < MAX_RETRIES)) {
        this->setRetries(this->retries() + 1);
        this->retry(redirect);
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    if (QDir().mkpath(this->localPath().left(this->localPath().lastIndexOf('/')))) {
        QImage image;

        if (image.loadFromData(m_reply->readAll())) {
            image.save(this->localPath());
        }
    }

    emit finished(this);
}
