/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0
import com.cuteradio.enums 1.0

MySheet {
    id: root

    function loadPodcast(podcastItem) {
        podcast.loadStation(podcastItem);
    }

    function addPodcast() {
        Database.asyncAddPodcast(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            downloader.localPath,
                            sourceField.text,
                            Services.TuneIn);

        clear();
    }

    function clear() {
        if (podcast.result) {
            podcast.result.clear();
        }
    }

    Station {
        id: podcast
    }

    LogoDownloader {
        id: downloader

        url: podcast.logo
        localPath: Utils.logoPath + podcast.id + "." + podcast.logo.toString().split(".").pop()
        onFinished: {
            progressDialog.close();
            root.addPodcast();
        }
    }

    hideHeaderWhenInputContextIsVisible: true
    acceptButtonText: (titleField.text) && (sourceField.text) && (sourceField.acceptableInput) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        KeyNavFlickable {
            id: flicker

            anchors {
                fill: parent
                margins: platformStyle.paddingLarge
            }
            contentHeight: column.height + platformStyle.paddingLarge

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: platformStyle.paddingLarge

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                    visible: titleField.visible
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    text: podcast.title
                    visible: (!inputContext.visible) || (focus)
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                    visible: descriptionField.visible
                }

                TextArea {
                    id: descriptionField

                    width: parent.width
                    text: podcast.description
                    visible: (!inputContext.visible) || (focus)
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                    visible: sourceField.visible
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: podcast.source
                    visible: (!inputContext.visible) || (focus)
                    validator: RegExpValidator {
                        regExp: /^http(s|):\/\/.+/
                    }
                    onAccepted: closeSoftwareInputPanel()
                }
            }
        }

        MyScrollBar {
            flickableItem: flicker
        }
    }

    Connections {
        target: progressDialog
        onRejected: {
            downloader.cancel();
            clear();
        }
    }

    onAccepted: {
        progressDialog.showBusyMessage(qsTr("Adding podcast"), 1);
        downloader.start();
    }
    onRejected: clear()
}
