/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.models 1.0
import com.cuteradio.items 1.0
import com.cuteradio.enums 1.0

MyPage {
    id: root

    function searchStations(query) {
        stationModel.searchStations(query);
    }

    function showStations() {
        stationModel.showStations();
    }

    function showRecentlyPlayedStations() {
        stationModel.showRecentlyPlayedStations();
    }

    function showFavouriteStations() {
        stationModel.showFavouriteStations();
    }

    function showStationsByGenre(genre) {
        stationModel.showStationsByGenre(genre);
    }

    function showStationsByCountry(country) {
        stationModel.showStationsByCountry(country);
    }

    function showStationsByLanguage(language) {
        stationModel.showStationsByLanguage(language);
    }

    tools: ToolBarLayout {

        BackToolButton {}

        NowPlayingButton {}
    }

    MyListView {
        id: view

        property int selectedIndex: -1

        anchors.fill: parent
        model: StationListModel {
            id: stationModel
        }
        delegate: StationDelegate {
            onClicked: appWindow.playStation(stationModel.get(index))
            onPressAndHold: {
                view.selectedIndex = -1;
                view.selectedIndex = index;
                contextMenu.open();
            }
        }
        section.delegate: SectionDelegate {
            text: section
        }
        section.property: "section"
        section.criteria: stationModel.queryType === Queries.RecentlyPlayedStations ? ViewSection.FullString : ViewSection.FirstCharacter
    }

    MySectionScroller {
        listView: view
    }

    Label {
        anchors {
            fill: parent
            margins: platformStyle.paddingLarge
        }
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        wrapMode: Text.WordWrap
        color: platformStyle.colorNormalMid
        font.bold: true
        font.pixelSize: 32
        text: qsTr("No stations found")
        visible: (!stationModel.count) && (!stationModel.loading)
    }

    BusyIndicator {
        anchors.centerIn: parent
        width: 64
        height: 64
        visible: stationModel.loading
        running: visible
    }

    ContextMenu {
        id: contextMenu

        MenuLayout {            

            MenuItem {
                text: qsTr("View info")
                onClicked: appWindow.pageStack.push(Qt.resolvedUrl("StationInfoPage.qml")).loadStation(stationModel.get(view.selectedIndex))
            }

            MenuItem {
                text: stationModel.data(view.selectedIndex, "favourite") ? qsTr("Delete from favourites")
                                                                         : qsTr("Add to favourites")

                onClicked: Database.asyncUpdateStation(stationModel.data(view.selectedIndex, "id"), "favourite", stationModel.data(view.selectedIndex, "favourite") ? 0 : 1, false)
            }

            MenuItem {
                text: qsTr("Edit")
                onClicked: {
                    dialogLoader.sourceComponent = editStationDialog;
                    dialogLoader.item.loadStation(stationModel.get(view.selectedIndex));
                    dialogLoader.item.open();
                }
            }

            MenuItem {
                text: qsTr("Delete")
                onClicked: {
                    dialogLoader.sourceComponent = confirmDeleteDialog;
                    dialogLoader.item.open();
                }
            }
        }

        onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
    }

    Loader {
        id: dialogLoader
    }

    Component {
        id: editStationDialog

        AddStationDialog {
            onStatusChanged: {
                switch (status) {
                case DialogStatus.Closed:
                    if (root.status === PageStatus.Active) view.forceActiveFocus();
                    return;
                default:
                    return;
                }
            }
        }
    }

    Component {
        id: confirmDeleteDialog

        MyQueryDialog {
            titleText: qsTr("Delete station?")
            icon: "images/info.png"
            message: qsTr("Do you wish to delete the station") + " '" + stationModel.data(view.selectedIndex, "title") + "' " + qsTr("from the database?")
            acceptButtonText: qsTr("Yes")
            rejectButtonText: qsTr("No")
            onAccepted: Database.asyncDeleteStation(stationModel.data(view.selectedIndex, "id"))
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
        }
    }
}
