/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.enums 1.0


MyPage {
    id: root

    title: !MediaPlayer.title ? MediaPlayer.currentStation.title : MediaPlayer.title
    tools: ToolBarLayout {

        BackToolButton {}
    }

    MyFlickable {
        id: flicker

        anchors {
            left: parent.left
            leftMargin: platformStyle.paddingLarge
            right: parent.right
            rightMargin: platformStyle.paddingLarge
            top: parent.top
            bottom: controls.top
            bottomMargin: platformStyle.paddingLarge
        }
        clip: true
        contentHeight: column.height + platformStyle.paddingLarge

        Column {
            id: column

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }
            spacing: platformStyle.paddingLarge

            Item {
                width: parent.width
                height: platformStyle.paddingLarge
            }

            Logo {
                width: 120
                height: 120
                image: MediaPlayer.currentStation.logo
                text: !MediaPlayer.title ? MediaPlayer.currentStation.title : MediaPlayer.title
                enabled: false
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                text: !MediaPlayer.description ? MediaPlayer.currentStation.description : MediaPlayer.description
            }

            SeparatorLabel {
                width: parent.width
                text: qsTr("Now playing")
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: platformStyle.fontSizeSmall
                text: !MediaPlayer.title ? qsTr("Title") + ": " + MediaPlayer.currentStation.title :
                                           qsTr("Title") + ": " + MediaPlayer.title
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Artist") + ": " + MediaPlayer.artist
                visible: MediaPlayer.artist != ""
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Album title") + ": " + MediaPlayer.album
                visible: MediaPlayer.album != ""
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: platformStyle.fontSizeSmall
                text: !MediaPlayer.genre ? qsTr("Genre") + ": " + MediaPlayer.currentStation.genre :
                                           qsTr("Genre") + ": " + MediaPlayer.genre
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: platformStyle.fontSizeSmall
                text: !MediaPlayer.bitRate ? qsTr("Bitrate") + ": " + qsTr("Unknown")
                                           : qsTr("Bitrate") + ": " + Utils.fileSizeFromBytes(MediaPlayer.bitRate) + "/s"
            }
        }
    }

    MyScrollBar {
        flickableItem: flicker
    }

    Column {
        id: controls

        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
            margins: platformStyle.paddingLarge
        }
        spacing: platformStyle.paddingLarge

        Rectangle {
            height: 1
            width: parent.width
            color: platformStyle.colorNormalMid
            visible: appWindow.inPortrait
        }

        ButtonRow {
            width: parent.width
            exclusive: false

            MyButton {
                id: playButton

                height: 80
                iconSource: (MediaPlayer.stopped) || (MediaPlayer.paused) ? "image://theme/toolbar-mediacontrol-play"
                                                : "image://theme/toolbar-mediacontrol-" + (MediaPlayer.duration > 0 ? "pause" : "stop")
                onClicked: (MediaPlayer.stopped) || (MediaPlayer.paused) ? MediaPlayer.play() : MediaPlayer.duration > 0 ? MediaPlayer.pause() : MediaPlayer.stop()
            }

            MyButton {
                id: sleepButton

                height: 80
                iconSource: "images/clock.png"
                text: MediaPlayer.sleepTimerEnabled ? Utils.durationFromMSecs(MediaPlayer.sleepTimerRemaining) : ""
                checkable: true
                onClicked: MediaPlayer.sleepTimerEnabled = !MediaPlayer.sleepTimerEnabled
                Component.onCompleted: checked = MediaPlayer.sleepTimerEnabled
            }
        }

        ProgressBar {
            id: progressBar

            width: parent.width
            maximumValue: (MediaPlayer.duration > 0) ? MediaPlayer.duration : 100
            enabled: MediaPlayer.duration > 0
            value: enabled ? MediaPlayer.position : 0

            SeekBubble {
                id: seekBubble

                anchors.bottom: parent.top
                opacity: value != "" ? 1 : 0
                value: (seekMouseArea.drag.active) && (seekMouseArea.posInsideDragArea) ? Utils.durationFromMSecs(Math.floor((seekMouseArea.mouseX / seekMouseArea.width) * MediaPlayer.duration)) : ""
            }

            MouseArea {
                id: seekMouseArea

                property bool posInsideMouseArea: false
                property bool posInsideDragArea: (seekMouseArea.mouseX >= 0) && (seekMouseArea.mouseX <= seekMouseArea.drag.maximumX)

                width: parent.width
                height: 60
                anchors.centerIn: parent
                drag.target: seekBubble
                drag.axis: Drag.XAxis
                drag.minimumX: -40
                drag.maximumX: width - 10
                enabled: MediaPlayer.duration > 0
                onExited: posInsideMouseArea = false
                onEntered: posInsideMouseArea = true
                onPressed: {
                    posInsideMouseArea = true;
                    seekBubble.x = mouseX - 40;
                }
                onReleased: {
                    if (posInsideMouseArea) {
                        MediaPlayer.position = Math.floor((mouseX / width) * MediaPlayer.duration);
                    }
                }
            }
        }

        Row {
            Label {
                id: positionLabel

                width: controls.width - durationLabel.width
                verticalAlignment: Text.AlignVCenter
                font.pixelSize: platformStyle.fontSizeSmall
                text: Utils.durationFromMSecs(MediaPlayer.position)
            }

            Label {
                id: durationLabel

                verticalAlignment: Text.AlignVCenter
                font.pixelSize: platformStyle.fontSizeSmall
                text: Utils.durationFromMSecs(MediaPlayer.duration)
                visible: MediaPlayer.duration > 0
            }
        }

        Label {
            width: parent.width
            elide: Text.ElideRight
            wrapMode: Text.WordWrap
            font.pixelSize: platformStyle.fontSizeSmall
            text: qsTr("Status") + ": " + MediaPlayer.stateString
        }
    }

    states: State {
        name: "landscape"
        when: !appWindow.inPortrait
        AnchorChanges { target: flicker; anchors { right: parent.horizontalCenter; bottom: parent.bottom } }
        AnchorChanges { target: controls; anchors { left: parent.horizontalCenter; bottom: undefined; verticalCenter: parent.verticalCenter } }
    }

    Connections {
        target: symbian
        onForegroundChanged: MediaPlayer.tickInterval = (symbian.foreground ? 1000 : 0)
    }

    onStatusChanged: MediaPlayer.tickInterval = (status === PageStatus.Active ? 1000 : 0)
}
