/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.meego 1.0
import com.cuteradio.items 1.0
import "file:///usr/lib/qt4/imports/com/nokia/meego/UIConstants.js" as UI

MySheet {
    id: root

    function loadStation(stationItem) {
        root.state = "loading";
        station.load(stationItem.id);
    }

    function addStation() {
        Database.asyncAddStation(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            downloader.localPath,
                            !genreField.text ? qsTr("Unknown genre") : genreField.text,
                            !countryField.text ? qsTr("Unknown country") : countryField.text,
                            !languageField.text ? qsTr("Unknown language") : languageField.text,
                            sourceField.text,
                            favouriteCheckBox.checked);

        clear();
    }

    function clear() {
        if (station.result) {
            station.result.clear();
        }
    }

    TuneInStation {
        id: station

        onFinished: {
            if (station.result) {
                root.state = "finished";
            }
            else {
                root.state = "error";
                infoBanner.displayMessage(station.errorString);
            }
        }
    }

    LogoDownloader {
        id: downloader

        url: station.result ? station.result.logo : ""
        localPath: station.result ? Utils.logoPath + station.result.id + "." + station.result.logo.toString().split(".").pop() : ""
        onFinished: {
            progressDialog.close();
            root.addStation();
        }
    }

    acceptButtonText: (titleField.text) && (sourceField.text) && (sourceField.acceptableInput) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        Flickable {
            id: flicker

            anchors {
                fill: parent
                margins: UI.PADDING_DOUBLE
            }
            contentHeight: column.height + UI.PADDING_DOUBLE

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: UI.PADDING_DOUBLE

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.result ? station.result.title : ""
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                }

                MyTextArea {
                    id: descriptionField

                    width: parent.width
                    text: station.result ? station.result.description : ""
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Genre")
                }

                MyTextField {
                    id: genreField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.result ? station.result.genre : ""
                    onAccepted: countryField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Country")
                }

                MyTextField {
                    id: countryField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.result ? station.result.country : ""
                    onAccepted: languageField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Language")
                }

                MyTextField {
                    id: languageField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.result ? station.result.language : ""
                    onAccepted: sourceField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: station.result ? station.result.source : ""
                    validator: RegExpValidator {
                        regExp: /^http(s|):\/\/.+/
                    }
                    onAccepted: platformCloseSoftwareInputPanel()
                }

                MyCheckBox {
                    id: favouriteCheckBox

                    text: qsTr("Favourite?")
                    checked: station.result ? station.result.favourite : false
                }
            }
        }

        ScrollDecorator {
            flickableItem: flicker
        }

        Label {
            id: errorLabel

            anchors {
                fill: parent
                margins: UI.PADDING_DOUBLE
            }
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            wrapMode: Text.WordWrap
            color: UI.COLOR_INVERTED_SECONDARY_FOREGROUND
            font.bold: true
            font.pixelSize: 40
            text: qsTr("No information found")
            visible: false
        }

        BusyIndicator {
            id: busyIndicator

            anchors.centerIn: parent
            platformStyle: BusyIndicatorStyle {
                size: "large"
            }
            visible: false
            running: visible
        }
    }

    states: [
        State {
            name: "loading"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: true }
        },

        State {
            name: "finished"
            PropertyChanges { target: flicker; visible: true }
            PropertyChanges { target: busyIndicator; visible: false }
        },

        State {
            name: "error"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: false }
            PropertyChanges { target: errorLabel; visible: true }
        }
    ]

    Connections {
        target: progressDialog
        onRejected: {
            downloader.cancel();
            clear();
        }
    }

    onAccepted: {
        progressDialog.showBusyMessage(qsTr("Adding station"), 1);
        downloader.start();
    }
    onRejected: clear()
}
