/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QDomDocument>
#include <QDomElement>
#include "tuneinpodcastepisodelist.h"
#include "tuneinurls.h"

TuneInPodcastEpisodeList::TuneInPodcastEpisodeList(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

TuneInPodcastEpisodeList::~TuneInPodcastEpisodeList() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

int TuneInPodcastEpisodeList::count() const {
    return m_results.size();
}

QList<Station*> TuneInPodcastEpisodeList::results() const {
    return m_results;
}

bool TuneInPodcastEpisodeList::moreResults() const {
    return !m_nextUrl.isEmpty();
}

QUrl TuneInPodcastEpisodeList::nextUrl() const {
    return m_nextUrl;
}

void TuneInPodcastEpisodeList::setNextUrl(const QUrl &url) {
    m_nextUrl = url;
}

QNetworkReply::NetworkError TuneInPodcastEpisodeList::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString TuneInPodcastEpisodeList::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void TuneInPodcastEpisodeList::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void TuneInPodcastEpisodeList::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomNodeList results = doc.documentElement().firstChildElement("body").elementsByTagName("outline");

    for (int i = 0; i < results.size(); i++) {
        QDomElement result = results.at(i).toElement();

        if (result.attribute("item") == "topic") {
            Station *station = new Station;
            station->setStationType(Stations::Podcast);
            station->setService(Services::TuneIn);
            station->setId(result.attribute("guide_id"));
            station->setTitle(result.attribute("text"));
            station->setDescription(result.attribute("subtext"));
            station->setLogo(result.attribute("image"));
            station->setSource(result.attribute("URL"));
            m_results.append(station);
        }
        else if (result.attribute("key") == "nextTopics") {
            m_nextUrl.setUrl(result.attribute("URL"));
        }
    }

    emit finished(this);
}
