/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QTextEdit>
#include <QCheckBox>
#include <QLabel>
#include <QDialogButtonBox>
#include <QGridLayout>
#include "addstationdialog.h"
#include "scrollarea.h"
#include "../base/station.h"
#include "../base/database.h"
#include "../base/logodownloader.h"
#include "../base/utils.h"

AddStationDialog::AddStationDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_station(0),
    m_titleEdit(new QLineEdit(this)),
    m_logoEdit(new QLineEdit(this)),
    m_genreEdit(new QLineEdit(this)),
    m_countryEdit(new QLineEdit(this)),
    m_languageEdit(new QLineEdit(this)),
    m_sourceEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_favouriteCheckBox(new QCheckBox(tr("Favourite?"), this)),
    m_scrollArea(new ScrollArea(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Ok, Qt::Vertical, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Add station"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(new QLabel(tr("Title"), this));
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(new QLabel(tr("Description"), this));
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(new QLabel(tr("Logo"), this));
    vbox->addWidget(m_logoEdit);
    vbox->addWidget(new QLabel(tr("Genre"), this));
    vbox->addWidget(m_genreEdit);
    vbox->addWidget(new QLabel(tr("Country"), this));
    vbox->addWidget(m_countryEdit);
    vbox->addWidget(new QLabel(tr("Language"), this));
    vbox->addWidget(m_languageEdit);
    vbox->addWidget(new QLabel(tr("Source"), this));
    vbox->addWidget(m_sourceEdit);
    vbox->addWidget(m_favouriteCheckBox);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    m_buttonBox->setEnabled(false);

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleChanged(QString)));
    this->connect(m_sourceEdit, SIGNAL(textChanged(QString)), this, SLOT(onSourceChanged(QString)));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(addStation()));

    this->onOrientationChanged();
}

AddStationDialog::AddStationDialog(Station *station, QWidget *parent) :
    AbstractDialog(parent),
    m_station(station),
    m_titleEdit(new QLineEdit(station->title(), this)),
    m_logoEdit(new QLineEdit(station->logo().toString(), this)),
    m_genreEdit(new QLineEdit(station->genre(), this)),
    m_countryEdit(new QLineEdit(station->country(), this)),
    m_languageEdit(new QLineEdit(station->language(), this)),
    m_sourceEdit(new QLineEdit(station->source().toString(), this)),
    m_descriptionEdit(new QTextEdit(station->description(), this)),
    m_favouriteCheckBox(new QCheckBox(tr("Favourite?"), this)),
    m_scrollArea(new ScrollArea(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Ok, Qt::Vertical, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Edit station"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(new QLabel(tr("Title"), this));
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(new QLabel(tr("Description"), this));
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(new QLabel(tr("Logo"), this));
    vbox->addWidget(m_logoEdit);
    vbox->addWidget(new QLabel(tr("Genre"), this));
    vbox->addWidget(m_genreEdit);
    vbox->addWidget(new QLabel(tr("Country"), this));
    vbox->addWidget(m_countryEdit);
    vbox->addWidget(new QLabel(tr("Language"), this));
    vbox->addWidget(m_languageEdit);
    vbox->addWidget(new QLabel(tr("Source"), this));
    vbox->addWidget(m_sourceEdit);
    vbox->addWidget(m_favouriteCheckBox);

    m_favouriteCheckBox->setChecked(station->favourite());

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    m_buttonBox->setEnabled((!m_titleEdit->text().isEmpty()) && (!m_sourceEdit->text().isEmpty()));

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleChanged(QString)));
    this->connect(m_sourceEdit, SIGNAL(textChanged(QString)), this, SLOT(onSourceChanged(QString)));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(editStation()));

    this->onOrientationChanged();
}

void AddStationDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void AddStationDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void AddStationDialog::onTitleChanged(const QString &title) {
    m_buttonBox->setEnabled((!title.isEmpty()) && (!m_sourceEdit->text().isEmpty()));
}

void AddStationDialog::onSourceChanged(const QString &source) {
    m_buttonBox->setEnabled((!m_titleEdit->text().isEmpty()) && (!source.isEmpty()));
}

void AddStationDialog::addStation() {
    if (!m_logoEdit->text().startsWith("http")) {
        Database::asyncAddStation(m_titleEdit->text(),
                                  m_descriptionEdit->toPlainText(),
                                  QUrl(m_logoEdit->text()),
                                  m_genreEdit->text(),
                                  m_countryEdit->text(),
                                  m_languageEdit->text(),
                                  m_sourceEdit->text(),
                                  m_favouriteCheckBox->isChecked());

        this->accept();
    }
    else {
        this->toggleBusy(true);

        LogoDownloader *downloader = new LogoDownloader(QUrl(m_logoEdit->text()), QString("%1%2.%3").arg(Utils::logoPath()).arg(m_titleEdit->text()).arg(m_logoEdit->text().section('.', -1)), this);
        this->connect(downloader, SIGNAL(finished(LogoDownloader*)), this, SLOT(onLogoDownloaderFinished(LogoDownloader*)));
        downloader->start();
    }
}

void AddStationDialog::editStation() {
    if (m_station) {
        if (!m_logoEdit->text().startsWith("http")) {
            QVariantMap properties;
            properties["title"] = m_titleEdit->text();
            properties["description"] = m_descriptionEdit->toPlainText();
            properties["logo"] = m_logoEdit->text();
            properties["genre"] = m_genreEdit->text();
            properties["country"] = m_countryEdit->text();
            properties["language"] = m_languageEdit->text();
            properties["source"] = m_sourceEdit->text();
            properties["favourite"] = m_favouriteCheckBox->isChecked() ? 1 : 0;

            Database::asyncUpdateStation(m_station->id(), properties);

            this->accept();
        }
        else {
            this->toggleBusy(true);

            LogoDownloader *downloader = new LogoDownloader(QUrl(m_logoEdit->text()), QString("%1%2.%3").arg(Utils::logoPath()).arg(m_station->id()).arg(m_logoEdit->text().section('.', -1)), this);
            this->connect(downloader, SIGNAL(finished(LogoDownloader*)), this, SLOT(onLogoDownloaderFinished(LogoDownloader*)));
            downloader->start();
        }
    }
    else {
        this->reject();
    }
}

void AddStationDialog::onLogoDownloaderFinished(LogoDownloader *downloader) {
    if (m_station) {
        QVariantMap properties;
        properties["title"] = m_titleEdit->text();
        properties["description"] = m_descriptionEdit->toPlainText();
        properties["logo"] = QUrl(downloader->localPath());
        properties["genre"] = m_genreEdit->text();
        properties["country"] = m_countryEdit->text();
        properties["language"] = m_languageEdit->text();
        properties["source"] = m_sourceEdit->text();
        properties["favourite"] = m_favouriteCheckBox->isChecked() ? 1 : 0;

        Database::asyncUpdateStation(m_station->id(), properties);
    }
    else {
        Database::asyncAddStation(m_titleEdit->text(),
                                  m_descriptionEdit->toPlainText(),
                                  QUrl(downloader->localPath()),
                                  m_genreEdit->text(),
                                  m_countryEdit->text(),
                                  m_languageEdit->text(),
                                  m_sourceEdit->text(),
                                  m_favouriteCheckBox->isChecked());
    }

    this->toggleBusy(false);
    this->accept();
}
