/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "stationlistmodel.h"
#include "database.h"

StationListModel::StationListModel(QObject *parent) :
    QAbstractListModel(parent),
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[ServiceRole] = "service";
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[DescriptionRole] = "description";
    m_roleNames[LogoRole] = "logo";
    m_roleNames[GenreRole] = "genre";
    m_roleNames[CountryRole] = "country";
    m_roleNames[LanguageRole] = "language";
    m_roleNames[SourceRole] = "source";
    m_roleNames[FavouriteRole] = "favourite";
    m_roleNames[LastPlayedRole] = "lastPlayed";
    m_roleNames[SectionRole] = "section";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif

    this->connect(Database::instance(), SIGNAL(stationAdded()), this, SLOT(reload()), Qt::UniqueConnection);
    this->connect(Database::instance(), SIGNAL(stationDeleted(QString)), this, SLOT(onStationDeleted(QString)), Qt::UniqueConnection);
}

StationListModel::~StationListModel() {
    this->clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> StationListModel::roleNames() const {
    return m_roleNames;
}
#endif

int StationListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant StationListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case ServiceRole:
        return m_list.at(index.row())->service();
    case IdRole:
        return m_list.at(index.row())->id();
    case TitleRole:
        return m_list.at(index.row())->title();
    case DescriptionRole:
        return m_list.at(index.row())->description();
    case LogoRole:
        return m_list.at(index.row())->logo();
    case GenreRole:
        return m_list.at(index.row())->genre();
    case CountryRole:
        return m_list.at(index.row())->country();
    case LanguageRole:
        return m_list.at(index.row())->language();
    case SourceRole:
        return m_list.at(index.row())->source();
    case FavouriteRole:
        return m_list.at(index.row())->favourite();
    case LastPlayedRole:
        return m_list.at(index.row())->lastPlayed();
    case SectionRole:
        switch (this->queryType()) {
        case Queries::RecentlyPlayedStations:
            return m_list.at(index.row())->lastPlayedString();
        default:
            return m_list.at(index.row())->title().left(1).toUpper();
        }
    default:
        return QVariant();
    }
}

QVariant StationListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool StationListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if ((!this->rowCount()) || (!index.isValid())) {
        return false;
    }

    switch (role) {
    case IdRole:
    case LastPlayedRole:
    case SectionRole:
        return false;
    default:
        return Database::asyncUpdateStation(index.data(IdRole).toString(), this->roleNames().value(role), value);
    }
}

bool StationListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return this->setData(this->index(row), value, this->roleNames().key(role));
}

bool StationListModel::loading() const {
    return m_loading;
}

void StationListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QString StationListModel::searchQuery() const {
    return m_searchQuery;
}

void StationListModel::setSearchQuery(const QString &query) {
    if (query != this->searchQuery()) {
        m_searchQuery = query;
        emit searchQueryChanged(query);
    }
}

Queries::QueryType StationListModel::queryType() const {
    return m_queryType;
}

void StationListModel::setQueryType(Queries::QueryType queryType) {
    if (queryType != this->queryType()) {
        m_queryType = queryType;
        emit queryTypeChanged(queryType);
    }
}

Station* StationListModel::get(const QModelIndex &index) const {
    return this->get(index.row());
}

Station* StationListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : 0;
}

void StationListModel::searchStations(const QString &query) {
    this->setLoading(true);
    this->setSearchQuery(query);
    this->setQueryType(Queries::StationSearch);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetStationsByTitle", Qt::QueuedConnection, Q_ARG(QString, query));
}

void StationListModel::showRecentlyPlayedStations() {
    this->setLoading(true);
    this->setQueryType(Queries::RecentlyPlayedStations);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetRecentlyPlayedStations", Qt::QueuedConnection);
}

void StationListModel::showFavouriteStations() {
    this->setLoading(true);
    this->setQueryType(Queries::FavouriteStations);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetFavouriteStations", Qt::QueuedConnection);
}

void StationListModel::showStationsByGenre(const QString &genre) {
    this->setLoading(true);
    this->setSearchQuery(genre);
    this->setQueryType(Queries::Genres);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetStationsByGenre", Qt::QueuedConnection, Q_ARG(QString, genre));
}

void StationListModel::showStationsByCountry(const QString &country) {
    this->setLoading(true);
    this->setSearchQuery(country);
    this->setQueryType(Queries::Countries);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetStationsByCountry", Qt::QueuedConnection, Q_ARG(QString, country));
}

void StationListModel::showStationsByLanguage(const QString &language) {
    this->setLoading(true);
    this->setSearchQuery(language);
    this->setQueryType(Queries::Languages);
    this->connect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetStationsByLanguage", Qt::QueuedConnection, Q_ARG(QString, language));
}

void StationListModel::clear() {
    this->beginResetModel();
    qDeleteAll(m_list);
    m_list.clear();
    this->endResetModel();
}

void StationListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::StationSearch:
        this->searchStations(this->searchQuery());
        return;
    case Queries::RecentlyPlayedStations:
        this->showRecentlyPlayedStations();
        return;
    case Queries::FavouriteStations:
        this->showFavouriteStations();
        return;
    case Queries::Genres:
        this->showStationsByGenre(this->searchQuery());
        return;
    case Queries::Countries:
        this->showStationsByCountry(this->searchQuery());
        return;
    case Queries::Languages:
        this->showStationsByLanguage(this->searchQuery());
        return;
    default:
        return;
    }
}

void StationListModel::addStations(QList<Station *> stations) {
    this->disconnect(Database::instance(), SIGNAL(gotStations(QList<Station*>)), this, SLOT(addStations(QList<Station*>)));

    if (stations.isEmpty()) {
        this->setLoading(false);
        return;
    }

    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount() + stations.size() - 1);
    m_list << stations;
    this->endInsertRows();
    emit countChanged(this->rowCount());

    foreach (Station *station, stations) {
        this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
        this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
    }

    this->setLoading(false);
}

void StationListModel::addStation(Station *station) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(station);
    this->endInsertRows();
    emit countChanged(this->rowCount());

    this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
    this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
}

void StationListModel::insertStation(int row, Station *station) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginInsertRows(QModelIndex(), row, row);
        m_list.insert(row, station);
        this->endInsertRows();
        emit countChanged(this->rowCount());

        this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
        this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
    }
}

void StationListModel::removeStation(int row) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row)->deleteLater();
        this->endRemoveRows();
        emit countChanged(this->rowCount());
    }
}

void StationListModel::removeStation(int role, const QVariant &value) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removeStation(indexes.first().row());
    }
}

void StationListModel::onStationUpdated(const QString &id, const QVariantMap &properties) {
    Q_UNUSED(properties)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::FavouriteStations:
            if (!indexes.first().data(FavouriteRole).toBool()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Genres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Countries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Languages:
            if (indexes.first().data(LanguageRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void StationListModel::onStationUpdated(const QString &id, const QString &property, const QVariant &value) {
    Q_UNUSED(property)
    Q_UNUSED(value)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::FavouriteStations:
            if (!indexes.first().data(FavouriteRole).toBool()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Genres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Countries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Languages:
            if (indexes.first().data(LanguageRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void StationListModel::onStationDeleted(const QString &id) {
    this->removeStation(IdRole, id);
}
