/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0

MyPage {
    id: root

    function loadStation(stationItem) {
        root.state = "loading";
        root.title = stationItem.title;
        station.load(stationItem.id);
    }

    title: qsTr("Station Info")
    tools: ToolBarLayout {

        BackToolButton {}

        MyToolButton {
            iconSource: "toolbar-mediacontrol-play"
            toolTipText: qsTr("Play")
            enabled: (station.result != null) && ((!MediaPlayer.currentStation) || (MediaPlayer.currentStation.id != station.result.id))
            onClicked: Settings.mediaPlayer == "cuteradio" ? MediaPlayer.play(station.result)
                                                           : Utils.urlIsPlaylist(station.result.source)
                                                             ? StreamExtractor.getStreamUrl(station.result.source)
                                                             : Launcher.launch(station.result.source)
        }

        MyToolButton {
            iconSource: "toolbar-add"
            toolTipText: qsTr("Add to database")
            onClicked: {
                loader.sourceComponent = addStationDialog;
                loader.item.loadStation(station.result);
                loader.item.open();
            }
        }
    }

    TuneInStation {
        id: station

        onFinished: {
            if (station.result) {
                root.state = "";
            }
            else {
                root.state = "error";
                infoBanner.displayMessage(station.errorString);
            }
        }
    }

    MyFlickable {
        id: flicker

        anchors {
            fill: parent
            margins: platformStyle.paddingLarge
        }
        contentHeight: column.height + platformStyle.paddingLarge

        Column {
            id: column

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }
            spacing: platformStyle.paddingLarge

            Logo {
                id: logo

                width: 120
                height: 120
                image: station.result ? station.result.logo : ""
                text: station.result ? station.result.title : ""
                enabled: (station.result != null) && ((!MediaPlayer.currentStation) || (MediaPlayer.currentStation.id != station.result.id))
                onClicked: Settings.mediaPlayer == "cuteradio" ? MediaPlayer.play(station.result)
                                                               : Utils.urlIsPlaylist(station.result.source)
                                                                 ? StreamExtractor.getStreamUrl(station.result.source)
                                                                 : Launcher.launch(station.result.source)
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                text: station.result ? station.result.description : ""
            }

            SeparatorLabel {
                width: parent.width
                text: qsTr("Properties")
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: station.result ? qsTr("Genre") + ": " + station.result.genre : ""
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: station.result ? qsTr("Country") + ": " + station.result.country : ""
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: station.result ? qsTr("Language") + ": " + station.result.language : ""
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: station.result ? qsTr("Source") + ": " + station.result.source : ""
            }
        }
    }

    MyScrollBar {
        flickableItem: flicker
    }

    Label {
        id: errorLabel

        anchors {
            fill: parent
            margins: platformStyle.paddingLarge
        }
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        wrapMode: Text.WordWrap
        color: platformStyle.colorNormalMid
        font.bold: true
        font.pixelSize: 32
        text: qsTr("No information found")
        visible: false
    }

    BusyIndicator {
        id: busyIndicator

        anchors.centerIn: parent
        width: 64
        height: 64
        visible: false
        running: visible
    }

    Loader {
        id: loader
    }

    Component {
        id: addStationDialog

        TuneInAddStationDialog {}
    }

    states: [
        State {
            name: "loading"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: true }
        },

        State {
            name: "finished"
            PropertyChanges { target: flicker; visible: true }
            PropertyChanges { target: busyIndicator; visible: false }
        },

        State {
            name: "error"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: false }
            PropertyChanges { target: errorLabel; visible: true }
        }
    ]
}
