/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QNetworkProxy>
#include "settings.h"

Settings* Settings::self = 0;

Settings::Settings(QObject *parent) :
    QSettings("cuteRadio", "cuteRadio", parent)
{
    if (!self) {
        self = this;
    }
}

Settings::~Settings() {}

Settings* Settings::instance() {
    return !self ? new Settings : self;
}

ScreenOrientation::Orientation Settings::screenOrientation() const {
    return static_cast<ScreenOrientation::Orientation>(this->value("Other/screenOrientation", ScreenOrientation::Automatic).toInt());
}

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        this->setValue("Other/screenOrientation", ScreenOrientation::Orientation(orientation));
        emit screenOrientationChanged(orientation);
    }
}

int Settings::sleepTimerDuration() const {
    return this->value("Media/sleepTimerDuration", 30).toInt();
}

void Settings::setSleepTimerDuration(int duration) {
    if (duration != this->sleepTimerDuration()) {
        this->setValue("Media/sleepTimerDuration", duration);
        emit sleepTimerDurationChanged(duration);
    }
}

QString Settings::language() const {
    return this->value("Other/language", "en").toString();
}

void Settings::setLanguage(const QString &language) {
    if (language != this->language()) {
        this->setValue("Other/language", language);
        emit languageChanged(language);
    }
}

bool Settings::networkProxyEnabled() const {
    return this->value("Network/networkProxyEnabled", false).toBool();
}

void Settings::setNetworkProxyEnabled(bool enabled) {
    if (enabled != this->networkProxyEnabled()) {
        this->setValue("Network/networkProxyEnabled", enabled);
        emit networkProxyEnabledChanged(enabled);
    }
}

NetworkProxyType::ProxyType Settings::networkProxyType() const {
    return static_cast<NetworkProxyType::ProxyType>(this->value("Network/networkProxyType", NetworkProxyType::HttpProxy).toInt());
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType type) {
    if (type != this->networkProxyType()) {
        this->setValue("Network/networkProxyType", NetworkProxyType::ProxyType(type));
        emit networkProxyTypeChanged(type);
    }
}

QString Settings::networkProxyHostName() const {
    return this->value("Network/networkProxyHostName").toString();
}

void Settings::setNetworkProxyHostName(const QString &host) {
    if (host != this->networkProxyHostName()) {
        this->setValue("Network/networkProxyHostName", host);
        emit networkProxyHostNameChanged(host);
    }
}

quint16 Settings::networkProxyPort() const {
    return this->value("Network/networkProxyPort", 80).toLongLong();
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        this->setValue("Network/networkProxyPort", port);
        emit networkProxyPortChanged(port);
    }
}

QString Settings::networkProxyUser() const {
    return this->value("Network/networkProxyUser").toString();
}

void Settings::setNetworkProxyUser(const QString &username) {
    if (username != this->networkProxyUser()) {
        this->setValue("Network/networkProxyUser", username);
        emit networkProxyUserChanged(username);
    }
}

QString Settings::networkProxyPassword() const {
    return QByteArray::fromBase64(this->value("Network/networkProxyPassword").toByteArray());
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        this->setValue("Network/networkProxyPassword", password.toUtf8().toBase64());
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    if (!this->networkProxyEnabled()) {
        QNetworkProxy::setApplicationProxy(QNetworkProxy::NoProxy);
        return;
    }

    QNetworkProxy proxy;
    proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));
    proxy.setHostName(this->networkProxyHostName());
    proxy.setPort(this->networkProxyPort());

    if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
        proxy.setUser(this->networkProxyUser());
        proxy.setPassword(this->networkProxyPassword());
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString Settings::activeColor() const {
    return this->value("Appearance/activeColor", "#0881cb").toString();
}

#ifndef Q_OS_SYMBIAN
void Settings::setActiveColor(const QString &color) {
    if (color != this->activeColor()) {
        this->setValue("Appearance/activeColor", color);
        emit activeColorChanged(color);
    }
}
#endif

QString Settings::activeColorString() const {
    return this->value("Appearance/activeColorString", "color7").toString();
}

#ifndef Q_OS_SYMBIAN
void Settings::setActiveColorString(const QString &colorString) {
    if (colorString != this->activeColorString()) {
        this->setValue("Appearance/activeColorString", colorString);
        emit activeColorStringChanged(colorString);
    }
}
#endif

#endif
