/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.meego 1.0
import com.cuteradio.enums 1.0
import "file:///usr/lib/qt4/imports/com/nokia/meego/UIConstants.js" as UI

MyPage {
    id: root

    title: !MediaPlayer.currentStation.title ? qsTr("Now Playing") : MediaPlayer.currentStation.title
    hideNowPlayingIndicator: true
    tools: ToolBarLayout {

        BackToolIcon {}
    }

    Flickable {
        id: flicker

        anchors {
            left: parent.left
            leftMargin: UI.PADDING_DOUBLE
            right: parent.right
            rightMargin: UI.PADDING_DOUBLE
            top: parent.top
            bottom: controls.top
            bottomMargin: UI.PADDING_DOUBLE
        }
        clip: true
        contentHeight: column.height + UI.PADDING_DOUBLE

        Column {
            id: column

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }
            spacing: UI.PADDING_DOUBLE

            Item {
                width: parent.width
                height: UI.PADDING_DOUBLE
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                text: !MediaPlayer.description ? MediaPlayer.currentStation.description : MediaPlayer.description
            }

            SeparatorLabel {
                width: parent.width
                text: qsTr("Now playing")
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: UI.FONT_SMALL
                text: !MediaPlayer.title ? qsTr("Title") + ": " + MediaPlayer.currentStation.title :
                                           qsTr("Title") + ": " + MediaPlayer.title
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Artist") + ": " + MediaPlayer.artist
                visible: MediaPlayer.artist != ""
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Album title") + ": " + MediaPlayer.album
                visible: MediaPlayer.album != ""
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: UI.FONT_SMALL
                text: !MediaPlayer.genre ? qsTr("Genre") + ": " + MediaPlayer.currentStation.genre :
                                           qsTr("Genre") + ": " + MediaPlayer.genre
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                font.pixelSize: UI.FONT_SMALL
                text: !MediaPlayer.bitRate ? qsTr("Bitrate") + ": " + qsTr("Unknown")
                                           : qsTr("Bitrate") + ": " + Utils.fileSizeFromBytes(MediaPlayer.bitRate) + "/s"
            }
        }
    }

    ScrollDecorator {
        flickableItem: flicker
    }

    Column {
        id: controls

        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
            margins: UI.PADDING_DOUBLE
        }
        spacing: UI.PADDING_XXLARGE

        Rectangle {
            height: 1
            width: parent.width
            color: UI.COLOR_INVERTED_SECONDARY_FOREGROUND
            visible: appWindow.inPortrait
        }

        ButtonRow {
            width: parent.width
            exclusive: false

            MyButton {
                id: playButton

                height: 100
                iconSource: MediaPlayer.playing ? "image://theme/icon-m-toolbar-mediacontrol-stop-white"
                                                : "image://theme/icon-m-toolbar-mediacontrol-play-white"
                onClicked: MediaPlayer.playing ? MediaPlayer.stop() : MediaPlayer.play()
            }

            MyButton {
                height: 100
                iconSource: "image://theme/icon-m-toolbar-clock-white"
                text: MediaPlayer.sleepTimerEnabled ? Utils.durationFromMSecs(MediaPlayer.sleepTimerRemaining) : ""
                checkable: true
                enabled: MediaPlayer.playing
                onEnabledChanged: if (!enabled) checked = false;
                onClicked: MediaPlayer.sleepTimerEnabled = !MediaPlayer.sleepTimerEnabled
                Component.onCompleted: checked = MediaPlayer.sleepTimerEnabled
            }
        }

        MyProgressBar {
            id: progressBar

            width: parent.width
            maximumValue: MediaPlayer.duration
            visible: MediaPlayer.playbackMode !== PlaybackMode.Radio
            value: visible ? MediaPlayer.position : 0

            SeekBubble {
                id: seekBubble

                anchors.bottom: parent.top
                opacity: value != "" ? 1 : 0
                value: (seekMouseArea.drag.active) && (seekMouseArea.posInsideDragArea) ? Utils.durationFromMSecs(Math.floor((seekMouseArea.mouseX / seekMouseArea.width) * MediaPlayer.duration)) : ""
            }

            MouseArea {
                id: seekMouseArea

                property bool posInsideMouseArea: false
                property bool posInsideDragArea: (seekMouseArea.mouseX >= 0) && (seekMouseArea.mouseX <= seekMouseArea.drag.maximumX)

                width: parent.width
                height: 60
                anchors.centerIn: parent
                drag.target: seekBubble
                drag.axis: Drag.XAxis
                drag.minimumX: -40
                drag.maximumX: width - 10
                enabled: MediaPlayer.playbackMode !== PlaybackMode.Radio
                onExited: posInsideMouseArea = false
                onEntered: posInsideMouseArea = true
                onPressed: {
                    posInsideMouseArea = true;
                    seekBubble.x = mouseX - 40;
                }
                onReleased: {
                    if (posInsideMouseArea) {
                        MediaPlayer.position = Math.floor((mouseX / width) * MediaPlayer.duration);
                    }
                }
            }
        }

        Row {
            Label {
                id: positionLabel

                width: controls.width - durationLabel.width
                verticalAlignment: Text.AlignVCenter
                font.pixelSize: UI.FONT_SMALL
                text: Utils.durationFromMSecs(MediaPlayer.position)
            }

            Label {
                id: durationLabel

                verticalAlignment: Text.AlignVCenter
                font.pixelSize: UI.FONT_SMALL
                text: Utils.durationFromMSecs(MediaPlayer.duration)
                visible: MediaPlayer.playbackMode !== PlaybackMode.Radio
            }
        }

        Label {
            width: parent.width
            elide: Text.ElideRight
            wrapMode: Text.WordWrap
            font.pixelSize: UI.FONT_SMALL
            text: qsTr("Status") + ": " + MediaPlayer.stateString
        }
    }

    states: State {
        name: "landscape"
        when: !appWindow.inPortrait
        AnchorChanges { target: flicker; anchors { right: parent.horizontalCenter; bottom: parent.bottom } }
        AnchorChanges { target: controls; anchors { left: parent.horizontalCenter; bottom: undefined; verticalCenter: parent.verticalCenter } }
    }

    Connections {
        target: platformWindow
        onViewModeChanged: MediaPlayer.tickInterval = (platformWindow.viewMode === WindowState.Fullsize ? 1000 : 10000)
    }

    onStatusChanged: MediaPlayer.tickInterval = (status === PageStatus.Active ? 1000 : 0)
}
