/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QVBoxLayout>
#include <QRegExpValidator>
#include <QMenuBar>
#include <QMaemo5InformationBox>
#include "mainwindow.h"
#include "listview.h"
#include "homescreenlistdelegate.h"
#include "browselistwindow.h"
#include "stationlistwindow.h"
#include "nowplayingbutton.h"
#include "addstationdialog.h"
#include "settingsdialog.h"
#include "aboutdialog.h"
#include "../base/homescreenmodel.h"
#include "../base/searchlistmodel.h"
#include "../base/database.h"
#include "../mediaplayer/phonon/mediaplayer.h"

MainWindow::MainWindow(QWidget *parent) :
    StackedWindow(parent),
    m_model(new HomescreenModel(this)),
    m_searchModel(new SearchListModel(this)),
    m_searchEdit(new QLineEdit(this)),
    m_view(new ListView(this)),
    m_nowPlayingButton(new NowPlayingButton(this))
{
    this->setWindowTitle(tr("cuteRadio"));
    this->setCentralWidget(new QWidget(this));
    this->menuBar()->addAction(tr("Add station"), this, SLOT(showAddStationDialog()));
    this->menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()));
    this->menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()));
    this->setAttribute(Qt::WA_DeleteOnClose, false);

    m_searchEdit->setPlaceholderText(tr("Search"));
    m_searchEdit->setValidator(new QRegExpValidator(QRegExp("^\\S.+"), m_searchEdit));

    m_view->setModel(m_model);
    m_view->setItemDelegate(new HomescreenListDelegate(m_view));

    m_vbox = new QVBoxLayout(this->centralWidget());
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_searchEdit);
    m_vbox->addWidget(m_view);

    this->connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(searchStations()));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    this->connect(Database::instance(), SIGNAL(alert(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(Database::instance(), SIGNAL(error(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(MediaPlayer::instance(), SIGNAL(error(QString)), this, SLOT(showInformationBanner(QString)));
}

void MainWindow::onItemClicked(const QModelIndex &index) {
    switch (index.row()) {
    case 0:
        this->showGenres();
        return;
    case 1:
        this->showCountries();
        return;
    case 2:
        this->showLanguages();
        return;
    case 3:
        this->showRecentlyPlayedStations();
        return;
    case 4:
        this->showFavouriteStations();
        return;
    default:
        return;
    }
}

void MainWindow::searchStations() {
    StationListWindow *window = new StationListWindow(this);
    window->searchStations(m_searchEdit->text());
    window->show();
    m_searchModel->addSearch(m_searchEdit->text());
    m_searchEdit->clear();
}

void MainWindow::showGenres() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showGenres();
    window->show();
}

void MainWindow::showCountries() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showCountries();
    window->show();
}

void MainWindow::showLanguages() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showLanguages();
    window->show();
}

void MainWindow::showRecentlyPlayedStations() {
    StationListWindow *window = new StationListWindow(this);
    window->showRecentlyPlayedStations();
    window->show();
}

void MainWindow::showFavouriteStations() {
    StationListWindow *window = new StationListWindow(this);
    window->showFavouriteStations();
    window->show();
}

void MainWindow::showAddStationDialog() {
    AddStationDialog *dialog = new AddStationDialog(this);
    dialog->open();
}

void MainWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void MainWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}

void MainWindow::showInformationBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message);
}
