/*
 * Copyright (C) 2015 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtQml 2.2
import QtQuick.Controls 1.1
import QtQuick.Layouts 1.1
import cuteNews 1.0

ApplicationWindow {
    id: window
    
    minimumWidth: 800
    minimumHeight: 800
    visible: true
    title: "cuteNews"
    menuBar: MenuBar {
        Menu {
            id: subscriptionsMenu
            
            title: qsTr("Subscriptions")
            
            MenuItem {
                action: Action {
                    id: newSubscriptionAction
                    
                    text: qsTr("New subscription")
                    iconName: "list-add"
                    shortcut: "Ctrl+N"
                    onTriggered: {
                        loader.sourceComponent = subscriptionDialog;
                        loader.item.open();
                    }
                }
            }
            
            MenuItem {
                action: Action {
                    id: updateAllAction
                    
                    text: qsTr("Update all")
                    iconName: "view-refresh"
                    shortcut: "Ctrl+R"
                    onTriggered: {
                        for (var i = 0; i < subscriptionModel.count; i++) {
                            subscriptions.update(subscriptionModel.data(i, "id"));
                        }
                    }
                }
            }
            
            MenuItem {
                action: Action {
                    id: quitAction
                    
                    text: qsTr("Quit")
                    iconName: "application-exit"
                    shortcut: "Ctrl+Q"
                    onTriggered: Qt.quit()
                }
            }
        }
        
        Menu {
            id: subscriptionMenu
            
            title: qsTr("Feed")
            
            MenuItem {
                action: Action {
                    id: updateSubscriptionAction
                    
                    text: qsTr("Update")
                    iconName: "view-refresh"
                    onTriggered: subscriptions.update(subscriptionModel.data(subscriptionsView.currentRow, "id"))
                }
            }
            
            MenuItem {
                action: Action {
                    id: readSubscriptionAction
                    
                    text: qsTr("Mark all as read")
                    iconName: "dialog-yes"
                    onTriggered: if (!subscriptionModel.data(subscriptionsView.currentRow, "read"))
                    database.markSubscriptionRead(subscriptionModel.data(subscriptionsView.currentRow, "id"), true);
                }
            }
            
            MenuItem {
                action: Action {
                    id: deleteSubscriptionAction
                    
                    text: qsTr("Delete")
                    iconName: "edit-delete"
                    onTriggered: database.deleteSubscription(subscriptionModel.data(subscriptionsView.currentRow, "id"))
                }
            }
            
            MenuItem {
                action: Action {
                    id: subscriptionPropertiesAction
                    
                    text: qsTr("Properties")
                    iconName: ""
                }
            }
        }
        
        Menu {
            id: articleMenu
            
            title: qsTr("Item")
            
            MenuItem {
                action: Action {
                    id: unreadArticleAction
                    
                    text: qsTr("Next unread item")
                    onTriggered: {
                        var i = articleModel.match("read", false);
                        
                        if (i >= 0) {
                            articlesView.currentRow = i;
                        }
                    }
                }
            }
            
            MenuItem {
                action: Action {
                    id: toggleArticleReadAction
                    
                    text: qsTr("Toggle read status")
                    shortcut: "Ctrl+M"
                    onTriggered: database.markArticleRead(articleModel.data(articlesView.currentRow, "id"),
                                                          !articleModel.data(articlesView.currentRow, "read"))
                }
            }
            
            MenuItem {
                action: Action {
                    id: toggleArticleFavouriteAction
                    
                    text: qsTr("Toggle item flag")
                    shortcut: "Ctrl+T"
                    onTriggered: database.markArticleFavourite(articleModel.data(articlesView.currentRow, "id"),
                                                               !articleModel.data(articlesView.currentRow, "favourite"))
                }
            }
            
            MenuItem {
                action: Action {
                    id: deleteArticleAction
                    
                    text: qsTr("Delete")
                    iconName: "edit-delete"
                    onTriggered: database.deleteArticle(articleModel.data(articlesView.currentRow, "id"))
                }
            }
        }
    }
    
    toolBar: ToolBar {
        RowLayout {            
            ToolButton {
                action: newSubscriptionAction
            }
            
            ToolButton {
                action: readSubscriptionAction
            }
            
            ToolButton {
                action: unreadArticleAction
            }
            
            ToolButton {
                action: updateAllAction
            }
        }
    }
    
    SplitView {
        id: horizontalSplit
        
        anchors.fill: parent
    
        TableView {
            id: subscriptionsView
        
            Layout.minimumWidth: 100
            Layout.maximumWidth: 400
            width: 200
            alternatingRowColors: false
            headerVisible: false
            model: SubscriptionModel {
                id: subscriptionModel
            }
            itemDelegate: SubscriptionDelegate {
                onPressed: {
                    subscriptionsView.forceActiveFocus();
                    subscriptionsView.selection.clear();
                    subscriptionsView.selection.select(styleData.row);
                    subscriptionsView.currentRow = styleData.row;
                }
                onRightClicked: subscriptionsContextMenu.popup()
            }
        
            TableViewColumn {
                role: "title"
            }

            Menu {
                id: subscriptionsContextMenu
                
                MenuItem {
                    action: updateSubscriptionAction
                }
                
                MenuItem {
                    action: readSubscriptionAction
                }
                
                MenuItem {
                    action: deleteSubscriptionAction
                }
                
                MenuItem {
                    action: subscriptionPropertiesAction
                }
            }
            
            onCurrentRowChanged: articleModel.load(subscriptionModel.data(currentRow, "id"))
            onDoubleClicked: subscriptionsContextMenu.popup()
        }
        
        SplitView {
            id: verticalSplit
            
            orientation: Qt.Vertical
    
            TableView {
                id: articlesView
        
                Layout.minimumHeight: 100
                Layout.maximumHeight: 500
                height: 200
                alternatingRowColors: false
                model: ArticleModel {
                    id: articleModel
                }
                itemDelegate: ArticleDelegate {
                    onPressed: {
                        articlesView.forceActiveFocus();
                        articlesView.selection.clear();
                        articlesView.selection.select(styleData.row);
                        articlesView.currentRow = styleData.row;
                    }
                    onRightClicked: articlesContextMenu.popup()
                }
                
                TableViewColumn {
                    role: "favourite"
                }
        
                TableViewColumn {
                    title: qsTr("Date")
                    role: "date"
                }
        
                TableViewColumn {
                    title: qsTr("Title")
                    role: "title"
                }
                
                Menu {
                    id: articlesContextMenu
                    
                    MenuItem {
                        action: toggleArticleReadAction
                    }
                    
                    MenuItem {
                        action: toggleArticleFavouriteAction
                    }
                    
                    MenuItem {
                        action: deleteArticleAction
                    }
                }
        
                onCurrentRowChanged: articleView.article = articleModel.get(currentRow)
                onDoubleClicked: articlesContextMenu.popup()
            }

            GridLayout {
                id: articleView
                
                property Article article
                
                columns: 2
                
                Label {
                    text: qsTr("Filed under")
                }
                
                Label {
                    Layout.fillWidth: true
                    wrapMode: Text.WordWrap
                    font.bold: true
                    text: articleView.article ? articleView.article.categories.join(", ") : ""
                }
                
                Label {
                    text: qsTr("Creator")
                }
                
                Label {
                    Layout.fillWidth: true
                    font.bold: true
                    wrapMode: Text.WordWrap
                    text: articleView.article ? articleView.article.author : ""
                }
                                
                TextArea {
                    Layout.columnSpan: 2
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    textFormat: TextEdit.RichText
                    readOnly: true
                    baseUrl: articleView.article ? articleView.article.url : ""
                    text: articleView.article ? articleView.article.body : ""
                }
                
                Label {
                    Layout.columnSpan: 2
                    Layout.fillWidth: true
                    font.bold: true
                    elide: Text.ElideRight
                    text: qsTr("Enclosures")
                    visible: repeater.count > 0
                }
                
                Repeater {
                    id: repeater
                    
                    model: articleView.article ? articleView.article.enclosures : []
                    
                    Label {
                        Layout.columnSpan: 2
                        Layout.fillWidth: true
                        elide: Text.ElideRight
                        text: utils.formatBytes(modelData.length) + " " + utils.toRichText(modelData.url) + " "
                              + modelData.type
                        onLinkActivated: if (!urlopener.open(modelData.url)) Qt.openUrlExternally(modelData.url);
                    }
                }
                
                onArticleChanged: if ((article) && (!article.read)) database.markArticleRead(article.id, true);
            }
        }
    }
    
    SystemPalette {
        id: palette
    }
    
    Loader {
        id: loader
    }
    
    Component {
        id: subscriptionDialog
        
        SubscriptionDialog {}
    }
    
    Component.onCompleted: {
        database.init();
        subscriptionModel.load();
        urlopener.load();
        downloads.load();
    }
}
