/*
 *  CUE Sheet Widget for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libosso.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <glib/gi18n-lib.h>

#include <libmafw/mafw.h>
#include <libmafw/mafw-log.h>
#include <libmafw/mafw-registry.h>
#include <libmafw-shared/mafw-shared.h>

#include <libcue/libcue.h>
#include <libcue/time.h>

#include "cuesheet_widget.h"

#define BACKGROUND_IMAGE DATADIR"/themes/alpha/images/wmIncomingEvent.png"

#define WANTED_RENDERER	"Mafw-Gst-Renderer"
#define WANTED_SOURCE	"Mafw-Tracker-Source"
#define WANTED_PLAYLIST	"FmpAudioPlaylist"

#define _MP(str) dgettext("mediaplayer",str)

HD_DEFINE_PLUGIN_MODULE (CueSheetWidget, cuesheet_widget, HD_TYPE_HOME_PLUGIN_ITEM)

typedef struct _TrackInfoStruct TrackInfoStruct;

struct _TrackInfoStruct {
	unsigned int track_id;
	long start_time;
	long end_time;
	char *performer;
	char *title;
};

static void
cuesheet_widget_renderer_get_position_cb (MafwRenderer *self,
					  gint position,
					  gpointer user_data,
					  const GError *error)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);
	GList *track = NULL;

	g_return_if_fail (priv);

	if (error != NULL) {
		g_warning ("cuesheet_widget_renderer_get_position_cb: %s", error->message);
		return;
	}

	for (track = priv->track_list; track != NULL; track = track->next) {
		TrackInfoStruct *track_info = track->data;

		if (position >= track_info->start_time) {
			if ((track_info->start_time == track_info->end_time) ||
			    (position < track_info->end_time)) {

				if (priv->current_track != track_info->track_id) {
					gtk_label_set_text (GTK_LABEL (priv->performer), track_info->performer);
					gtk_label_set_text (GTK_LABEL (priv->title), track_info->title);
					gtk_widget_show (priv->title);
					priv->current_track = track_info->track_id;
				}
				break;
			}
		}
	}
}

static gboolean
cuesheet_widget_renderer_on_timeout (gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_val_if_fail (priv, FALSE);

	mafw_renderer_get_position (priv->renderer, cuesheet_widget_renderer_get_position_cb, priv);

	return TRUE;
}

static void
cuesheet_widget_renderer_state_changed_cb (MafwRenderer *renderer,
					   MafwPlayState state,
					   gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	switch (state) {
		case Playing:
			if (priv->cuesheet_available) {
				priv->timer = g_timeout_add (2000, cuesheet_widget_renderer_on_timeout,
							     priv);
			}
			break;

		case Stopped:
			/* Remove the timout handler: */
			if (priv->timer) {
				g_source_remove (priv->timer);
				priv->timer = 0;
			}

			gtk_label_set_text (GTK_LABEL (priv->performer), _("(no track informations)"));
			gtk_widget_hide (priv->title);
			break;

		case Paused:
		case Transitioning:
			/* Remove the timout handler: */
			if (priv->timer) {
				g_source_remove (priv->timer);
				priv->timer = 0;
			}
			break;
		default:
			break;
	}
}

static void
cuesheet_widget_metadata_request_cb (MafwSource *source,
				     const gchar *object_id,
				     GHashTable *metadata,
				     gpointer user_data,
				     const GError *error)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (error != NULL) {
		g_warning ("cuesheet_widget_metadata_request_cb: %s", error->message);
		priv->cuesheet_available = FALSE;
		return;
	}

	if (metadata != NULL) {
		const gchar *uri = NULL;
		const gchar *filename = NULL;
		GValue *value;
		FILE *fd;
		Cd *cd;
		Track *track;
		Cdtext *cdtext;
		unsigned int i = 0;

		value = mafw_metadata_first (metadata, MAFW_METADATA_KEY_URI);
		if (value == NULL) {
			priv->cuesheet_available = FALSE;
			return;
		}

		uri = g_value_get_string (value);
		if (g_str_has_prefix (uri, "file://")) {
			gchar *tmp = NULL;

			tmp = g_strndup (uri, strlen (uri) - 4);
			filename = g_strdup_printf ("%s.cue", tmp);
			g_free (tmp);
		}

		/* display artist and title from metadata */
		value = mafw_metadata_first (metadata, MAFW_METADATA_KEY_ARTIST);
		if (value != NULL) {
			gtk_label_set_text (GTK_LABEL (priv->performer), g_value_get_string (value));
		} else {
			gtk_label_set_text (GTK_LABEL (priv->performer), _MP("mp_li_unknown_artist"));
		}

		value = mafw_metadata_first (metadata, MAFW_METADATA_KEY_TITLE);
		if (value != NULL) {
			gtk_label_set_text (GTK_LABEL (priv->title), g_value_get_string (value));
		} else {
			gtk_label_set_text (GTK_LABEL (priv->title), _MP("mp_li_unknown_song"));
		}
		gtk_widget_show (priv->title);

		value = mafw_metadata_first (metadata, MAFW_METADATA_KEY_ALBUM_ART_URI);
		if (value != NULL) {
			g_print("album-art-uri: %s\n", g_value_get_string (value));
		}

		/* destroy previous track list */
		if (priv->track_list) {
			g_list_foreach (priv->track_list, (GFunc) g_free, NULL);
			g_list_free (priv->track_list);
			priv->track_list = NULL;
		}

		g_print ("cuesheet_widget_metadata_request_cb: opening %s\n", filename);

		fd = g_fopen (g_filename_from_uri (filename, NULL, NULL), "r");
		if (!fd) {
			g_print ("cuesheet_widget_metadata_request_cb: unable to open\n");
			priv->cuesheet_available = FALSE;
			return;
		}

		cd = cd_init ();
		cd = cue_parse_file (fd);

		/* error parsing cuesheet file */
		if (cd == NULL) {
			g_warning ("cuesheet_widget_metadata_request_cb: error parsing cue sheet file");

			if (fd != NULL)
				fclose (fd);
			return;
		}

		track = track_init ();
		cdtext = cdtext_init ();

		priv->cuesheet_available = TRUE;
		priv->current_track = 0;

		for (i = 1; i <= cd_get_ntrack (cd); i++) {
			TrackInfoStruct *track_info;
			long start_time = 0, length = 0;
			int min = 0, sec = 0, frames = 0;

			track = cd_get_track (cd, i);
			cdtext = track_get_cdtext (track);

			if (track_get_mode (track) != MODE_AUDIO)
				continue;

			if (cdtext_is_empty (cdtext) == 0)
				continue;

			/* convert times in frames to number of seconds */
			time_frame_to_msf (track_get_start (track), &min, &sec, &frames);
			start_time = 60 * min + sec;
			time_frame_to_msf (track_get_length (track), &min, &sec, &frames);
			length = 60 * min + sec;

			track_info = g_new0 (TrackInfoStruct, 1);
			track_info->track_id = i;
			track_info->start_time = start_time;
			track_info->end_time = start_time + length;
			track_info->performer = g_strdup (cdtext_get (PTI_PERFORMER, cdtext));
			track_info->title = g_strdup (cdtext_get (PTI_TITLE, cdtext));
			priv->track_list = g_list_append (priv->track_list, track_info);
		}

		if (cd != NULL)
			cd_delete (cd);

		if (fd != NULL)
			fclose (fd);
	} else {
		priv->cuesheet_available = FALSE;
	}
}

static void
cuesheet_widget_renderer_media_changed_cb (MafwRenderer *renderer,
					   gint index,
					   const gchar *object_id,
					   gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);
	const gchar *const *keys;

	g_return_if_fail (priv);

	if (priv->source == NULL) {
		g_critical ("cuesheet-widget: Source %s is not available\n", WANTED_SOURCE);
		return;
	}

	g_print ("cuesheet-widget: Media changed: assigned media is %d - %s\n", index, object_id);

	if (index >= 0) {
		keys = MAFW_SOURCE_LIST (MAFW_METADATA_KEY_URI,
					 MAFW_METADATA_KEY_ARTIST,
					 MAFW_METADATA_KEY_TITLE,
					 MAFW_METADATA_KEY_ALBUM_ART_URI);

		mafw_source_get_metadata (priv->source,
					  object_id,
					  keys,
					  cuesheet_widget_metadata_request_cb,
					  priv);
	} else {
		priv->cuesheet_available = FALSE;
	}
}

static void
cuesheet_widget_renderer_status_cb (MafwRenderer *self,
				    MafwPlaylist *playlist,
				    guint index,
				    MafwPlayState state,
				    const gchar *object_id,
				    gpointer user_data,
				    const GError *error)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (error != NULL) {
		g_warning ("cuesheet_widget_renderer_status_cb: %s", error->message);
		return;
	}

	cuesheet_widget_renderer_media_changed_cb (self, index, object_id, priv);

	if (state == Playing) {
		priv->timer = g_timeout_add (2000, cuesheet_widget_renderer_on_timeout,
					     priv);
	}
}

static void
cuesheet_widget_source_added_cb (MafwRegistry *registry,
				 GObject *source,
				 gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (MAFW_IS_SOURCE(source)) {
		const gchar *name = mafw_extension_get_name(MAFW_EXTENSION(source));

		if (strcmp (name, WANTED_SOURCE) == 0) {
			g_print ("cuesheet-widget: Source %s available.\n", name);
			priv->source = g_object_ref (source);
		}

	}
}

static void
cuesheet_widget_source_removed_cb (MafwRegistry *registry,
				   GObject *source,
				   gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (MAFW_IS_SOURCE(source)) {
		if (MAFW_SOURCE (source) == priv->source) {
			g_critical ("cuesheet-widget: Source %s removed.\n", 
				    mafw_extension_get_name (MAFW_EXTENSION (source)));
			g_object_unref (priv->source);
		}
	}
}

static void
cuesheet_widget_renderer_added_cb (MafwRegistry *registry,
				   GObject *renderer,
				   gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (MAFW_IS_RENDERER(renderer)) {
		const gchar *name = mafw_extension_get_name (MAFW_EXTENSION(renderer));

		if (strcmp (name, WANTED_RENDERER) == 0) {
			g_print ("cuesheet-widget: Renderer %s available.\n", name);
			priv->renderer = g_object_ref (renderer);

			/* Connect to a few interesting signals */
			g_signal_connect (renderer, "media-changed",
					  G_CALLBACK (cuesheet_widget_renderer_media_changed_cb),
					  priv);

			g_signal_connect (renderer, "state-changed",
					  G_CALLBACK (cuesheet_widget_renderer_state_changed_cb),
					  priv);

			/* force update of status */
			mafw_renderer_get_status (MAFW_RENDERER (renderer),
						  cuesheet_widget_renderer_status_cb, priv);
		}
	}
}

static void
cuesheet_widget_renderer_removed_cb (MafwRegistry * registry,
				     GObject *renderer,
				     gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (user_data);

	g_return_if_fail (priv);

	if (MAFW_IS_RENDERER(renderer)) {
		if (MAFW_RENDERER (renderer) == priv->renderer) {
			g_critical ("Renderer %s removed.\n", 
				    mafw_extension_get_name (MAFW_EXTENSION (renderer)));
			g_object_unref (priv->renderer);
		}
	}
}

static GtkWidget *
cuesheet_widget_create_widget (CueSheetWidget *priv)
{
	GtkWidget *alignment, *vbox;
	GdkColor style_color;

	g_return_val_if_fail (priv, NULL);

	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment),
				   HILDON_MARGIN_DEFAULT,
				   HILDON_MARGIN_DEFAULT,
				   HILDON_MARGIN_DOUBLE,
				   HILDON_MARGIN_DOUBLE);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (alignment), vbox);

	priv->performer = gtk_label_new (_("(no track informations)"));
	gtk_misc_set_alignment ( GTK_MISC (priv->performer), 0.5, 0.5);
	gtk_label_set_ellipsize (GTK_LABEL (priv->performer), PANGO_ELLIPSIZE_END);
	gtk_box_pack_start (GTK_BOX (vbox), priv->performer, TRUE, TRUE, 0);

	priv->title = gtk_label_new ("");
	gtk_misc_set_alignment ( GTK_MISC (priv->title), 0.5, 0.5);
	gtk_label_set_ellipsize (GTK_LABEL (priv->title), PANGO_ELLIPSIZE_END);
	gtk_box_pack_start (GTK_BOX (vbox), priv->title, TRUE, TRUE, 0);
	gtk_widget_set_no_show_all (priv->title, TRUE);

	if (!gtk_style_lookup_color (GTK_WIDGET (priv->performer)->style, "SecondaryTextColor",
				     &style_color)) {
		gdk_color_parse ("black", &style_color);
	}
	gtk_widget_modify_fg (GTK_WIDGET (priv->performer), GTK_STATE_NORMAL, &style_color);

	gtk_widget_show_all (GTK_WIDGET (alignment));
	return GTK_WIDGET (alignment);
}

static void
cuesheet_widget_on_button_release_event (GtkWidget *widget,
					 gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (widget);

	g_return_if_fail (priv);
	g_return_if_fail (priv->osso);

	if (osso_rpc_run (priv->osso,
			  "com.nokia.mediaplayer",
			  "/com/nokia/mediaplayer",
			  "com.nokia.mediaplayer",
			  "open_mp_main_view",
			  NULL,
			  DBUS_TYPE_INVALID) != OSSO_OK) {
		g_warning ("cuesheet-widget_on_button_release_event: Failed to launch mediaplayer");
	}
}

static void
cuesheet_widget_on_current_desktop (GtkWidget *widget,
				    gpointer user_data)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (widget);
	gboolean on;

	g_return_if_fail (priv);

	g_object_get (widget, "is-on-current-desktop", &on, NULL);
	if (!on) {
		/* Remove the timout handler: */
		if (priv->timer) {
			g_source_remove (priv->timer);
			priv->timer = 0;
		}
	} else {
		if (priv->cuesheet_available) {
			priv->timer = g_timeout_add (2000, cuesheet_widget_renderer_on_timeout,
						     priv);
		}
	}
}

static void
cuesheet_widget_realize (GtkWidget *widget)
{
	/* Use An RGBA colormap rather than RGB,
	 * so we can use transparency in our expose_event() implementation. 
	 */
	GdkScreen *screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	/* Call the base class's implementation: */
	GTK_WIDGET_CLASS (cuesheet_widget_parent_class)->realize (widget);
}

static gboolean
cuesheet_widget_expose_event (GtkWidget *widget,
			      GdkEventExpose *event)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (widget);

	g_return_val_if_fail (priv, FALSE);

	cairo_t *cr  = gdk_cairo_create (widget->window);

	gdk_cairo_region (cr, event->region);
	cairo_clip (cr);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	if (priv->background) {
		cairo_set_source_surface(cr, priv->background, 0.0, 0.0);
	} else {
		cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 0.9);
	}

	cairo_paint (cr);

	cairo_destroy (cr);

	return GTK_WIDGET_CLASS (cuesheet_widget_parent_class)->expose_event (widget, event);
}

static void
cuesheet_widget_dispose (GObject *object)
{
	CueSheetWidget *priv = CUESHEET_WIDGET (object);

	g_return_if_fail (priv);

	/* Remove the timout handler: */
	if (priv->timer) {
		g_source_remove (priv->timer);
		priv->timer = 0;
	}

	if (priv->track_list) {
		g_list_foreach (priv->track_list, (GFunc) g_free, NULL);
		g_list_free (priv->track_list);
		priv->track_list = NULL;
	}

	if (priv->osso) {
		osso_deinitialize (priv->osso);
		priv->osso = NULL;
	}

	/* Call the base class's implementation: */
	G_OBJECT_CLASS (cuesheet_widget_parent_class)->dispose (object);
}

static void
cuesheet_widget_finalize (GObject *object)
{
	/* Call the base class's implementation: */
	G_OBJECT_CLASS (cuesheet_widget_parent_class)->finalize (object);
}

static void
cuesheet_widget_init (CueSheetWidget *priv)
{
	GtkWidget *widget;
	GError *error = NULL;
	MafwRegistry *registry = NULL;
	GList *extension_list = NULL;

	priv->cuesheet_available = FALSE;
	priv->current_track = -1;
	priv->track_list =  NULL;
	priv->timer = 0;

	priv->osso = osso_initialize ("cuesheet-widget", "0.1", TRUE, NULL);
	if (priv->osso == NULL) {
		g_error ("cuesheet-widget: Failed to initialize OSSO\n");
		return;
	}

	widget = cuesheet_widget_create_widget (priv);
	gtk_container_add (GTK_CONTAINER (priv), widget);

	g_signal_connect (priv, "button-release-event",
			  G_CALLBACK (cuesheet_widget_on_button_release_event), NULL);
	g_signal_connect (priv, "notify::is-on-current-desktop",
			  G_CALLBACK (cuesheet_widget_on_current_desktop), NULL);

	gtk_widget_set_double_buffered (GTK_WIDGET (priv), FALSE);
	priv->background = cairo_image_surface_create_from_png (BACKGROUND_IMAGE);
	if (cairo_surface_status (priv->background) != CAIRO_STATUS_SUCCESS) {
		cairo_surface_destroy (priv->background);
		priv->background = NULL;
	}

	if (priv->background) {
		gtk_widget_set_size_request (GTK_WIDGET (priv), cairo_image_surface_get_width (priv->background),
					     cairo_image_surface_get_height (priv->background));
	} else {
		gtk_widget_set_size_request (GTK_WIDGET (priv), 338, 80);
	}

	/* Check available plugins  */
	registry = MAFW_REGISTRY (mafw_registry_get_instance ());
	if (registry == NULL) {
		g_error ("cuesheet-widget: Failed to get MafwRegistry reference\n");
		return;
	}

	/* Start out-of-process extension discovery */
	mafw_shared_init (registry, &error);
	if (error != NULL)
	{
		g_warning ("cuesheet-widget: Ext. discovery failed: %s", error->message);
		g_error_free (error);
		error = NULL;
	}

	/* Connect to extension discovery signals. These signals will be
	   emitted when new extensions are started or removed */
	g_signal_connect (registry, "renderer_added",
			  G_CALLBACK (cuesheet_widget_renderer_added_cb), priv);

	g_signal_connect (registry, "renderer_removed",
			  G_CALLBACK (cuesheet_widget_renderer_removed_cb), priv);

	g_signal_connect (registry, "source_added",
			  G_CALLBACK (cuesheet_widget_source_added_cb), priv);

	g_signal_connect (registry, "source_removed",
			  G_CALLBACK (cuesheet_widget_source_removed_cb), priv);

	/* Also, check for already started extensions */
	extension_list = mafw_registry_get_renderers(registry);
	while (extension_list)
	{
		cuesheet_widget_renderer_added_cb (registry,
						   G_OBJECT (extension_list->data), priv);
		extension_list = g_list_next(extension_list);
	}

	extension_list = mafw_registry_get_sources(registry);
	while (extension_list)
	{
		cuesheet_widget_source_added_cb (registry,
						 G_OBJECT(extension_list->data), priv);
		extension_list = g_list_next(extension_list);
	}

}

static void
cuesheet_widget_class_init (CueSheetWidgetClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

	object_class->dispose = cuesheet_widget_dispose;
	object_class->finalize = cuesheet_widget_finalize;

	widget_class->realize = cuesheet_widget_realize;
	widget_class->expose_event = cuesheet_widget_expose_event;
}

static void
cuesheet_widget_class_finalize (CueSheetWidgetClass *class)
{
}
