/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 *  Copyright (C) 2009 Artem Garmash. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Artem Garmash <artem.garmash@nokia.com>
 *
 */

#include "config.h"
#include "el-home-applet.h"

#include <hildon/hildon.h>
#include <rtcom-eventlogger/eventlogger.h>
#include <sqlite3.h>
#include <string.h>

#define EL_HOME_APPLET_GET_PRIVATE(obj) ( \
        G_TYPE_INSTANCE_GET_PRIVATE (obj, \
                EL_TYPE_HOME_APPLET, ELHomeAppletPrivate))

#define BOX_WIDTH 352
#define BOX_HEIGHT 252

#define C_WIDTH (BOX_WIDTH - 2*HILDON_MARGIN_HALF)
#define C_HEIGHT (BOX_HEIGHT - 2*HILDON_MARGIN_HALF)
#define C_X HILDON_MARGIN_HALF
#define C_Y HILDON_MARGIN_HALF

#define HEADER_HEIGHT 48
#define MESSAGE_HEIGHT (C_HEIGHT - HEADER_HEIGHT)
#define MESSAGE_WIDTH (C_WIDTH - 2*HILDON_MARGIN_DEFAULT)

#define BOX_RADIOUS 10

struct _ELHomeAppletPrivate
{
        RTComEl *eventlogger;

        GtkWidget *sender;
        GtkWidget *message;
        GtkWidget *icon;
        GtkWidget *unread;

        gint       event_id;

        gboolean   active;

        guint unread_count;

        const gchar *current_font;
};

HD_DEFINE_PLUGIN_MODULE (ELHomeApplet, el_home_applet, HD_TYPE_HOME_PLUGIN_ITEM);

const gchar* g_module_check_init(GModule *module);
const gchar*
g_module_check_init(GModule *module)
{
	g_module_make_resident (module);
	return NULL;
}

static void
el_home_applet_class_finalize (ELHomeAppletClass *klass)
{
}

static void
el_home_applet_realize (GtkWidget *widget)
{
        GdkScreen *screen;

        screen = gtk_widget_get_screen (widget);
        gtk_widget_set_colormap (widget,
                                 gdk_screen_get_rgba_colormap (screen));

        gtk_widget_set_app_paintable (widget,
                                      TRUE);

        GTK_WIDGET_CLASS (el_home_applet_parent_class)->realize (widget);
}

/*
 * Thanks http://cairographics.org/cookbook/roundedrectangles/
 */
static void
rounded_rectangle (cairo_t *cr,
                   double   x,
                   double   y,
                   double   w,
                   double   h,
                   double   r)
{
        cairo_move_to (cr, x + r, y);
        cairo_line_to (cr, x + w - r, y);
        cairo_curve_to (cr, x + w, y,
                        x + w, y,
                        x + w, y + r);
        cairo_line_to (cr, x + w, y + h - r);
        cairo_curve_to (cr, x + w, y + h,
                        x + w, y + h,
                        x + w - r, y + h);
        cairo_line_to (cr, x + r, y + h);
        cairo_curve_to (cr, x, y + h,
                        x, y + h,
                        x, y + h - r);
        cairo_line_to (cr, x, y + r);
        cairo_curve_to (cr, x, y,
                        x, y,
                        x + r, y);
}

static gboolean
expose_event (GtkWidget *self, GdkEventExpose *event)
{
        ELHomeAppletPrivate *priv = EL_HOME_APPLET(self)->priv;

        cairo_t *cr;
        GdkColor color;
        float red, green, blue;

        /* find theme active color */
        gtk_style_lookup_color (self->style, "ActiveTextColor", &color);
        red = color.red/(float)G_MAXUINT16;
        green = color.green/(float)G_MAXUINT16;
        blue = color.blue/(float)G_MAXUINT16;

        cr = gdk_cairo_create (self->window);
        gdk_cairo_region (cr, event->region);
        cairo_clip (cr);

        cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

        cairo_set_source_rgba (cr, 0.4f, 0.4f, 0.4f, 0.1f);
        cairo_set_line_width (cr, 3.0f);

        rounded_rectangle (cr,
                           C_X,
                           C_Y,
                           BOX_WIDTH - 2*C_X,
                           BOX_HEIGHT - 2*C_Y,
                           BOX_RADIOUS);

        cairo_close_path(cr);
        cairo_stroke (cr);

        cairo_set_line_width (cr, 1.0f);

        cairo_translate (cr, C_X, C_Y);
        cairo_move_to (cr, 0, HEADER_HEIGHT);
        cairo_line_to (cr, 0, BOX_RADIOUS);
        cairo_curve_to (cr, 0, 0, 0, 0, BOX_RADIOUS, 0);
        cairo_line_to (cr, C_WIDTH - BOX_RADIOUS, 0);
        cairo_curve_to (cr, C_WIDTH, 0, C_WIDTH, 0, C_WIDTH, BOX_RADIOUS);
        cairo_line_to (cr, C_WIDTH, HEADER_HEIGHT);
        cairo_line_to (cr, 0, HEADER_HEIGHT);

        cairo_close_path(cr);

        cairo_set_source_rgba (cr, 0.2f, 0.2f, 0.2f, 0.8f);
        cairo_fill_preserve (cr);
        cairo_set_source_rgba (cr, red, green, blue, 1.0f);
        cairo_stroke (cr);

        cairo_move_to (cr, 0, HEADER_HEIGHT);
        cairo_line_to (cr, 0, C_HEIGHT - BOX_RADIOUS);
        cairo_curve_to (cr, 0, C_HEIGHT, 0, C_HEIGHT, BOX_RADIOUS, C_HEIGHT);
        cairo_line_to (cr, C_WIDTH - BOX_RADIOUS, C_HEIGHT);
        cairo_curve_to (cr, C_WIDTH, C_HEIGHT, C_WIDTH, C_HEIGHT, C_WIDTH, C_HEIGHT - BOX_RADIOUS);
        cairo_line_to (cr, C_WIDTH, HEADER_HEIGHT);
        cairo_line_to (cr, 0, HEADER_HEIGHT);
        cairo_close_path(cr);

        if (priv->active)
                cairo_set_source_rgba (cr, red, green, blue, 0.8f);
        else
                cairo_set_source_rgba (cr, 0.4f, 0.4f, 0.4f, 0.8f);
        cairo_fill (cr);

        /* cairo_set_source_rgba (cr, red, green, blue, 1.0f); */
        /* cairo_translate (cr, -C_X, -C_Y); */
        /* rounded_rectangle (cr, */
        /*                    C_X, */
        /*                    C_Y, */
        /*                    BOX_WIDTH - 2*C_X, */
        /*                    BOX_HEIGHT - 2*C_Y, */
        /*                    BOX_RADIOUS); */
        /* cairo_close_path(cr); */
        /* cairo_stroke (cr); */

        cairo_destroy (cr);

        return GTK_WIDGET_CLASS (el_home_applet_parent_class)->expose_event (self, event);
}

static void
dispose (GObject *self)
{
        ELHomeAppletPrivate *priv = EL_HOME_APPLET(self)->priv;

        if (priv->eventlogger){
                g_object_unref (priv->eventlogger);
                priv->eventlogger = NULL;
        }

        G_OBJECT_CLASS (el_home_applet_parent_class)->dispose (self);
}

static void
finalize (GObject *self)
{
        G_OBJECT_CLASS (el_home_applet_parent_class)->finalize (self);
}

static void
show_event (ELHomeApplet *self, RTComElIter *it)
{
        ELHomeAppletPrivate *priv = self->priv;

        gchar *message = NULL;
        gchar *remote = NULL;
        const gchar *icon_name = NULL;

        if (it && rtcom_el_iter_first (it)){
                rtcom_el_iter_dup_string (it, "free-text", &message);
                if (message){
                        const gchar *service;

                        rtcom_el_iter_get_int (it, "id", &priv->event_id);

                        if(!rtcom_el_iter_dup_string (it, "remote-name", &remote))
                                rtcom_el_iter_dup_string (it, "remote-id", &remote);
                        service = rtcom_el_iter_get_service (it);
                        if (!g_strcmp0 (service, "RTCOM_EL_SERVICE_SMS"))
                                icon_name = "chat_unread_sms";
                        else if (!g_strcmp0 (service, "RTCOM_EL_SERVICE_CHAT"))
                                icon_name = "chat_unread_chat";
                }
        }
        else{
                priv->event_id = -1;
        }

        gtk_label_set_text (GTK_LABEL (priv->message), message);
        gtk_label_set_text (GTK_LABEL (priv->sender), remote);
        if (icon_name){
                const gchar *current_icon_name;
                gtk_image_get_icon_name (GTK_IMAGE (priv->icon),
                                         &current_icon_name,
                                         NULL);
                if (g_strcmp0 (current_icon_name, icon_name))
                        gtk_image_set_from_icon_name (GTK_IMAGE (priv->icon),
                                                      icon_name,
                                                      HILDON_ICON_SIZE_FINGER);
                gtk_widget_show (priv->icon);
        }
        else
                gtk_widget_hide (priv->icon);

        g_free (message);
        g_free (remote);
}

static RTComElIter*
make_query (RTComEl *el, gint event_id)
{
        RTComElQuery *query = NULL;
        RTComElIter *it = NULL;

        static const gchar *services[] = {"RTCOM_EL_SERVICE_SMS",
                                          "RTCOM_EL_SERVICE_CHAT",
                                          NULL};
        static const gchar *event_types[] = {"RTCOM_EL_EVENTTYPE_SMS_INBOUND",
                                             "RTCOM_EL_EVENTTYPE_CHAT_INBOUND",
                                             NULL};

        query = rtcom_el_query_new (el);
        rtcom_el_query_set_limit (query, 1);
        if (event_id >= 0){
                rtcom_el_query_prepare (query,
                                        "is-read", FALSE, RTCOM_EL_OP_EQUAL,
                                        "id", event_id, RTCOM_EL_OP_EQUAL,
                                        "service", services, RTCOM_EL_OP_IN_STRV,
                                        "event-type", event_types, RTCOM_EL_OP_IN_STRV,
                                        NULL);
        }
        else{
                rtcom_el_query_prepare (query,
                                        "is-read", FALSE, RTCOM_EL_OP_EQUAL,
                                        "service", services, RTCOM_EL_OP_IN_STRV,
                                        "event-type", event_types, RTCOM_EL_OP_IN_STRV,
                                        NULL);
        }
        it = rtcom_el_get_events (el, query);
        g_object_unref(query);

        return it;
}

static void
update_unread_label (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        gchar *text;

        text = g_strdup_printf ("%d", priv->unread_count);
        gtk_label_set_text (GTK_LABEL (priv->unread), text);
        g_free (text);
}

static gint
query_unread_events (RTComEl *el)
{
        sqlite3 *db;
        sqlite3_stmt *stmt;
        int ret;
        gint count = 0;

        g_object_get (el, "db", &db, NULL);

        if (sqlite3_prepare_v2 (db,
                                "SELECT SUM(total_events)-SUM(read_events) FROM GroupCache;",
                                -1,
                                &stmt,
                                NULL) != SQLITE_OK){
                g_error ("%s: can't compile SQL", G_STRFUNC);
                return -1;
        }

        while (SQLITE_BUSY == (ret = sqlite3_step (stmt)));

        if (ret == SQLITE_ROW){
                count = sqlite3_column_int (stmt, 0);
        }
        else{
                g_error ("%s: error while executing SQL", G_STRFUNC);
        }

        sqlite3_finalize (stmt);

        return count;
}

static void
read_event (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        RTComElIter *it = NULL;

        it = make_query (priv->eventlogger, -1);
        show_event (self, it);
        if (it) g_object_unref (it);
}

static void
mark_as_read (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->event_id >= 0){
                rtcom_el_set_read_event (priv->eventlogger,
                                         priv->event_id,
                                         TRUE,
                                         NULL);
                read_event (self);
                priv->unread_count--;
                update_unread_label (self);
        }
}

static void
new_event_cb (RTComEl      *backend,
              gint          event_id,
              const gchar  *local_uid,
              const gchar  *remote_uid,
              const gchar  *remote_ebook_uid,
              const gchar  *group_uid,
              const gchar  *service,
              ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;
        RTComElIter *it = NULL;

        it = make_query (priv->eventlogger, event_id);
        if (it){
                if (rtcom_el_iter_first (it)){
                        show_event (self, it);
                        priv->unread_count++;
                        update_unread_label (self);
                }
                g_object_unref (it);
        }
}

static void
event_updated_cb (RTComEl      *backend,
                  gint          event_id,
                  const gchar  *local_uid,
                  const gchar  *remote_uid,
                  const gchar  *remote_ebook_uid,
                  const gchar  *group_uid,
                  const gchar  *service,
                  ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (event_id == priv->event_id)
                read_event (self);

        priv->unread_count = query_unread_events (priv->eventlogger);
        update_unread_label (self);
}

static gboolean
button_release_event_cb (GtkWidget      *widget,
                         GdkEventButton *event,
                         ELHomeApplet   *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->active){
                priv->active = FALSE;
                gtk_widget_queue_draw (widget);
#ifndef DEBUG_LAYOUT
                mark_as_read (self);
#endif
        }

        return TRUE;
}

static gboolean
button_press_event_cb (GtkWidget      *widget,
                       GdkEventButton *event,
                       ELHomeApplet   *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->event_id > 0){
                priv->active = TRUE;
                gtk_widget_queue_draw (widget);
        }

        return TRUE;
}

static gboolean
leave_notify_event_cb (GtkWidget        *widget,
                       GdkEventCrossing *event,
                       ELHomeApplet     *self)
{
        ELHomeAppletPrivate *priv = self->priv;

        if (priv->active){
                priv->active = FALSE;
                gtk_widget_queue_draw (widget);
        }

        return FALSE;
}

static void
el_home_applet_init (ELHomeApplet *self)
{
        ELHomeAppletPrivate *priv;
        GtkWidget *event_box;
        GtkWidget *hbox, *vbox, *align;

        self->priv = EL_HOME_APPLET_GET_PRIVATE (self);
        priv = self->priv;

        gtk_widget_set_app_paintable (GTK_WIDGET (self), TRUE);

        priv->unread = gtk_label_new ("12");
        hildon_helper_set_logical_color (priv->unread,
                                         GTK_RC_FG,
                                         GTK_STATE_NORMAL,
                                         "ActiveTextColor");
        gtk_misc_set_alignment (GTK_MISC (priv->unread),
                                1.0f,
                                0.5f);
        gtk_widget_set_size_request (priv->unread,
                                     -1,
                                     HEADER_HEIGHT);

        priv->icon = gtk_image_new_from_icon_name ("chat_unread_sms",
                                                   HILDON_ICON_SIZE_FINGER);
        gtk_misc_set_alignment (GTK_MISC (priv->icon),
                                0.5f,
                                0.5f);

        priv->sender = gtk_label_new ("asdf asdf asdf asdf asdf");
        gtk_misc_set_alignment (GTK_MISC (priv->sender),
                                0.5f,
                                0.5f);
        gtk_label_set_ellipsize (GTK_LABEL (priv->sender),
                                 PANGO_ELLIPSIZE_END);
        gtk_widget_set_name (priv->sender, "hildon-shadow-label");
        hildon_helper_set_logical_font (priv->sender, "SystemFont");

        priv->message = g_object_new (GTK_TYPE_LABEL,
                                      "label", "asdf asdf adsf asdf asdf asdf asdf asdf",
                                      "wrap", TRUE,
                                      "wrap-mode", PANGO_WRAP_WORD_CHAR,
                                      NULL);

        gtk_misc_set_alignment (GTK_MISC (priv->message),
                                0.0f,
                                0.0f);
        gtk_widget_set_size_request (priv->message,
                                     MESSAGE_WIDTH,
                                     MESSAGE_HEIGHT);
        gtk_widget_set_name (priv->message, "hildon-shadow-label");

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), priv->unread, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), priv->icon, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), priv->sender, TRUE, TRUE, 0);

        vbox = gtk_vbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (vbox), priv->message, TRUE, TRUE, 0);

        align = gtk_alignment_new (0.5f, 0.0f, 1.0f, 1.0f);
        gtk_alignment_set_padding (GTK_ALIGNMENT (align),
                                   0, 0, HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);

        gtk_container_set_border_width (GTK_CONTAINER (vbox), HILDON_MARGIN_HALF);

        event_box = gtk_event_box_new ();
        gtk_event_box_set_visible_window (GTK_EVENT_BOX (event_box), FALSE);
        gtk_widget_set_size_request (event_box, BOX_WIDTH, BOX_HEIGHT);

        gtk_container_add (GTK_CONTAINER (align), vbox);
        gtk_container_add (GTK_CONTAINER (event_box), align);
        gtk_container_add (GTK_CONTAINER (self), event_box);

        g_signal_connect (event_box, "button-press-event",
                G_CALLBACK (button_press_event_cb), self);
        g_signal_connect (event_box, "button-release-event",
                G_CALLBACK (button_release_event_cb), self);
        g_signal_connect (event_box, "leave-notify-event",
                G_CALLBACK (leave_notify_event_cb), self);

        gtk_widget_show_all (GTK_WIDGET (event_box));

#ifndef DEBUG_LAYOUT
        priv->eventlogger = rtcom_el_new ();
        g_signal_connect (priv->eventlogger,
                          "new-event",
                          G_CALLBACK (new_event_cb),
                          self);
        g_signal_connect (priv->eventlogger,
                          "event-updated",
                          G_CALLBACK (event_updated_cb),
                          self);

        read_event (self);
        priv->unread_count = query_unread_events (priv->eventlogger);
        update_unread_label (self);
#endif
}

static void
el_home_applet_class_init (ELHomeAppletClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);
        GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

        object_class->dispose = dispose;
        object_class->finalize = finalize;
        widget_class->expose_event = expose_event;
        widget_class->realize = el_home_applet_realize;

        g_type_class_add_private (klass, sizeof (ELHomeAppletPrivate));
}

