/*
 * Copyright (C) 2008 Marco Barisione <marco@barisione.org>
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <string.h>
#include "utils.h"


#define MAX_CHARS_FOR_UCHAR 6

gchar *
string_simplify (const gchar *str)
{
    gint size;
    gchar *ret;
    gint ret_pos;
    const gchar *p;

    g_return_val_if_fail (str, NULL);

    if (*str == '\0')
        return g_strdup ("");

    /* Every character can be at most MAX_CHARS_FOR_UCHAR, but usually
     * this doesn't happen. Create a buffer that is bigger enough to
     * contain the original string, plus another gunichar converted to
     * UTF-8, plus the terminating '\0', plus a bit more space (another
     * MAX_CHARS_FOR_UCHAR) to avoid reallocating in most cases. */
    size = strlen (str) + MAX_CHARS_FOR_UCHAR * 2 + 1;
    ret = g_malloc (size);
    ret_pos = 0;

    for (p = str; *p != '\0'; p = g_utf8_next_char (p)) {
        gunichar *buff;
        gsize buff_len;
        gsize i;

        buff = g_unicode_canonical_decomposition (g_utf8_get_char (p),
                    &buff_len);

        for (i = 0; i < buff_len; i++) {
            gunichar c;

            if (ret_pos + MAX_CHARS_FOR_UCHAR + 1 == size) {
                /* Ops, the buffer is too small to get another character */
                size += MAX (size / 2, MAX_CHARS_FOR_UCHAR * 2);
                ret = (gchar*) g_realloc (ret, size);
            }

            switch (g_unichar_type (buff[i])) {
                case G_UNICODE_COMBINING_MARK:
                case G_UNICODE_ENCLOSING_MARK:
                case G_UNICODE_NON_SPACING_MARK:
                    /* We want to strip accents and other marks, so let's
                     * ignore this */
                    c = 0;
                    break;
                case G_UNICODE_UPPERCASE_LETTER:
                case G_UNICODE_TITLECASE_LETTER:
                    /* All lower case */
                    c = g_unichar_tolower (buff[i]);
                    break;
                default:
                    /* Other char, just keep it as it is */
                    c = buff[i];
            }

            if (c) {
                gint written = g_unichar_to_utf8 (c, &ret[ret_pos]);
                ret_pos += written;
            }
        }

        g_free (buff);
    }

    ret[ret_pos] = '\0';

    return ret;
}

#define PIXBUF_DATA_NAME "avatar-pixbuf"

GdkPixbuf *
get_avatar (OssoABookContact *contact,
            GtkWidget        *widget)
{
    GdkPixbuf *pixbuf;

    g_return_val_if_fail (contact, NULL);
    g_return_val_if_fail (widget, NULL);

    pixbuf = g_object_get_data (G_OBJECT (widget->style), PIXBUF_DATA_NAME);

    if (!pixbuf) {
        pixbuf = osso_abook_avatar_get_image_rounded (
                OSSO_ABOOK_AVATAR (contact), HILDON_ICON_PIXEL_SIZE_FINGER,
                HILDON_ICON_PIXEL_SIZE_FINGER, TRUE, -1, NULL);

        if (!pixbuf) {
            const gchar *icon_name;
            GtkIconTheme *theme;
            GdkScreen *screen;

            icon_name = osso_abook_avatar_get_fallback_icon_name (
                    OSSO_ABOOK_AVATAR (contact));
            screen = gtk_widget_get_screen (widget);
            theme = gtk_icon_theme_get_for_screen (screen);

            pixbuf = gtk_icon_theme_load_icon (theme, icon_name,
                    HILDON_ICON_PIXEL_SIZE_FINGER, 0, NULL);
        }

        if (pixbuf) {
            g_object_set_data_full (G_OBJECT (contact), PIXBUF_DATA_NAME,
                    pixbuf, (GDestroyNotify) g_object_unref);
        }
    }

    return pixbuf;
}
