#include "ColumbusDialogWaypoints.h"

#include "widgets/WaypointListDelegate.h"

#include "ColumbusDialogWaypointsForm.h"
#include "ColumbusDialogWaypointsItemMenu.h"
#include "ColumbusDialogWaypointsAverage.h"

#include <rx/columbus/ColumbusWaypointModel.h>

#include <QDebug>

#include <QtGui>
#include <QtMaemo5>

class ColumbusDialogWaypointsPrivate
{
public:
    ColumbusDialogWaypointsPrivate();

    ColumbusWaypointModel   *model;

    QSortFilterProxyModel   *sortproxy;
    QModelIndex              current;

    QScrollArea             *scrollarea;
    QListView               *listview;
    WaypointListDelegate    *delegate;
    QBoxLayout              *bbox;

    ColumbusDialogWaypointsForm     *formdialog;
    ColumbusDialogWaypointsItemMenu *itemmenu;
    ColumbusDialogWaypointsAverage  *averagedialog;
    QMessageBox                     *deleteconfirm;
};

ColumbusDialogWaypointsPrivate::ColumbusDialogWaypointsPrivate()
    : model(NULL),
      sortproxy(NULL),
      scrollarea(NULL),
      listview(NULL),
      delegate(NULL),
      bbox(NULL),
      formdialog(NULL),
      itemmenu(NULL),
      averagedialog(NULL),
      deleteconfirm(NULL)
{ /* ... */ }

ColumbusDialogWaypoints::ColumbusDialogWaypoints(QWidget *parent)
    : ColumbusDialog(parent),
      d(new ColumbusDialogWaypointsPrivate)
{
    this->setWindowTitle(tr("Waypoints"));
}

ColumbusDialogWaypoints::~ColumbusDialogWaypoints()
{
    delete this->d;
}

void ColumbusDialogWaypoints::setController(ApplicationController *controller)
{
    ColumbusDialog::setController(controller);

    if(controller != NULL)
    {
        d->model = controller->waypoints();
        d->sortproxy = new QSortFilterProxyModel(this);
        d->delegate = new WaypointListDelegate(this->controller()->model(), this->controller()->waypoints(), this);

        d->sortproxy->setSourceModel(d->model);
        d->sortproxy->setDynamicSortFilter(true);
        d->sortproxy->setSortRole(Qt::DisplayRole);

        // FIXME: We need this to push distance/bearing updates to listview.
        // May break if filtering is enabled.
        QObject::connect(d->model, SIGNAL(dataChanged(QModelIndex,QModelIndex)),
                         d->sortproxy, SIGNAL(dataChanged(QModelIndex,QModelIndex)));

        this->setupUi();
        this->updateUi();

        d->formdialog->setController(controller);
    }
}

void ColumbusDialogWaypoints::setupUi()
{
    QVBoxLayout *vbox = new QVBoxLayout(this);
    QHBoxLayout *hbox = new QHBoxLayout();

    d->listview = new QListView(this);
    d->listview->setEditTriggers(QAbstractItemView::NoEditTriggers);
    d->listview->setAlternatingRowColors(true);

    QObject::connect(d->listview, SIGNAL(clicked(QModelIndex)), this, SLOT(onWaypointSelected(QModelIndex)));

    d->scrollarea = new QScrollArea(this);
    d->scrollarea->setWidget(d->listview);
    d->scrollarea->setWidgetResizable(true);

    d->itemmenu = new ColumbusDialogWaypointsItemMenu(this);
    d->formdialog = new ColumbusDialogWaypointsForm(this);
    d->averagedialog = new ColumbusDialogWaypointsAverage(this);
    d->deleteconfirm = new QMessageBox(this);
    d->deleteconfirm->setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    d->deleteconfirm->setWindowTitle(tr("Delete Waypoint?"));

    QObject::connect(d->itemmenu, SIGNAL(navigateTo(ColumbusWaypoint)), this, SLOT(onNavigateTo()));
    QObject::connect(d->itemmenu, SIGNAL(updateWaypoint(ColumbusWaypoint)), this, SLOT(showUpdateForm()));
    QObject::connect(d->itemmenu, SIGNAL(deleteWaypoint(ColumbusWaypoint)), this, SLOT(showDeleteConfirmation()));

    QObject::connect(d->deleteconfirm, SIGNAL(finished(int)), this, SLOT(onDeleteConfirmation(int)));

    QPushButton *createBtn  = new QPushButton(tr("Create"), this);
    //QPushButton *averageBtn = new QPushButton(tr("Average"), this);
    QPushButton *clearBtn   = new QPushButton(tr("Clear Destination"), this);

    hbox->addWidget(createBtn);
    //hbox->addWidget(averageBtn);
    hbox->addWidget(clearBtn);

    vbox->addWidget(d->scrollarea);
    vbox->addLayout(hbox);
    //vbox->addWidget(clearBtn);

    QObject::connect(createBtn, SIGNAL(clicked()), this, SLOT(showCreateForm()));
    //QObject::connect(averageBtn, SIGNAL(clicked()), this, SLOT(showAverageDialog()));
    QObject::connect(clearBtn, SIGNAL(clicked()), this, SLOT(onClearDestination()));
}

void ColumbusDialogWaypoints::updateUi()
{
    d->listview->setModel(d->sortproxy);
    d->listview->setItemDelegate(d->delegate);
    d->sortproxy->sort(ColumbusWaypointModel::Distance, Qt::AscendingOrder);
}

void ColumbusDialogWaypoints::onWaypointSelected(const QModelIndex &index)
{
    d->current = index;
    d->itemmenu->setWaypoint(d->model->waypoint(index));
    d->itemmenu->show();
}

void ColumbusDialogWaypoints::onNavigateTo()
{
    d->model->setDestination(d->current);
    d->itemmenu->hide();
}

void ColumbusDialogWaypoints::onClearDestination()
{
    d->model->setDestination(QModelIndex());
}

void ColumbusDialogWaypoints::showCreateForm()
{
    QObject::connect(d->formdialog, SIGNAL(finished(int)), this, SLOT(onFormCreateFinished(int)));

    d->formdialog->setWindowTitle("Create Waypoint");
    d->formdialog->setName("WP#" + QDateTime::currentDateTime().toString("yyMMddhhmmss"));

    if(this->controller()->model()->datalog()->size() > 0)
    {
        d->formdialog->setCoordinate(this->controller()->model()->currentPosition());
    }
    else
    {
        d->formdialog->setCoordinate(QGeoCoordinate());
    }

    d->formdialog->show();
}

void ColumbusDialogWaypoints::showAverageDialog()
{
    d->averagedialog->setName("WP#" + QDateTime::currentDateTime().toString("yyMMddhhmmss"));
    d->averagedialog->show();
}

void ColumbusDialogWaypoints::showUpdateForm()
{
    QObject::connect(d->formdialog, SIGNAL(finished(int)), this, SLOT(onFormUpdateFinished(int)));

    d->formdialog->setWindowTitle("Update Waypoint");
    d->formdialog->setName(d->model->waypoint(d->current).name());
    d->formdialog->setCoordinate(d->model->waypoint(d->current).coordinate());
    d->formdialog->show();
}

void ColumbusDialogWaypoints::showDeleteConfirmation()
{
    d->deleteconfirm->setText(tr("Delete waypoint \"") + d->itemmenu->windowTitle() + "\"?");
    d->deleteconfirm->show();
}

void ColumbusDialogWaypoints::onFormCreateFinished(int result)
{
    if(result != QDialog::Accepted) return;

    QObject::disconnect(d->formdialog, SIGNAL(finished(int)), this, SLOT(onFormCreateFinished(int)));
    d->model->addWaypoint(d->formdialog->name(), d->formdialog->coordinate());
}

void ColumbusDialogWaypoints::onFormUpdateFinished(int result)
{
    if(result != QDialog::Accepted) return;

    QObject::disconnect(d->formdialog, SIGNAL(finished(int)), this, SLOT(onFormUpdateFinished(int)));
    d->model->setWaypoint(d->current.row(), d->formdialog->name(), d->formdialog->coordinate());
    d->itemmenu->setWindowTitle(d->formdialog->name());
}

void ColumbusDialogWaypoints::onDeleteConfirmation(int result)
{
    if(result != QMessageBox::Yes) return;

    d->model->removeRow(d->current.row());
    d->model->submitAll();
    d->itemmenu->hide();
}
