#include <QDebug>
#include "DraggableViewArea.h"

class DraggableViewAreaPrivate
{
public:
    static DraggableViewArea *dragging;

    Qt::Orientation      orientation;

    QWidget             *content;

    int                  currentIndex;
    QList<QWidget *>     childWidgets;

    QPropertyAnimation  *transition;

    QPoint               mouseAnchor;
    QPointF              mouseAnchorFiltered;
};

// To make sure we're only dragging one area at a time.
DraggableViewArea *DraggableViewAreaPrivate::dragging = NULL;

DraggableViewArea::DraggableViewArea(Qt::Orientation orientation, QWidget *parent) :
    QWidget(parent)
{
    this->d = new DraggableViewAreaPrivate;
    d->orientation   = orientation;
    d->currentIndex  = 0;
    d->content       = new QWidget(this);
    d->transition    = new QPropertyAnimation(d->content, "geometry", this);

    this->installEventFilter(this);
    QObject::connect(d->transition, SIGNAL(finished()), this, SLOT(animationFinished()));
}

DraggableViewArea::~DraggableViewArea()
{
    delete this->d;
}

void DraggableViewArea::addWidget(QWidget *widget)
{
    widget->setParent(d->content);

    d->childWidgets.push_back(widget);

    if(d->orientation == Qt::Horizontal)
    {
        widget->setGeometry(0,
                            (d->childWidgets.size() - 1) * this->height(),
                            this->width(),
                            this->height());
        d->content->setGeometry(0,
                               -d->currentIndex * this->height(),
                               this->width(),
                               d->childWidgets.size() * this->height());
    }
    else
    {
        widget->setGeometry((d->childWidgets.size() - 1) * this->width(),
                            0,
                            this->width(),
                            this->height());
        d->content->setGeometry(-d->currentIndex * this->width(),
                               0,
                               d->childWidgets.size() * this->width(),
                               this->height());
    }

    widget->show();
}

bool DraggableViewArea::eventFilter(QObject *object, QEvent *event)
{
    if(object == this)
    {
        switch(event->type())
        {
        case QEvent::Resize:
            for(int i = 0; i < d->childWidgets.size(); i++)
            {
                QWidget *widget = d->childWidgets[i];

                widget->setFixedSize(this->width(), this->height());

                if(d->orientation == Qt::Horizontal)
                {
                    widget->setGeometry(this->width() * i, 0, this->width(), this->height());
                }
                else
                {
                    widget->setGeometry(0, this->height() * i, this->width(), this->height());
                }
            }

            if(d->orientation == Qt::Horizontal)
            {
                d->content->setGeometry(-d->currentIndex*width(),
                                       0,
                                       this->width() * d->childWidgets.size(),
                                       this->height());
            }
            else
            {
                d->content->setGeometry(0,
                                       -d->currentIndex * this->height(),
                                        this->width(),
                                        this->height() * d->childWidgets.size());
            }
            break;

        default:
            break;
        }
    }

    return QWidget::eventFilter(object, event);
}

void DraggableViewArea::jumpTo(int index)
{
    if (d->dragging == this) return;

    if (index < 0)
    {
        this->jumpTo(0);
        return;
    }

    if (index >= d->childWidgets.size())
    {
        this->jumpTo(d->childWidgets.size() - 1);
        return;
    }

    QWidget *widget = d->childWidgets[index];
    d->content->setGeometry(-widget->x(), -widget->y(), d->content->width(), d->content->height());
    d->currentIndex = index;
}

void DraggableViewArea::jumpTo(QWidget *widget)
{
    for(int i = 0; i < d->childWidgets.size(); i++)
    {
        if(d->childWidgets[i] == widget)
        {
            this->jumpTo(i);
            return;
        }
    }
}

void DraggableViewArea::animationFinished()
{
    emit slidTo(d->currentIndex);
    emit slidTo(d->childWidgets[d->currentIndex]);
}

void DraggableViewArea::slideTo(int index)
{
    if (d->dragging == this) return;

    if (index < 0)
    {
        this->slideTo(0);
        return;
    }
    if (index >= d->childWidgets.size())
    {
        this->slideTo(d->childWidgets.size() - 1);
        return;
    }

    QWidget *widget = d->childWidgets[index];

    d->currentIndex = index;

    d->transition->setDuration(250);
    d->transition->setEasingCurve(QEasingCurve::InOutQuad);
    d->transition->setStartValue(d->content->geometry());
    d->transition->setEndValue(QRect(-widget->x(),
                                    -widget->y(),
                                    this-> d->content->width(),
                                    d->content->height()));
    d->transition->start();
}

void DraggableViewArea::slideTo(QWidget *widget)
{
    for (int i = 0; i < d->childWidgets.size(); i++)
    {
        if (d->childWidgets[i] == widget)
        {
            this->slideTo(i);
            return;
        }
    }
}

void DraggableViewArea::mousePressEvent(QMouseEvent *event)
{
    d->mouseAnchor = event->globalPos();
    event->ignore();
}

void DraggableViewArea::mouseMoveEvent(QMouseEvent *event)
{
    int dx = event->globalX() - d->mouseAnchor.x();
    int dy = event->globalY() - d->mouseAnchor.y();

    if(d->orientation == Qt::Horizontal)
    {
        if (abs(dy) < abs(dx) && abs(dx) > 40 && d->dragging == NULL)
        {
            d->dragging = this;
            d->mouseAnchorFiltered = d->mouseAnchor;
        }
        else if (d->dragging != this)
        {
            event->ignore();
            return;
        }
    }
    else
    {
        if (abs(dy) > abs(dx) && abs(dy) > 40 && d->dragging == NULL)
        {
            d->dragging = this;
            d->mouseAnchorFiltered = d->mouseAnchor;
        }
        else if (d->dragging != this)
        {
            event->ignore();
            return;
        }
    }

    if (d->dragging == this)
    {
        // stop any animation
        d->transition->stop();

        if(d->orientation == Qt::Horizontal)
        {
            int newX = d->content->x()+dx;

            if (newX > 0) newX = 0;
            if (newX < this->width() - d->content->width()) newX = this->width() - d->content->width();

            d->content->setGeometry(newX,
                                   d->content->y(),
                                   d->content->width(),
                                   d->content->height());
            d->mouseAnchor = event->globalPos();
            d->mouseAnchorFiltered += d->mouseAnchor;
            d->mouseAnchorFiltered *= 0.5;
        }
        else
        {
            int newY = d->content->y()+dy;

            if (newY > 0) newY = 0;
            if (newY < this->height() - d->content->height()) newY = this->height() - d->content->height();

            d->content->setGeometry(d->content->x(),
                                   newY,
                                   d->content->width(),
                                   d->content->height());
            d->mouseAnchor = event->globalPos();
            d->mouseAnchorFiltered += d->mouseAnchor;
            d->mouseAnchorFiltered *= 0.5;
        }
    }
    else
    {
        event->ignore();
    }
}

void DraggableViewArea::mouseReleaseEvent(QMouseEvent *event)
{
    if (d->dragging == this)
    {
        qreal dx = event->globalX() - d->mouseAnchorFiltered.x();
        qreal dy = event->globalY() - d->mouseAnchorFiltered.y();

        int dv, dc, dt;

        if(d->orientation == Qt::Horizontal)
        {
            dv = dx;
            dc = d->content->x();
            dt = this->width();
        }
        else
        {
            dv = dy;
            dc = d->content->y();
            dt = this->height();
        }

        d->transition->setStartValue(d->content->geometry());

        int delta = 0;

        if (dv > 3 && d->currentIndex > 0 && dc > -dt * d->currentIndex)
        {
            delta = -1;
        }
        else if (dv < -3 && d->currentIndex < d->childWidgets.size() - 1 && dc < -dt * d->currentIndex)
        {
            delta = +1;
        }
        else if (abs(dv) < 3)
        {
            int prev    = abs(dc + dt * (d->currentIndex - 1));
            int here    = abs(dc + dt * d->currentIndex);
            int next    = abs(dc + dt * (d->currentIndex + 1));

            if (prev < here && prev < next)
            {
                delta = -1;
            }
            else if (next < here)
            {
                delta = +1;
            }
        }

        d->currentIndex += delta;

        if(d->orientation == Qt::Horizontal)
        {
            d->transition->setEndValue(QRect(-d->currentIndex * this->width(),
                                            d->content->y(),
                                            d->content->width(),
                                            d->content->height()));

        }
        else
        {
            d->transition->setEndValue(QRect(d->content->x(),
                                            -d->currentIndex * this->height(),
                                            d->content->width(),
                                            d->content->height()));
        }

        d->transition->setDuration(100);
        d->transition->setEasingCurve(QEasingCurve::OutQuad);
        d->transition->start();

        d->dragging = NULL;
    } else {
        event->ignore();
    }
}
