#include "TripModel.h"

#include "units.h"

#include <qnumeric.h>
#include <QDateTime>
#include <QStringList>
#include <QGeoCoordinate>

class TripModelPrivate
{
public:
    QString name;
    QString comment;

    TripModel::DisplayUnits displayUnits;

    qreal distanceCounter;

    qreal currentSpeed;
    qreal sumSpeed;
    qreal maximumSpeed;

    qreal currentAltitude;
    qreal minimumAltitude;
    qreal maximumAltitude;

    QGeoCoordinate::CoordinateFormat positionFormat;

    int travelTime;

    QDateTime tripStart;
    QDateTime tripEnd;

    QList<QGeoPositionInfo> datalog;
    QList<QGeoCoordinate> tracklog;
};

TripModel::TripModel(QObject *parent)
    : QObject(parent)
{
    this->d = new TripModelPrivate;
    d->name = "";
    d->comment = "";

    d->displayUnits = TripModel::Metric;

    d->distanceCounter = 0;

    d->currentSpeed = 0;
    d->sumSpeed = 0;
    d->maximumSpeed = 0;

    d->currentAltitude = 0;
    d->minimumAltitude = 100000;
    d->maximumAltitude = 0;

    d->travelTime = 0;
    d->tripStart = QDateTime::currentDateTimeUtc();
    d->tripEnd = QDateTime::currentDateTimeUtc();
}

TripModel::~TripModel()
{
    delete d;
}

QString TripModel::name() const
{
    return d->name;
}

void TripModel::setName(const QString &name)
{
    d->name = name;
    emit this->nameChanged(name);
    emit this->updated();
}

QString TripModel::comment() const
{
    return d->comment;
}

void TripModel::setComment(const QString &comment)
{
    d->comment = comment;
    emit this->commentChanged(comment);
    emit this->updated();
}

const QList<QGeoCoordinate>* TripModel::tracklog() const
{
    return &d->tracklog;
}

const QList<QGeoPositionInfo>* TripModel::datalog() const
{
    return &d->datalog;
}

qreal TripModel::convertDistanceToUnits(DisplayUnits units, qreal distance)
{
    switch(units)
    {
    case TripModel::Metric:
        return CONVERT_TO_KI(distance);
        break;

    case TripModel::Imperial:
        return CONVERT_TO_MI(distance);
        break;

    case TripModel::Nautical:
        return CONVERT_TO_NM(distance);
        break;
    }

    return distance;
}

qreal TripModel::convertSpeedToUnits(DisplayUnits units, qreal speed)
{
    switch(units)
    {
    case TripModel::Metric:
        return CONVERT_TO_KPH(speed);
        break;

    case TripModel::Imperial:
        return CONVERT_TO_MPH(speed);
        break;

    case TripModel::Nautical:
        return CONVERT_TO_KN(speed);
        break;
    }

    return speed;
}

qreal TripModel::convertAltitudeToUnits(DisplayUnits units, qreal altitude)
{
    switch(units)
    {
    case TripModel::Metric:
        break;

    case TripModel::Imperial:
    case TripModel::Nautical:
        return CONVERT_TO_FT(altitude);
        break;
    }

    return altitude;
}

TripModel::DisplayUnits TripModel::displayUnits() const
{
    return d->displayUnits;
}

QGeoCoordinate::CoordinateFormat TripModel::positionFormat() const
{
    return this->positionFormat();
}

void TripModel::setPositionFormat(QGeoCoordinate::CoordinateFormat format)
{
    d->positionFormat = format;
    emit this->positionFormatChanged(format);
}

void TripModel::setDisplayUnits(DisplayUnits units)
{
    d->displayUnits = units;
    emit this->displayUnitsChanged(d->displayUnits);

    emit this->distanceUpdated(this->distance());

    emit this->currentSpeedUpdated(this->currentSpeed());
    emit this->averageSpeedUpdated(this->averageSpeed());
    emit this->maximumSpeedUpdated(this->maximumSpeed());

    emit this->currentAltitudeUpdated(this->currentAltitude());
    emit this->minimumAltitudeUpdated(this->minimumAltitude());
    emit this->maximumAltitudeUpdated(this->maximumAltitude());

    emit this->updated();
}

QString TripModel::distanceUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case TripModel::Metric:
        units = "km";
        break;
    case TripModel::Imperial:
        units = "mi";
        break;
    case TripModel::Nautical:
        units = "nm";
        break;
    }

    return units;
}

QString TripModel::speedUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case TripModel::Metric:
        units = "kph";
        break;
    case TripModel::Imperial:
        units = "mph";
        break;
    case TripModel::Nautical:
        units = "kn";
        break;
    }

    return units;
}

QString TripModel::altitudeUnits() const
{
    QString units;

    switch(d->displayUnits)
    {
    case TripModel::Metric:
        units = "m";
        break;

    case TripModel::Imperial:
    case TripModel::Nautical:
        units = "ft";
        break;
    }

    return units;
}

qreal TripModel::distance() const
{
    return this->convertDistanceToUnits(d->displayUnits, d->distanceCounter);
}

qreal TripModel::currentSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->currentSpeed);
}

qreal TripModel::averageSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->sumSpeed / d->tracklog.count());
}

qreal TripModel::maximumSpeed() const
{
    return this->convertSpeedToUnits(d->displayUnits, d->maximumSpeed);
}

qreal TripModel::currentAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->currentAltitude);
}

qreal TripModel::minimumAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->minimumAltitude);
}

qreal TripModel::maximumAltitude() const
{
    return this->convertAltitudeToUnits(d->displayUnits, d->maximumAltitude);
}

qreal TripModel::currentHeading() const
{
    if(d->datalog.length() > 0)
    {
        if(d->datalog.last().hasAttribute(QGeoPositionInfo::Direction))
        {
            qreal heading = d->datalog.last().attribute(QGeoPositionInfo::Direction);
            if(!qIsNaN(heading)) return heading;
        }
    }

    return 0.0;
}

QGeoCoordinate TripModel::currentPosition() const
{
    return d->tracklog.last();
}

QString TripModel::currentLatitude() const
{
    QGeoCoordinate coord = this->currentPosition();
    QStringList format = coord.toString((QGeoCoordinate::CoordinateFormat)d->positionFormat).split(',');
    return format.at(0).trimmed();
}

QString TripModel::currentLongitude() const
{
    QGeoCoordinate coord = this->currentPosition();
    QStringList format = coord.toString((QGeoCoordinate::CoordinateFormat)d->positionFormat).split(',');
    return format.at(1).trimmed();
}

qreal TripModel::currentHorizontalAccuracy() const
{
    if(d->datalog.last().hasAttribute(QGeoPositionInfo::HorizontalAccuracy))
    {
        qreal hacc = d->datalog.last().attribute(QGeoPositionInfo::HorizontalAccuracy);
        return hacc;
    }

    return -1;
}

qreal TripModel::currentVerticalAccuracy() const
{
    if(d->datalog.last().hasAttribute(QGeoPositionInfo::VerticalAccuracy))
    {
        qreal vacc = d->datalog.last().attribute(QGeoPositionInfo::VerticalAccuracy);
        return vacc;
    }

    return -1;
}

int TripModel::travelTime() const
{
    return 0;
}

int TripModel::elapsedTime() const
{
    return d->tripStart.secsTo(d->tripEnd);
}

void TripModel::onPositionUpdated(const QGeoPositionInfo &position)
{
    QGeoCoordinate coord = position.coordinate();

    if(position.hasAttribute(QGeoPositionInfo::Direction))
    {
        qreal direction = position.attribute(QGeoPositionInfo::Direction);
        if(!qIsNaN(direction))
        {
            emit this->currentHeadingUpdated(direction);
        }
    }

    if(position.hasAttribute(QGeoPositionInfo::GroundSpeed))
    {
        qreal speed = position.attribute(QGeoPositionInfo::GroundSpeed);

        d->currentSpeed = speed;
        emit this->currentSpeedUpdated(this->currentSpeed());

        if(speed > d->maximumSpeed)
        {
            d->maximumSpeed = speed;
            emit this->maximumSpeedUpdated(this->maximumSpeed());
        }

        d->sumSpeed += speed;
        emit this->averageSpeedUpdated(this->averageSpeed());
    }

    if(coord.type() == QGeoCoordinate::Coordinate3D)
    {
        d->currentAltitude = coord.altitude();
        emit this->currentAltitudeUpdated(this->currentAltitude());

        if(coord.altitude() < d->minimumAltitude)
        {
            d->minimumAltitude = coord.altitude();
            emit this->minimumAltitudeUpdated(this->minimumAltitude());
        }
        else if(coord.altitude() > d->maximumAltitude)
        {
            d->maximumAltitude = coord.altitude();
            emit this->maximumAltitudeUpdated(this->maximumAltitude());
        }
    }

    if(d->tracklog.count() == 0)
    {
        d->tripStart = QDateTime::currentDateTimeUtc();
    }
    else
    {
        d->distanceCounter += d->tracklog.last().distanceTo(coord);
        emit this->distanceUpdated(this->distance());
    }

    d->datalog.append(position);
    d->tracklog.append(coord);

    d->tripEnd = QDateTime::currentDateTimeUtc();
    emit this->elapsedTimeUpdated(this->elapsedTime());

    emit this->updated();
}
