#include "units.h"
#include "NMEADataFormatter.h"

#include <QDebug>

class NMEADataFormatterPrivate
{
public:
    QGeoPositionInfo             position;
    QList<int>                   satsInUse;
    QList<QGeoSatelliteInfo>     satsInView;
};

NMEASentence::NMEASentence(const QString &command)
    : m_Command(command)
{ /* ... */ }

void NMEASentence::addParameter(const QString &parameter)
{
    m_Parameters.append(parameter);
}

char NMEASentence::checksum(const QByteArray &sentence) const
{
    char result = 0x00;

    for(int i = 0; i < sentence.length(); i++)
    {
        result ^= sentence.at(i);
    }

    return result;
}

QByteArray NMEASentence::format() const
{
    return (this->toString() + "\r\n").toAscii();
}

QString NMEASentence::toString() const
{
    QString format = m_Command;

    for(int i = 0; i < m_Parameters.length(); i++)
    {
        format.append("," + m_Parameters.at(i));
    }

    QString checksum = QString::number(this->checksum(format.toAscii()), 16);
    return QString("$" + format + "*" + QString("%1").arg(checksum, 2, '0').toUpper());
}

NMEADataFormatter::NMEADataFormatter(QObject *parent)
    : AbstractDataFormatter(parent)
{
    this->d = new NMEADataFormatterPrivate;
}

NMEADataFormatter::~NMEADataFormatter()
{
    delete this->d;
}

QString NMEADataFormatter::id() const
{
    static const QString id = "nmea";
    return id;
}

QString NMEADataFormatter::friendlyName() const
{
    static const QString name = tr("NMEA SiRF Compliant");
    return name;
}

void NMEADataFormatter::onPositionUpdate(const QGeoPositionInfo &posinfo)
{
    if(posinfo.isValid())
    {
        QGeoCoordinate coord = posinfo.coordinate();

        NMEASentence *gpgga = this->GPGGA(posinfo);
        NMEASentence *gpgll = this->GPGLL(posinfo);
        NMEASentence *gprmc = this->GPRMC(posinfo);
        NMEASentence *gpvtg = this->GPVTG(posinfo);
        NMEASentence *gpzda = this->GPZDA(posinfo);

        emit this->dataReady(gpgga->format());
        emit this->dataReady(gpgll->format());
        emit this->dataReady(gprmc->format());
        emit this->dataReady(gpvtg->format());
        emit this->dataReady(gpzda->format());

        delete gpgga;
        delete gpgll;
        delete gprmc;
        delete gpvtg;
        delete gpzda;

        d->position = posinfo;
    }
}

void NMEADataFormatter::onSatellitesInUse(const QList<QGeoSatelliteInfo> &satellites)
{
    d->satsInUse.clear();
    foreach(QGeoSatelliteInfo satinfo, satellites)
    {
        d->satsInUse.append(satinfo.prnNumber());
    }

    NMEASentence *gpgsa = this->GPGSA(satellites, d->position);
    emit this->dataReady(gpgsa->format());
    delete gpgsa;
}

void NMEADataFormatter::onSatellitesInView(const QList<QGeoSatelliteInfo> &satellites)
{
    d->satsInView.clear();
    foreach(QGeoSatelliteInfo satinfo, satellites)
    {
        d->satsInView.append(satinfo);
    }

    QList<NMEASentence*> sentences = this->GPGSV(satellites, d->position);
    foreach(NMEASentence *sentence, sentences)
    {
        emit this->dataReady(sentence->format());
        delete sentence;
    }
}

QString NMEADataFormatter::datestamp(const QDateTime &datetime) const
{
    return datetime.toString("ddMMyy");
}

QString NMEADataFormatter::timestamp(const QDateTime &datetime) const
{
    return datetime.toString("HHmmss.zzz");
}

QString NMEADataFormatter::latlon(double latitude, double longitude) const
{
    QString format;

    int     latdeg = latitude;
    double  latmin = qAbs((latdeg - latitude) * 60);

    int     londeg = longitude;
    double  lonmin = qAbs((londeg - longitude) * 60);

    format.append(QString("%1").arg(QString::number(qAbs(latdeg)), 2, '0'));
    format.append(QString("%1").arg(QString::number(latmin, 'f', 4), 7, '0'));
    format.append(latdeg >= 0 ? ",N," : ",S");

    format.append(QString("%1").arg(QString::number(qAbs(londeg)), 3, '0'));
    format.append(QString("%1").arg(QString::number(lonmin, 'f', 4), 7, '0'));
    format.append(londeg >= 0 ? ",E" : ",W");

    return format;
}

NMEASentence* NMEADataFormatter::GPGGA(const QGeoPositionInfo &posinfo) const
{
    QGeoCoordinate coord = posinfo.coordinate();

    NMEASentence *sentence = new NMEASentence("GPGGA");

    sentence->addParameter(this->timestamp(posinfo.timestamp().toUTC()));
    sentence->addParameter(this->latlon(coord.latitude(), coord.longitude()));
    sentence->addParameter(QString::number(posinfo.isValid() ? 1 : 0)); // GPS Fix
    sentence->addParameter(QString::number(d->satsInView.count()));
    sentence->addParameter("1.0"); // HDOP
    sentence->addParameter(QString::number(coord.altitude()) + ",M,46.9,M,,");

    return sentence;
}

NMEASentence* NMEADataFormatter::GPGLL(const QGeoPositionInfo &posinfo) const
{
    QGeoCoordinate coord = posinfo.coordinate();

    NMEASentence *sentence = new NMEASentence("GPGLL");

    sentence->addParameter(this->latlon(coord.latitude(), coord.longitude()));
    sentence->addParameter(this->timestamp(posinfo.timestamp().toUTC()));
    sentence->addParameter("A");

    return sentence;
}

NMEASentence* NMEADataFormatter::GPGSA(const QList<QGeoSatelliteInfo> &sats, const QGeoPositionInfo &posinfo) const
{
    NMEASentence *sentence = new NMEASentence("GPGSA");

    sentence->addParameter("A");
    sentence->addParameter(posinfo.coordinate().type() == QGeoCoordinate::InvalidCoordinate ? "1" : posinfo.coordinate().type() == QGeoCoordinate::Coordinate3D ? "3" : "2");

    foreach(QGeoSatelliteInfo satinfo, sats)
    {
        sentence->addParameter(QString("%1").arg(QString::number(satinfo.prnNumber()), 2, '0'));
    }

    for(int i = 0; i < 12 - sats.length(); i++)
    {
        sentence->addParameter("");
    }

    sentence->addParameter("1.0");
    sentence->addParameter("1.0");
    sentence->addParameter("1.0");

    return sentence;
}

QList<NMEASentence*> NMEADataFormatter::GPGSV(const QList<QGeoSatelliteInfo> &sats, const QGeoPositionInfo &posinfo) const
{
    QList<NMEASentence*>    results;

    int     mcount  = ((sats.length() - 1) / 4) + 1;

    for(int i = 0; i < mcount; i++)
    {
        NMEASentence *sentence = new NMEASentence("GPGSV");

        sentence->addParameter(QString::number(mcount));           // Message Count.
        sentence->addParameter(QString::number(i+1));              // Message Number.
        sentence->addParameter(QString::number(sats.length()));    // Satellite Count.

        for(int j = 0; j < 4; j++)
        {
            if(i * 4 + j >= sats.length()) break;

            QGeoSatelliteInfo satinfo = sats.at(i * 4 + j);

            sentence->addParameter(QString("%0").arg(QString::number(satinfo.prnNumber()), 2, '0'));
            sentence->addParameter(QString::number(satinfo.attribute(QGeoSatelliteInfo::Elevation)));
            sentence->addParameter(QString("%0").arg(QString::number(satinfo.attribute(QGeoSatelliteInfo::Azimuth)), 3, '0'));
            sentence->addParameter(QString::number(satinfo.signalStrength()));
        }

        results.append(sentence);
    }

    return results;
}

NMEASentence* NMEADataFormatter::GPRMC(const QGeoPositionInfo &posinfo) const
{
    QGeoCoordinate coord = posinfo.coordinate();

    qreal gspeed    = CONVERT_TO_KN(posinfo.attribute(QGeoPositionInfo::GroundSpeed));
    qreal heading   = posinfo.attribute(QGeoPositionInfo::Direction);
    qreal variation = posinfo.attribute(QGeoPositionInfo::MagneticVariation);

    NMEASentence *sentence = new NMEASentence("GPRMC");

    sentence->addParameter(this->timestamp(posinfo.timestamp().toUTC()));
    sentence->addParameter("A");
    sentence->addParameter(this->latlon(coord.latitude(), coord.longitude()));
    sentence->addParameter(QString::number(gspeed, 'f', 2));
    sentence->addParameter(QString::number(heading, 'f', 2));
    sentence->addParameter(this->datestamp(posinfo.timestamp().toUTC()));
    sentence->addParameter(QString::number(qAbs(variation), 'f', 2) + (variation >= 0 ? ",E,A" : ",W,A"));

    return sentence;
}

NMEASentence* NMEADataFormatter::GPVTG(const QGeoPositionInfo &posinfo) const
{
    qreal gspeed    = posinfo.attribute(QGeoPositionInfo::GroundSpeed);
    qreal heading   = posinfo.attribute(QGeoPositionInfo::Direction);
    qreal variation = posinfo.attribute(QGeoPositionInfo::MagneticVariation);

    NMEASentence *sentence = new NMEASentence("GPVTG");

    sentence->addParameter(QString::number(heading, 'f', 2) + ",T");
    sentence->addParameter(QString::number(heading + variation, 'f', 2) + ",M");
    sentence->addParameter(QString::number(CONVERT_TO_KN(gspeed), 'f', 2) + ",N");
    sentence->addParameter(QString::number(gspeed * 3.6, 'f', 2) + ",K,A");

    return sentence;
}

NMEASentence* NMEADataFormatter::GPZDA(const QGeoPositionInfo &posinfo) const
{
    QDateTime timestamp = posinfo.timestamp().toUTC();

    NMEASentence *sentence = new NMEASentence("GPZDA");

    sentence->addParameter(this->timestamp(timestamp));
    sentence->addParameter(QString::number(timestamp.date().day()));
    sentence->addParameter(QString::number(timestamp.date().month()));
    sentence->addParameter(QString::number(timestamp.date().year()));
    sentence->addParameter("00,00");

    return sentence;
}
