#include "PluginManager.h"

#include "plugins/bluetooth-server/BluetoothServerPlugin.h"
#include "plugins/network-server/NetworkServerPlugin.h"
#include "plugins/usb-serial/USBSerialPlugin.h"

#include "ApplicationController.h"

#include <QDebug>

class PluginManagerPrivate
{
public:
    ApplicationController *controller;

    QHash<QString,ColumbusPlugin*> directory;
};

PluginManager::PluginManager(QObject *parent) : QObject(parent)
{
    qDebug() << "PluginManager: ctor()";
    this->d = new PluginManagerPrivate;
}

PluginManager::~PluginManager()
{
    qDebug() << "PluginManager: dtor()";

    qDebug() << "Unregistering plugins.";
    foreach(QString pluginId, this->plugins().keys())
    {
        this->unregisterPlugin(pluginId);
    }
}

ApplicationController* PluginManager::controller() const
{
    return d->controller;
}

QHash<QString,ColumbusPlugin*> PluginManager::plugins() const
{
    return d->directory;
}

void PluginManager::initialize(ApplicationController *controller)
{
    d->controller = controller;
    this->loadPlugins();
}

void PluginManager::loadPlugins()
{
    this->registerPlugin(new BluetoothServerPlugin(this));
    this->registerPlugin(new NetworkServerPlugin(this));
    this->registerPlugin(new USBSerialPlugin(this));
}

bool PluginManager::registerPlugin(ColumbusPlugin *plugin)
{
    if(plugin->initialize(this))
    {
        qDebug() << "PluginManager: Registered plugin:" << plugin->id();
        d->directory.insert(plugin->id(), plugin);
        return true;
    }
    else
    {
        qDebug() << "PluginManager: registerPlugin(): Failed to register plugin:" << plugin->id();
        return false;
    }
}

void PluginManager::unregisterPlugin(const QString &pluginId)
{
    ColumbusPlugin *plugin = d->directory.value(pluginId);

    if(plugin == NULL)
    {
        qWarning() << "PluginManager: Plugin" << pluginId << "not found!";
        return;
    }

    qDebug() << "PluginManager: Removing plugin" << pluginId;
    d->directory.remove(pluginId);
    delete plugin;
}
